/******************************************************************************
 * Product: Compiere ERP & CRM Smart Business Solution                        *
 * Copyright (C) 1999-2007 ComPiere, Inc. All Rights Reserved.                *
 * This program is free software, you can redistribute it and/or modify it    *
 * under the terms version 2 of the GNU General Public License as published   *
 * by the Free Software Foundation. This program is distributed in the hope   *
 * that it will be useful, but WITHOUT ANY WARRANTY, without even the implied *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.           *
 * See the GNU General Public License for more details.                       *
 * You should have received a copy of the GNU General Public License along    *
 * with this program, if not, write to the Free Software Foundation, Inc.,    *
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.                     *
 * For the text or an alternative of this public license, you may reach us    *
 * ComPiere, Inc., 3600 Bridge Parkway #102, Redwood City, CA 94065, USA      *
 * or via info@compiere.org or http://www.compiere.org/license.html           *
 *****************************************************************************/

package org.compiere.install;

import java.io.*;
import java.util.zip.*;

import org.compiere.framework.*;
import org.compiere.util.*;

/**
 * 	Component Info
 * 	@author Jorg Janke
 */
public class ComponentInfo
{
	/**
	 * 	Get Components
	 *	@param compiereHome home
	 *	@return array of car files
	 */
	public static ComponentInfo[] getComponents (String compiereHome)
	{
		File dataDir = new File(compiereHome + File.separator + "data");
		if (dataDir.exists())
		{
			String[] fileNames = dataDir.list(new ComponentFileFilter());
			ComponentInfo[] retValue = new ComponentInfo[fileNames.length];
			for (int i = 0; i < fileNames.length; i++)
			{
				retValue[i] = new ComponentInfo(new File(dataDir, fileNames[i]));
			}
			return retValue;
		}
		return null;
	}	//	getComponents
	
	
	/**
	 * 	Constructor
	 *	@param fileName file name
	 */
	public ComponentInfo(File file)
	{
		init(file);
	}	//	ComponentInfo

	/** File to Read				*/
	private File	m_componentFile;
	/** File Valid					*/
	private boolean m_valid = false;
	/** License Accepted			*/
	private boolean m_accepted = false;
	/** Component Name				*/
	private String	m_name = null;
	/** Component Version			*/
	private String	m_version = null;
	/** License Info	*/
	private String	m_license = null;
	/** Logo Data		*/
	private byte[]	m_logo = null;
	
	/**	Logger						*/
	protected CLogger s_log = CLogger.getCLogger(ComponentInfo.class);
	
	/**
	 * 	Initialize info from file
	 *	@param file file
	 */
	private void init(File file)
	{
		m_componentFile = file;
		if (!m_componentFile.exists())
		{
			s_log.warning("Not found: " + file);
			return;
		}
		try
		{
			ZipUtil zip = new ZipUtil(m_componentFile);
			ZipEntry entry = zip.getEntry("EntityType_VO");
            if (entry == null)
            {
            	s_log.warning("EntityType_VO not found: " + m_componentFile);
            	return;
            }
            InputStream in = zip.getInputStream(entry);
            ObjectInputStream oIn = new ObjectInputStream(in);
            Object obj = oIn.readObject();
            if (obj instanceof VO)
            {
            	VO vo = (VO)obj;
    			m_name = vo.get("Name");
    			String et = vo.get("EntityType");
    			if ("D".equals(et))
    				m_name = "Compiere Core";
    			m_version = vo.get("Version");
    			m_license = vo.get("LicenseText");
    			String data = vo.get("BinaryData");
    			if (data != null)
    				m_logo = Util.toByteArray(data);
    			m_valid = true;
            }
			m_accepted = (m_license == null);
		}
		catch (Exception e)
		{
			m_valid = false;
		}
	}	//	init
	
	
	/**
	 * 	Is Valid
	 *	@return Valid Flag
	 */
	public boolean isValid()
	{
		return m_valid;
	}	//	isValid
	
	/**
	 * 	Get car File Name
	 * 	@return file name or null
	 */
	public String getFileName()
	{
		if (m_componentFile == null)
			return null;
		return m_componentFile.getAbsolutePath();
	}	//	getFileName
	
	/**
	 * 	Get License
	 *	@return license
	 */
	public String getLicense()
	{
		return m_license;
	}	//	getLicense
	
	/**
	 * 	Construct License File Name
	 *	@return file name
	 */
	public String getLicenseFileName()
	{
		StringBuffer sb = new StringBuffer(m_name);
		if (m_version != null && m_version.length() > 0)
			sb.append(" ").append(m_version);
		sb.append(" license.html");
		Util.replace(sb, " ", "_");
		return sb.toString();
	}	//	getLicenseFileName
	
	/**
	 * 	Get Heading
	 *	@return heading with version
	 */
	public String getHeading()
	{
		StringBuffer sb = new StringBuffer();
		if (m_name != null)
			sb.append(m_name);
		else
			sb.append(getFileName());
		if (m_version != null)
			sb.append(" ").append(m_version);
		return sb.toString();
	}	//	getHeading
	
	/**
	 * 	License Accepted
	 *	@param accepted
	 */
	public void setLicenseAccepted(boolean accepted)
	{
		m_accepted = accepted;
	}	//	setLicenseAccepted
	
	/**
	 * 	Get Logo
	 *	@return logo
	 */
	public byte[] getLogo()
	{
		return m_logo;
	}	//	getLogo
	
	/**
	 * 	List Rendering
	 */
	public String toString()
	{
		StringBuffer sb = new StringBuffer();
		if (m_name != null)
			sb.append(m_name);
		else
			sb.append(getFileName());
		if (m_version != null)
			sb.append(" ").append(m_version);
		if (!isValid())
			sb.append(" ** Not Valid **");
		else if (!m_accepted)
			sb.append(" {Requires accepting License}");
		return sb.toString();
	}	//	toString
	
}	//	ComponentInfo

