package org.compiere.print;

/*
 * @(#)MediaSize.java	1.14 04/05/05
 *
 * Copyright 2004 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
import java.util.HashMap;
import java.util.Vector;

import javax.print.attribute.Attribute;
import javax.print.attribute.standard.MediaSize;
import javax.print.attribute.standard.MediaSizeName;

/**
 * Class MediaSize is a two-dimensional size valued printing attribute class 
 * that indicates the dimensions of the medium in a portrait orientation, with 
 * the X dimension running along the bottom edge and the Y dimension running 
 * along the left edge. Thus, the Y dimension must be greater than or equal to 
 * the X dimension. Class MediaSize declares many standard media size
 * values, organized into nested classes for ISO, JIS, North American,
 * engineering, and other media. 
 * <P>
 * MediaSize is not yet used to specify media. Its current role is
 * as a mapping for named media (see {@link MediaSizeName MediaSizeName}).
 * Clients can use the mapping method
 * <code>MediaSize.getMediaSizeForName(MediaSizeName)</code>
 * to find the physical dimensions of the MediaSizeName instances
 * enumerated in this API. This is useful for clients which need this 
 * information to format & paginate printing.
 * <P>
 *
 * @author  Jirimuto, Bug Fix for Media Size ( X > Y )
 */
public class CMediaSize extends MediaSize implements Attribute {

    private static final long serialVersionUID = -1967958664615414771L;

    private MediaSizeName mediaName;

    private static HashMap mediaMap = new HashMap(100, 10);

    private static Vector sizeVector = new Vector(100, 10);
    
    private boolean reversed = false;
 
    /**
     * Construct a new media size attribute from the given floating-point 
     * values. 
     *
     * @param  x  X dimension.
     * @param  y  Y dimension.
     * @param  units
     *     Unit conversion factor, e.g. <CODE>Size2DSyntax.INCH</CODE> or
     *     <CODE>Size2DSyntax.MM</CODE>. 
     *
     * @exception  IllegalArgumentException
     *   (Unchecked exception) Thrown if <CODE>x</CODE> < 0 or <CODE>y</CODE> 
     *     < 0 or <CODE>units</CODE> < 1 or <CODE>x</CODE> > <CODE>y</CODE>.
     */
    public CMediaSize(float x, float y,int units) {
	super (x, y, units);
//	if (x > y) {
//	    throw new IllegalArgumentException("X dimension > Y dimension");
//	}
	sizeVector.add(this);
    }
    public CMediaSize(float x, float y,int units, boolean reverse) {
    	super (y, x, units);
//    	if (x > y) {
//    	    throw new IllegalArgumentException("X dimension > Y dimension");
//    	}
    	reversed = true;
    	sizeVector.add(this);
        }

    /**
     * Construct a new media size attribute from the given integer values. 
     *
     * @param  x  X dimension.
     * @param  y  Y dimension.
     * @param  units
     *     Unit conversion factor, e.g. <CODE>Size2DSyntax.INCH</CODE> or
     *     <CODE>Size2DSyntax.MM</CODE>. 
     *
     * @exception  IllegalArgumentException
     *   (Unchecked exception) Thrown if <CODE>x</CODE> < 0 or <CODE>y</CODE> 
     *     < 0 or <CODE>units</CODE> < 1 or <CODE>x</CODE> > <CODE>y</CODE>.
     */
    public CMediaSize(int x, int y,int units) {
	super (x, y, units);
//	if (x > y) {
//	    throw new IllegalArgumentException("X dimension > Y dimension");
//	}
	sizeVector.add(this);
    }
    
    public CMediaSize(int x, int y,int units, boolean reverse) {
    	super (y, x, units);
//    	if (x > y) {
//    	    throw new IllegalArgumentException("X dimension > Y dimension");
//    	}
    	reversed = true;
    	sizeVector.add(this);
    }

   /**
     * Construct a new media size attribute from the given floating-point 
     * values. 
     *
     * @param  x  X dimension.
     * @param  y  Y dimension.
     * @param  units
     *     Unit conversion factor, e.g. <CODE>Size2DSyntax.INCH</CODE> or
     *     <CODE>Size2DSyntax.MM</CODE>. 
     * @param media a media name to associate with this MediaSize
     *
     * @exception  IllegalArgumentException
     *   (Unchecked exception) Thrown if <CODE>x</CODE> < 0 or <CODE>y</CODE> 
     *     < 0 or <CODE>units</CODE> < 1 or <CODE>x</CODE> > <CODE>y</CODE>.
     */
    public CMediaSize(float x, float y,int units, MediaSizeName media) {
	super (x, y, units);
//	if (x > y) {
//	    throw new IllegalArgumentException("X dimension > Y dimension");
//	}
	mediaName = media;
	mediaMap.put(mediaName, this);
	sizeVector.add(this);
    }
    public CMediaSize(float x, float y,int units, MediaSizeName media, boolean reverse) {
    	super (y, x, units);
//    	if (x > y) {
//    	    throw new IllegalArgumentException("X dimension > Y dimension");
//    	}
    	reversed = true;
    	mediaName = media;
    	mediaMap.put(mediaName, this);
    	sizeVector.add(this);
        }

    /**
     * Construct a new media size attribute from the given integer values. 
     *
     * @param  x  X dimension.
     * @param  y  Y dimension.
     * @param  units
     *     Unit conversion factor, e.g. <CODE>Size2DSyntax.INCH</CODE> or
     *     <CODE>Size2DSyntax.MM</CODE>. 
     * @param media a media name to associate with this MediaSize
     *
     * @exception  IllegalArgumentException
     *   (Unchecked exception) Thrown if <CODE>x</CODE> < 0 or <CODE>y</CODE> 
     *     < 0 or <CODE>units</CODE> < 1 or <CODE>x</CODE> > <CODE>y</CODE>.
     */
    public CMediaSize(int x, int y,int units, MediaSizeName media) {
	super (x, y, units);
//	if (x > y) {
//	    throw new IllegalArgumentException("X dimension > Y dimension");
//	}
	mediaName = media;
	mediaMap.put(mediaName, this);
	sizeVector.add(this);
    }
    public CMediaSize(int x, int y,int units, MediaSizeName media, boolean reverse) {
    	super (y, x, units);
//    	if (x > y) {
//    	    throw new IllegalArgumentException("X dimension > Y dimension");
//    	}
    	reversed = true;
    	mediaName = media;
    	mediaMap.put(mediaName, this);
    	sizeVector.add(this);
        }

    /**
     * Get the media name, if any, for this size.
     *
     * @return the name for this media size, or null if no name was
     * associated with this size (an anonymous size).
     */
    public MediaSizeName getMediaSizeName() {
	return mediaName;
    }

    /**
     * Get the MediaSize for the specified named media.
     *
     * @param media - the name of the media for which the size is sought
     * @return size of the media, or null if this media is not associated
     * with any size.
     */
    public static MediaSize getMediaSizeForName(MediaSizeName media) {
	return (MediaSize)mediaMap.get(media);
    }

    /**
     * The specified dimensions are used to locate a matching MediaSize
     * instance from amongst all the standard MediaSize instances.
     * If there is no exact match, the closest match is used.
     * <p>
     * The MediaSize is in turn used to locate the MediaSizeName object.
     * This method may return null if the closest matching MediaSize
     * has no corresponding Media instance.
     * <p>
     * This method is useful for clients which have only dimensions and
     * want to find a Media which corresponds to the dimensions.
     * @param x - X dimension
     * @param y - Y dimension.
     * @param  units
     *     Unit conversion factor, e.g. <CODE>Size2DSyntax.INCH</CODE> or
     *     <CODE>Size2DSyntax.MM</CODE>
     * @return MediaSizeName matching these dimensions, or null.
     * @exception IllegalArgumentException if x <= 0, y <= 0, or units < 1
     * 
     */
    public static MediaSizeName findMedia(float x, float y, int units) {

	MediaSize match = MediaSize.ISO.A4;

	if (x <= 0.0f || y <= 0.0f || units < 1) {
	    throw new IllegalArgumentException("args must be +ve values");
	}

	double ls = x * x + y * y;
	double tmp_ls;
	float []dim;
	float diffx = x;
	float diffy = y;

	for (int i=0; i < sizeVector.size() ; i++) {
	    MediaSize mediaSize = (MediaSize)sizeVector.elementAt(i);
	    dim = mediaSize.getSize(units);
	    if (x == dim[0] && y == dim[1]) {
		match = mediaSize;
		break;
	    } else {
		diffx = x - dim[0];
		diffy = y - dim[1];
		tmp_ls = diffx * diffx + diffy * diffy;
		if (tmp_ls < ls) {
		    ls = tmp_ls;
		    match = mediaSize;
		}
	    }
	}

	return match.getMediaSizeName();
    }

    /**
     * Returns whether this media size attribute is equivalent to the passed 
     * in object.
     * To be equivalent, all of the following conditions must be true: 
     * <OL TYPE=1>
     * <LI>
     * <CODE>object</CODE> is not null.
     * <LI>
     * <CODE>object</CODE> is an instance of class MediaSize.
     * <LI>
     * This media size attribute's X dimension is equal to 
     * <CODE>object</CODE>'s X dimension. 
     * <LI>
     * This media size attribute's Y dimension is equal to 
     * <CODE>object</CODE>'s Y dimension. 
     * </OL>
     *
     * @param  object  Object to compare to.
     *
     * @return  True if <CODE>object</CODE> is equivalent to this media size
     *          attribute, false otherwise. 
     */
    public boolean equals(Object object) {
	return (super.equals(object) && object instanceof MediaSize);
    }

    /* force loading of all the subclasses so that the instances
     * are created and inserted into the hashmap.
     */
    static {
	MediaSize ISOA4 = ISO.A4;
	MediaSize JISB5 = JIS.B5;
	MediaSize NALETTER = NA.LETTER;
	MediaSize EngineeringC = Engineering.C;
	MediaSize OtherEXECUTIVE = Other.EXECUTIVE;
    }

	public boolean isReversed() {
		return reversed;
	}
	
    public float getX(int units){
    	if( reversed )
    		return super.getY( units );
    	else 
    		return super.getX( units );
    }

    public float getY(int units){
    	if( reversed )
    		return super.getX( units );
    	else 
    		return super.getY( units );
    }
    
}
