<?php
/**
 * CSSJドライバです。
 *
 * @package cssj
 * @subpackage driver
 * @example ../examples/client_resource.php アプリケーション側のファイルを変換する
 * @example ../examples/server_resource.php サーバー側のファイルを変換する
 * @example ../examples/ob.php 出力バッファを変換する
 * @example ../examples/var.php 変数から変数へ変換する
 * @example ../examples/file.php 一旦ファイルに出力してリダイレクトする
 */

/**
 * プロトコルレベルの処理はctipサブパッケージが行います。
 */
require_once ('cssj_ctip.php');

/**
 * エラーレベル定数:警告
 *
 * @access public
 */
define ('CSSJ_WARN', 1);
/**
 * エラーレベル定数:エラー
 *
 * @access public
 */
define ('CSSJ_ERROR', 2);
/**
 * エラーレベル定数:致命的エラー
 *
 * @access public
 */
define ('CSSJ_FATAL', 3);
/**
 * エラーレベル定数:情報
 *
 * @access public
 */
define ('CSSJ_INFO', 4);

/**
 * 指定されたホスト・ポートに接続するためのドライバを返します。
 * 
 * @access public
 * @param $host string 接続先ホスト
 * @param $port int 接続先ポート
 * @param $encoding string 文字のやりとりに使うキャラクタ・エンコーディング名
 * @return mixed ドライバ
 */
function &cssj_create_driver_for($host, $port, $encoding = 'ISO-8859-1') {
  $driver = array(
    'host' => $host,
    'port' => $port,
    'encoding' => $encoding
  );
  return $driver;
}

/**
 * セッションを作成します。
 * 
 * @access public
 * @param $driver mixed ドライバ
 * @param $user string ユーザーID
 * @param $password string パスワード
 * @return mixed セッション,エラーの場合はfalse
 */
function &cssj_create_session(&$driver, $user, $password) {
  if (($fp = fsockopen($driver['host'], $driver['port'], $errno, $errmsg)) === false) {
    trigger_error (__FUNCTION__.": socket_connect() failed: $errno / $errmsg", E_USER_ERROR);
    return false;
  }
  $session = array(
    'state' => 1,
    'driver' => &$driver,
    'out' => null,
    'fp' => &$fp,
    'errorFunc' => null,
    'progressFunc' => null
  );
  if (cssj_ctip_connect($fp, $driver['encoding']) === false) {
    return false;
  }
  if (cssj_ctip_req_property($fp, 'ctip.auth', 'PLAIN:'.$user.chr(0x0A).$password) === false) {
	  return false;
  }
  if (($response = cssj_ctip_res_next($fp)) === false) {
	  return false;
  }
  if ($response['type'] !== CSSJ_CTIP_RES_ERROR) {
	  return false;
  }
  if ($response['error'] !== 'OK') {
	  trigger_error (__FUNCTION__.": Authentication failed.", E_USER_ERROR);
	  return false;
  }
  return $session;
}

/**
 * 変換結果の出力先を指定します。
 *
 * cssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * この関数を呼び出さない場合、出力先はnull(標準出力)になります。
 * 出力先がnullの場合、header()関数によりContent-Lengthヘッダが設定されます。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $out mixed 出力先ストリーム(resource),出力先変数(string),または標準出力であればnull。
 * @param $mimeType 出力形式
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_set_output(&$session, &$out, $mimeType = 'application/pdf') {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  $session['out'] =& $out;
  return cssj_set_property($session, 'output.type', $mimeType);
}

/**
 * エラーメッセージ受信のためのコールバック関数を設定します。
 *
 * cssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * コールバック関数の引数は、エラーレベル(int)、メッセージ(string)です。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $errorFunc function コールバック関数
 * @return void
 */
function cssj_set_error_func(&$session, &$errorFunc) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return;
  }
  $session['errorFunc'] =& $errorFunc;
}

/**
 * 進行状況受信のためのコールバック関数を設定します。
 *
 * cssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * コールバック関数の引数は、読み込み済みバイト数(int)です。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $progressFunc function コールバック関数
 * @return void
 */
function cssj_set_progress_func(&$session, &$progressFunc) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return;
  }
  $session['progressFunc'] =& $progressFunc;
}

/**
 * プロパティを設定します。
 *
 * セッションを作成した直後に呼び出してください。
 * 利用可能なプロパティの一覧は「開発者ガイド」を参照してください。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $name string 名前
 * @param $value string 値
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_set_property(&$session, $name, $value) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  return cssj_ctip_req_property($session['fp'], $name, $value);
}

/**
 * アクセス可能なサーバー側リソースを設定します。
 *
 * cssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $uriPattern string URIパターン
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_include_resource(&$session, $uriPattern) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  return cssj_set_property($session, 'ctip.include', $uriPattern);
}

/**
 * 除外するサーバー側リソースを設定します。
 *
 * cssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $uriPattern string URIパターン
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_exclude_resource(&$session, $uriPattern) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  return cssj_set_property($session, 'ctip.exclude', $uriPattern);
}

/**
 * サーバー側リソースを変換します。
 *
 * この関数は1つのセッションにつき1度だけ呼ぶことができます。
 * その後、対象のセッションに対してcssj_close以外の操作はできません。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $uri string URI
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_format_main(&$session, $uri) {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  if (cssj_include_resource($session, $uri) === false) {
    return false;
  }
  if (cssj_set_property($session, 'ctip.main', $uri) === false) {
    return false;
  }
  $session['state'] = 2;
  if (cssj_ctip_req_end($session['fp']) === false) {
    return false;
  }
  return cssj_ctip_res_build($session['fp'], $session['out'], $session['errorFunc'], $session['progressFunc']);
  }

/**
 * リソース送信のための出力のバッファリングを有効にします。
 *
 * cssj_ob_start_resource,cssj_ob_end_flush_resourceは対となります。
 * これらの関数はcssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $uri string 仮想URI
 * @param $mimeType MIME型
 * @param $encoding キャラクタ・エンコーディング
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_ob_start_resource(&$session, $uri, $mimeType = 'text/css', $encoding = '') {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  $fp =& $session['fp'];
  if (cssj_ctip_req_resource($fp, $uri, $mimeType, $encoding) === false) {
    return false;
  }
  $GLOBALS['cssj_ob_session'] =& $session;
  // HACK ob_startのchunk_size指定によりヘッダが送信されるのを防ぐため、２重にバッファしています。
  ob_start();
  return ob_start('_cssj_ob_resource_handler', CSSJ_BUFFER_SIZE);
}

/**
 * バッファの内容を送信し、リソース送信のためのバッファリングを終了します。
 *
 * cssj_ob_start_resource,cssj_ob_end_flush_resourceは対となります。
 * これらの関数はcssj_format_mainおよびcssj_ob_start_mainの前に呼び出してください。
 * 
 * @access public
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_ob_end_flush_resource() {
  if (!isset($GLOBALS['cssj_ob_session']) || $GLOBALS['cssj_ob_session']['state'] >= 2) {
    trigger_error (__FUNCTION__.": cssj_ob_start_resource() was not called.", E_USER_WARNING);
    return false;
  }
  $err = ob_end_clean();
  ob_end_clean();
  if (isset($GLOBALS['cssj_ob_err'])) {
    $err = true;
  }
  unset($GLOBALS['cssj_ob_session']);
  unset($GLOBALS['cssj_ob_err']);
  return $err;
}

/**
 * リソースの送信のためのコールバック関数です。
 * 
 * @access private
 * @param $buffer
 * @return string
 */
function _cssj_ob_resource_handler($buffer) {
  if (isset($GLOBALS['cssj_ob_err'])) {
    return '';
  }
  $fp =& $GLOBALS['cssj_ob_session']['fp'];
  for (;;) {
    $buff = substr($buffer, 0, CSSJ_BUFFER_SIZE);
    $len = strlen($buff);
    if ($len <= 0) {
      break;
    }
    $buffer = substr($buffer, $len);
    if (cssj_ctip_req_write($fp, $buff) === false) {
      $GLOBALS['cssj_ob_err'] = true;
      return '';
    }
  }
  return '';
}

/**
 * 本体の変換のための出力のバッファリングを有効にします。
 *
 * cssj_ob_start_main,cssj_ob_end_flush_mainは対となります。
 * 本体の送信は1つのセッションにつき1度だけです。
 * その後、対象のセッションに対してcssj_close以外の操作はできません。
 * 
 * @access public
 * @param $session mixed セッション
 * @param $uri string 仮想URI
 * @param $mimeType MIME型
 * @param $encoding キャラクタ・エンコーディング
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_ob_start_main(&$session, $uri, $mimeType = 'text/html', $encoding = '') {
  if ($session['state'] >= 2) {
    trigger_error (__FUNCTION__.": Main content is already sent.", E_USER_WARNING);
    return false;
  }
  $session['state'] = 2;
  $fp =& $session['fp'];
  if (cssj_ctip_req_main($fp, $uri, $mimeType, $encoding) === false) {
    return false;
  }
  $GLOBALS['cssj_ob_builder'] = cssj_ctip_res_create_builder($fp, $session['out'], $session['errorFunc'], $session['progressFunc']);
  $GLOBALS['cssj_ob_session'] =& $session;
  // HACK ob_startのchunk_size指定によりヘッダが送信されるのを防ぐため、２重にバッファしています。
  ob_start();
  return ob_start('_cssj_ob_main_handler', CSSJ_BUFFER_SIZE);
}

/**
 * 変換結果を送信し、本体の変換のためのバッファリングを終了します。
 *
 * cssj_ob_start_main,cssj_ob_end_flush_mainは対となります。
 * 本体の送信は1つのセッションにつき1度だけです。
 * その後、対象のセッションに対してcssj_close以外の操作はできません。
 * 
 * @access public
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_ob_end_flush_main() {
  if (!isset($GLOBALS['cssj_ob_session']) || $GLOBALS['cssj_ob_session']['state'] != 2) {
    trigger_error (__FUNCTION__.": cssj_ob_start_main() was not called.", E_USER_WARNING);
    return false;
  }
  ob_end_clean();
  ob_end_clean();
  if (isset($GLOBALS['cssj_ob_err'])) {
    $err = false;
  }
  else {
    $fp =& $GLOBALS['cssj_ob_session']['fp'];
    cssj_ctip_req_end($fp);
    while (($err = cssj_ctip_res_build_next($GLOBALS['cssj_ob_builder'])) === true)
      ; 
  }
  unset($GLOBALS['cssj_ob_builder']);
  unset($GLOBALS['cssj_ob_session']);
  unset($GLOBALS['cssj_ob_err']);
  return ($err === false) ? false : true;
}

/**
 * 本体の変換のためのコールバック関数です。
 * 
 * @access private
 * @param $buffer
 * @return string
 */
function _cssj_ob_main_handler($buffer) {
  if (isset($GLOBALS['cssj_ob_err'])) {
    return '';
  }
  $fp =& $GLOBALS['cssj_ob_session']['fp'];
  for (;;) {
    $buff = substr($buffer, 0, CSSJ_BUFFER_SIZE);
    $len = strlen($buff);
    if ($len <= 0) {
      break;
    }
    $buffer = substr($buffer, $len);
    $packet = pack('NC', $len + 1, CSSJ_CTIP_REQ_DATA).$buff;
    $len = strlen($packet);
    for (;;) {
      $r = array($fp);
      $w = array($fp);
      $ex = null;
      if (($status = stream_select($r, $w, $ex, 0)) === false) {
        $GLOBALS['cssj_ob_err'] = true;
        return '';
      }
      if ($len > 0 && !empty($w)) {
        stream_set_blocking($fp, 0);
        if (($rlen = fwrite($fp, $packet)) === false) {
          $GLOBALS['cssj_ob_err'] = true;
          stream_set_blocking($fp, 1);
          return '';
        }
        stream_set_blocking($fp, 1);
      	$packet = substr($packet, $rlen);
        $len -= $rlen;
      }
      if (!empty($r)) {
        if (cssj_ctip_res_build_next($GLOBALS['cssj_ob_builder']) === false) {
          $GLOBALS['cssj_ob_err'] = true;
          return '';
        }
      }
	  if ($len <= 0) {
        break;
      }
    }
  }
  return '';
}

/**
 * セッションを閉じます。
 *
 * この関数の呼出し後、対象となったセッションに対するいかなる操作もできません。
 * 
 * @access public
 * @param $session
 * @return boolean 成功ならtrue,失敗ならfalse
 */
function cssj_close(&$session) {
  if ($session['state'] >= 3) {
    trigger_error (__FUNCTION__.": The session is already closed.", E_USER_WARNING);
    return false;
  }
  $session['state'] = 3;
  return fclose($session['fp']);
}

?>