/*

 Copyright 2001,2003  The Apache Software Foundation 

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 */
package jp.cssj.sakae.opentype.table;

import java.io.DataInputStream;
import java.io.IOException;

/**
 * @version $Id: CmapFormat4.java,v 1.2 2007-05-06 04:27:01 miyabe Exp $
 * @author <a href="mailto:david@steadystate.co.uk">David Schweinsberg</a>
 */
public class CmapFormat4 extends CmapFormat {

	public int language;

	private int segCountX2;

	private int searchRange;

	private int entrySelector;

	private int rangeShift;

	private int[] endCode;

	private int[] startCode;

	private int[] idDelta;

	private int[] idRangeOffset;

	private int[] glyphIdArray;

	private int segCount;

	private int first, last;

	protected CmapFormat4(DataInputStream in) throws IOException {
		super(in);
		format = 4;
		segCountX2 = in.readUnsignedShort();
		segCount = segCountX2 / 2;
		endCode = new int[segCount];
		startCode = new int[segCount];
		idDelta = new int[segCount];
		idRangeOffset = new int[segCount];
		searchRange = in.readUnsignedShort();
		entrySelector = in.readUnsignedShort();
		rangeShift = in.readUnsignedShort();
		last = -1;
		for (int i = 0; i < segCount; i++) {
			endCode[i] = in.readUnsignedShort();
			if (endCode[i] > last)
				last = endCode[i];
		}
		in.readUnsignedShort(); // reservePad
		for (int i = 0; i < segCount; i++) {
			startCode[i] = in.readUnsignedShort();
			if ((i == 0) || (startCode[i] < first))
				first = startCode[i];
		}
		for (int i = 0; i < segCount; i++) {
			idDelta[i] = in.readUnsignedShort();
		}
		for (int i = 0; i < segCount; i++) {
			idRangeOffset[i] = in.readUnsignedShort();
		}

		// Whatever remains of this header belongs in glyphIdArray
		int count = (length - 16 - (segCount * 8)) / 2;
		glyphIdArray = new int[count];
		for (int i = 0; i < count; i++) {
			glyphIdArray[i] = in.readUnsignedShort();
		}
	}

	public int getFirst() {
		return first;
	}

	public int getLast() {
		return last;
	}

	public int mapCharCode(int charCode) {
		try {
			/*
			 * Quoting :
			 * http://developer.apple.com/fonts/TTRefMan/RM06/Chap6cmap
			 * .html#Surrogates
			 * 
			 * The original architecture of the Unicode Standard allowed for all
			 * encoded characters to be represented using sixteen bit code
			 * points. This allowed for up to 65,354 characters to be encoded.
			 * (Unicode code points U+FFFE and U+FFFF are reserved and
			 * unavailable to represent characters. For more details, see The
			 * Unicode Standard.)
			 * 
			 * My comment : Isn't there a typo here ? Shouldn't we rather read
			 * 65,534 ?
			 */
			if ((charCode < 0) || (charCode >= 0xFFFE))
				return 0;

			for (int i = 0; i < segCount; i++) {
				if (endCode[i] >= charCode) {
					if (startCode[i] <= charCode) {
						if (idRangeOffset[i] > 0) {
							int off = idRangeOffset[i] / 2
									+ (charCode - startCode[i])
									- (segCount - i);
							return glyphIdArray[off];
						} else {
							return (idDelta[i] + charCode) % 65536;
						}
					} else {
						break;
					}
				}
			}
		} catch (ArrayIndexOutOfBoundsException e) {
			System.err
					.println("error: Array out of bounds - " + e.getMessage());
		}
		return 0;
	}

	public String toString() {
		return new StringBuffer().append(super.toString()).append(
				", language: ").append(language).append(", segCountX2: ")
				.append(segCountX2).append(", searchRange: ").append(
						searchRange).append(", entrySelector: ").append(
						entrySelector).append(", rangeShift: ").append(
						rangeShift).append(", endCode: ").append(endCode)
				.append(", startCode: ").append(endCode).append(", idDelta: ")
				.append(idDelta).append(", idRangeOffset: ").append(
						idRangeOffset).toString();
	}
}
