/*
** ksh.c --- kernel shell
*/

#include <coron.h>
#include <string.h>
#include <console.h>
#include <floppy.h>
#include <task.h>

static void main_ksh(void);
static byte stack_ksh[4096];

static Taskinfo ksh_task = {
  1, 0,
  (uint)main_ksh, 0,
  0, 0,
  0, 0,
  (uint)stack_ksh + sizeof(stack_ksh), sizeof(stack_ksh),
  0, 0,
  0, 0,
  0
};

typedef struct {
  const char *cmd;
  byte len;
  void (*proc)( char *cmd );
  const char *msg;
} cmd_t;

void help_ksh( char *cmd );
void reboot_ksh( char *cmd );
void test0_ksh( char *cmd );
void test1_ksh( char *cmd );
void test2_ksh( char *cmd );
void test3_ksh( char *cmd );
void test4_ksh( char *cmd );

static cmd_t ksh_cmds[] = {
  { "help", 4, help_ksh, "show this message" },
  { "reboot", 6, reboot_ksh, "reboot the computer" },
  { "0", 1, test0_ksh, "test sequence 0: text color change" },
  { "1", 1, test1_ksh, "test sequence 1: print '1' at one second cycle" },
  { "2", 1, test2_ksh, "test sequence 2: print '2' at two second cycle" },
  { "3", 1, test3_ksh, "test sequence 3: write floppy sector 0" },
  { "4", 1, test4_ksh, "test sequence 4: read floppy sector 1" },
  { NULL, 0, NULL, NULL },
};

void
cmd_ksh( char *cmd )
{
  int i;
  for( i = 0 ; ksh_cmds[i].cmd ; i++ )
    {
      if( strncmp(cmd, ksh_cmds[i].cmd, ksh_cmds[i].len) == 0 )
	{
	  ksh_cmds[i].proc( cmd );
	  return;
	}
    }

  if( *cmd )
    help_ksh( cmd );
}

void
help_ksh( char *cmd )
{
  int i;
  for( i = 0 ; ksh_cmds[i].cmd ; i++ )
    {
      printk("%12s%s\n", ksh_cmds[i].cmd, ksh_cmds[i].msg );
    }
}

void
reboot_ksh( char *cmd )
{
  reboot_system();
}

void
test0_ksh( char *cmd )
{
  static int attr = 1;
  set_console_attr( 0, attr++ );
  unlock_cpu();
}

static byte stack1[4096];
void
task1(void)
{
  unlock_cpu();
  printk("\n");
  for(;;)
    {
      sleep(1);
      printk("1");
    }
  exit();
}
static Taskinfo task1_info = {
  0x81, 0,
  (uint)task1, 0,
  0, 0, 0, 0,
  (uint)stack1 + sizeof(stack1), sizeof(stack1),
  0, 0, 0, 0,
  0,
};

static byte stack2[4096];
void
task2(void)
{
  unlock_cpu();
  printk("\n");
  for(;;)
    {
      sleep(2);
      printk("2");
    }
  exit();
}
static Taskinfo task2_info = {
  0x81, 0,
  (uint)task2, 0,
  0, 0, 0, 0,
  (uint)stack2 + sizeof(stack2), sizeof(stack2),
  0, 0, 0, 0,
  0,
};

void
test1_ksh( char *cmd )
{
  static bool run = 1;
  static Task *t;

  if( run )
    {
      task1_info.pgd = (uint)system_pgd;
      t = create_task( &task1_info );
      run = 0;
    }
  else
    {
      exit_task( t );
      run = 1;
    }
}

void
test2_ksh( char *cmd )
{
  static bool run = 1;
  static Task *t;

  if( run )
    {
      task2_info.pgd = (uint)system_pgd;
      t = create_task( &task2_info );
      run = 0;
    }
  else
    {
      exit_task( t );
      run = 1;
    }
}

void
test3_ksh( char *cmd )
{
  byte buf[2048];
  int i;

  memset( buf, 0, 2048 );
  for( i = 0 ; i < 10 ; i++ )
    {
      buf[512*0+i] = '0' + i;
      buf[512*1+i] = '0' + i;
      buf[512*2+i] = '0' + i;
      buf[512*3+i] = '0' + i;
    }
  
  printk("--INIT:%d\n", init_floppy(NULL));
  printk("--OPEN:%d\n", open_floppy(0, NULL));
  printk("--WRITE:%d\n", write_floppy_sector(0, buf, 0, 1));
}

void
test4_ksh( char *cmd )
{
  byte buf[2048];
  int i;

  memset( buf, 0, 2048 );
  for( i = 0 ; i < 10 ; i++ )
    {
      buf[512*0+i] = '0' + i;
      buf[512*1+i] = '0' + i;
      buf[512*2+i] = '0' + i;
      buf[512*3+i] = '0' + i;
    }
  
  printk("--INIT:%d\n", init_floppy(NULL));
  printk("--OPEN:%d\n", open_floppy(0, NULL));
  printk("--READ:%d\n", read_floppy_sector(0, buf, 0, 1));
}

static void
main_ksh(void)
{
  byte i = 0, input[256];
  int ch;

  unlock_cpu();
  printk("coron> ");

  for(;;)
    {
      ch = getc_system();
      if( ch == 0 )
	continue;
      putc_system(ch & 0xff);
      input[i++] = ch;
      if( ch == '\n' )
	{
	  input[i-1] = 0;
	  i = 0;
	  cmd_ksh( input );
	  printk("coron> ");
	}
    }
}

void
setup_ksh(void)
{
  ksh_task.pgd = (uint)system_pgd;
  create_task( &ksh_task );
}
