/*
** snprintf.c
*/

#include <coron.h>
#include <stdarg.h>

static char *
ntoa( int radix, int upper, int width, int pad, int minus, uint val )
{
  static char *map[] = { "0123456789abcdef", "0123456789ABCDEF" };
  static char buf[64];
  char *p = buf + sizeof(buf);

  if( width > 32 )
    width = 32;

  if( minus )
    width--;

  *--p = 0;

  do {
    width--;
    *--p = map[upper][val % radix];
    val /= radix;
  } while( val );

  if( minus && pad != '0' )
    *--p = '-';

  while( width-- > 0 )
    *--p = pad;

  if( minus && pad == '0' )
    *--p = '-';

  return p;
}

void
snprintf( char *str, uint size, const char *fmt, ... )
{
  va_list ap;
  va_start( ap, fmt );
  vsnprintf( str, size, fmt, ap );
  va_end( ap );
}

void
vsnprintf( char *str, uint size, const char *fmt, va_list ap )
{
  int i, ch, width, pad, val;
  char *p;

  i = 0;
  while( i < size && *fmt )
    {
      ch = *fmt++;

      if( ch != '%' )
	{
	  str[i++] = ch;
	  continue;
	}

      /*
       * parse % notation
       */
      width = pad = val = 0;
      ch = *fmt++;

      /* padding type */
      if( ch == ' ' || ch == '0' )
	{
	  pad = ch;
	  ch = *fmt++;
	}
      
      /* field width */
      while( '0' <= ch && ch <= '9' )
	{
	  width *= 10;
	  width += ch - '0';
	  ch = *fmt++;
	}

      /* format type */
      switch( ch )
	{
	case 'c':  // character
	  str[i++] = va_arg( ap, int );
	  break;

	case 's':  // string
	  p = va_arg( ap, char* );
	  while( i < size && *p )
	    {
	      width--;
	      str[i++] = *p++;
	    }
	  while( i < size && width-- > 0 )
	    str[i++] = ' ';
	  break;

	case 'd':  // integer
	  val = va_arg( ap, int );
	  p = ntoa( 10, 0, width, pad, (val < 0), (val < 0) ? -val : val );
	  while( i < size && *p )
	    str[i++] = *p++;
	  break;

	case 'u':  // unsigned integer
	  val = va_arg( ap, uint );
	  p = ntoa( 10, 0, width, pad, 0, val );
	  while( i < size && *p )
	    str[i++] = *p++;
	  break;

	case 'x':  // hexa-decimal
	  val = va_arg( ap, uint );
	  p = ntoa( 16, 0, width, pad, 0, val );
	  while( i < size && *p )
	    str[i++] = *p++;
	  break;

	case 'X':  // upper scale hexa-decimal
	  val = va_arg( ap, uint );
	  p = ntoa( 16, 1, width, pad, 0, val );
	  while( i < size && *p )
	    str[i++] = *p++;
	  break;

	case '%':  // %
	  str[i++] = '%';
	  break;

	case 0:
	  fmt--;
	  break;
	}
    }

  if( i == size )
    i = size - 1;

  str[i] = 0;
}
