/*
** console.c --- console driver
*/

#include <coron.h>
#include <string.h>
#include <keyboard.h>
#include <vga.h>
#include <console.h>

enum {
  TEXT_ADDR   = 0xb8000,
  TEXT_SIZE   = 8192,
  TEXT_PAGES  = 8,
  FONT_ADDR   = 0xa0000,
  FONT_SIZE   = 8192,
  FONT_PAGES  = 8,
  MAX_CONSOLE = 8,
};

enum {
  CHAR_WIDTH = 8,
  CHAR_HEIGHT = 16,
  CHAR_SIZE = 2,
  CHAR_ATTR = 7,
  CHAR_COLS = 80,
  CHAR_ROWS = 25,
};
  
static byte *textbuf = (byte*)TEXT_ADDR;
static byte *fontbuf = (byte*)FONT_ADDR;

typedef struct {
  byte *textbuf;
  byte *fontbuf;
  int char_attr;
  int x, y;
  int modifier;
} Console;

static Console con[MAX_CONSOLE];
static int active;

int
getc_console( int id )
{
  int ksym;
  ksym = get_keyboard_ksym( &(con[id].modifier) );
  if( ksym > 0 )
    return ksym;
  return 0;
}

void
putc_console( int id, int ch )
{
  BEGIN_CPULOCK

  int x, y, visible, line, block, pos;
  byte *textbuf;
  
  textbuf = con[id].textbuf;
  x = con[id].x;
  y = con[id].y;

  switch( ch )
    {
    case '\n':
      x = -1;
      y++;
      visible = 0;
      break;
    case 0x20 ... 0x7e:
      x++;
      visible = 1;
      break;
    default:
      ch = '~';
      x++;
      visible = 1;
      break;
    }
  
  if( visible )
    {
      if( x >= CHAR_COLS )
	{
	  /* line feed */
	  x = 0;
	  y++;
	}
      if( y >= CHAR_ROWS )
	{
	  /* scroll */
	  line = CHAR_COLS * CHAR_SIZE;
	  block = CHAR_COLS * CHAR_ROWS * CHAR_SIZE - line;
	  memcpy( textbuf, textbuf + line, block );
	  memset( textbuf + block, 0, line );
	  y = CHAR_ROWS - 1;
	}
      if( x >= 0 && y >= 0 )
	{
	  /* put */
	  pos = x + y * CHAR_COLS;
	  textbuf[pos*CHAR_SIZE] = ch;
	  textbuf[pos*CHAR_SIZE+1] = con[id].char_attr;
	  set_cursor_addr( id * TEXT_SIZE / 2 + pos + 1 );
	}
    }

  con[id].x = x;
  con[id].y = y;

  END_CPULOCK
}

void
set_console_attr( int id, int attr )
{
  con[id].char_attr = attr;
}

void
clear_console( int id )
{
  int i;
  for( i = 0 ; i < CHAR_COLS * CHAR_ROWS ; i++ )
    {
      con[id].textbuf[ i * CHAR_SIZE ] = ' ';
      con[id].textbuf[ i * CHAR_SIZE + 1 ] = con[id].char_attr;
    }
}

void
switch_console( int id )
{
  active = id;
  set_start_addr( id * TEXT_SIZE / 2 );
}

void
setup_console(void)
{
  setup_keyboard();
  setup_vga();
}

Eint
open_console( void *arg )
{
  int id = (int)arg;

  con[id].textbuf = textbuf + id * TEXT_SIZE;
  con[id].fontbuf = fontbuf;
  con[id].char_attr = CHAR_ATTR;
  con[id].x = -1;
  con[id].y = 0;
  con[id].modifier = 0;
  clear_console(id);
  switch_console(id);
  return id;
}

Err
close_console( int id )
{
  return 0;
}
