/*
** cpu.h --- CPU control operations
*/
#ifndef CPU_H
#define CPU_H

Inline byte
inb( uint port )
{
  byte ret;
  Asm("inb %%dx, %%al": "=a"(ret): "d"(port));
  return ret;
}

Inline void
outb( uint port, byte data )
{
  Asm("outb %%al, %%dx":: "a"(data), "d"(port));
}

Inline word16
inw( uint port )
{
  word16 data;
  Asm("inw %%dx, %%ax": "=a"(data): "d"(port));
  return data;
}

Inline void
outw( uint port, word16 data )
{
  Asm("outw %%ax, %%dx":: "a"(data), "d"(port));
}

Inline void
lock_cpu(void)
{
  Asm("cli");
}

Inline void
unlock_cpu(void)
{
  Asm("sti");
}

Inline bool
cpu_has_locked(void)
{
  uint eflags;
  Asm("pushf");
  Asm("popl %0": "=m"(eflags));
  return((eflags & 0x200) == 0);
}

#define hold_lock_cpu( flag )	\
{				\
  flag = cpu_has_locked();	\
  if( flag == 0 )		\
    lock_cpu();			\
}

#define hold_unlock_cpu( flag )	\
{				\
  if( flag == 0 )		\
    unlock_cpu();		\
}

#define BEGIN_CPULOCK\
  {\
    bool hold_locked;\
    hold_lock_cpu(hold_locked);

#define END_CPULOCK\
    hold_unlock_cpu(hold_locked);\
  }

Inline void
halt_cpu(void)
{
  Asm("hlt");
}

Inline void
disable_nmi(void)
{
  outb(0x70, 0x80);
}

Inline void
enable_nmi(void)
{
  outb(0x70, 0x00);
}

Inline void
set_cr3( void *cr3 )
{
  Asm("mov %0, %%cr3"::"r"(cr3));
}

Inline void *
get_cr3(void)
{
  void *cr3;
  Asm("mov %%cr3, %0":"=r"(cr3));
  return cr3;
}

Inline void
paging_on(void)
{
  Asm("mov %cr0, %eax");
  Asm("or  $0x80000000, %eax");
  Asm("mov %eax, %cr0");
}

Inline void
paging_off(void)
{
  Asm("mov %cr0, %eax");
  Asm("and $0x7fffffff, %eax");
  Asm("mov %eax, %cr0");
}

Inline bool
paging_has_enabled(void)
{
  bool enabled;
  Asm("mov %%cr0, %0":"=a"(enabled));
  return enabled & 0x80000000;
}

/*
 * CPUTLB(Translation Lookaside Buffer)PTEå夵ƤΤǡ
 * PDEPTEѹϡδؿƤǥå򥯥ꥢƤ
 */
Inline void
flush_page_cache(void)
{
  Asm("mov %cr3, %eax");
  Asm("mov %eax, %cr3");
}

#endif /* CPU_H */
