/*
** desc.c --- descriptor tables
**
** References:
** [1] Intel Architecture Software Developer's Manual, Volume 3:
**     System Programming (section 2.1, 2.4, 3.4, 3.5, 4.8, 5.9, 6.2)
**     http://developer.intel.com/design/pentiumii/manuals/243192.htm
*/

#include <coron.h>
#include <desc.h>

void
set_sdesc( sdesc_t *desc, word32 base, word32 limit, uint type, uint dpl )
{
  desc->baseL24	= base & 0xffffff;
  desc->baseH8	= (base >> 24) & 0xff;
  desc->limitL16= limit & 0xffff;
  desc->limitH4	= (limit >> 16) & 0xf;
  desc->type	= type;
  desc->dpl	= dpl;
  desc->s	= 1;	/* code/data segment */
  desc->p	= 1;	/* present */
  desc->avl	= 0;	/* available for use by system */
  desc->x	= 0;	/* reserved zero */
  desc->d	= 1;	/* use 32 bits segment */
  desc->g	= 1;	/* 4KB limit scale */
}

void
set_tss32_sdesc( sdesc_t *desc, word32 base, word32 limit, uint dpl )
{
  desc->baseL24	= base & 0xffffff;
  desc->baseH8	= (base >> 24) & 0xff;
  desc->limitL16= limit & 0xffff;
  desc->limitH4	= (limit >> 16) & 0xf;
  desc->type	= GATE_TSS32;
  desc->dpl	= dpl;
  desc->s	= 0;	/* system segment */
  desc->p	= 1;	/* present */
  desc->avl	= 0;	/* available for use by system */
  desc->x	= 0;	/* reserved zero */
  desc->d	= 0;	/* zero for TSS descriptor */
  desc->g	= 1;	/* 4KB limit scale */
}

void
set_gdesc( gdesc_t *desc, word16 sel, word32 offset, uint type, uint dpl )
{
  desc->selector  = sel;
  desc->offsetL16 = offset & 0xffff;
  desc->offsetH16 = (offset >> 16) & 0xffff;
  desc->count	= 0;
  desc->type	= type;
  desc->dpl	= dpl;
  desc->s	= 0;	/* system segment or gate */
  desc->p	= 1;	/* present */
}
