/*
** desc.h --- descriptor tables
*/
#ifndef DESC_H
#define DESC_H

/* code/data descriptor types */
enum {
  DESC_DATA	= 0x00,
  DESC_EXPAND	= 0x04,
  DESC_WRITABLE	= 0x02,
  DESC_CODE	= 0x08,
  DESC_CONFORM  = 0x04,
  DESC_READABLE	= 0x02,
  DESC_ACCESSED	= 0x01,
};

/* system/gate descriptor types */
enum {
  GATE_TSS16		= 1,
  GATE_LDT		= 2,
  GATE_TSS16_BUSY	= 3,
  GATE_CALL16		= 4,
  GATE_INTR16		= 6,
  GATE_TRAP16		= 7,
  GATE_TSS32		= 9,
  GATE_TSS32_BUSY	= 11,
  GATE_CALL32		= 12,
  GATE_INTR32		= 14,
  GATE_TRAP32		= 15,
};

/* code/data segment descriptor table */
typedef struct {
  unsigned limitL16:16;	/* limit 15:0 */
  unsigned baseL24:24 __PACKED__;	/* base 23:0 */
  unsigned type:4;	/* type */
  unsigned s:1;		/* descriptor type (1 for code/data segment) */
  unsigned dpl:2;	/* descriptor privilege level */
  unsigned p:1;		/* present bit */
  unsigned limitH4:4;	/* limit 19:16 */
  unsigned avl:1;	/* available for use by system software */
  unsigned x:1;		/* reserved zero */
  unsigned d:1;		/* default operation size 16/32 bits segment */
  unsigned g:1;		/* granularity (limit scale) bytes/4KB */
  unsigned baseH8:8;	/* base 31:24 */
} sdesc_t;

/* system segment or gate descriptor table */
typedef struct {
  unsigned offsetL16:16;/* offset 15:0 */
  unsigned selector:16;	/* selector */
  unsigned count:8;	/* count for stack switing */
  unsigned type:4;	/* type */
  unsigned s:1;		/* descriptor type (0 for system segment) */
  unsigned dpl:2;	/* descriptor privilege level */
  unsigned p:1;		/* present bit */
  unsigned offsetH16:16;/* offset 31:16 */
} gdesc_t;

/* descriptor table pointer */
typedef struct {
  unsigned limit:16;
  unsigned base:32 __PACKED__;
} descptr_t;

/* descriptor table entries */
enum {
  GDTNUM = 8,	/* global */
  IDTNUM = 256,	/* interrupt */
};

/* GDT selectors */
enum {
  GDT_EMPTY       = 0x00,
  GDT_KERNEL_CODE = 0x10,
  GDT_KERNEL_DATA = 0x18,
  GDT_USER_CODE   = 0x20,
  GDT_USER_DATA   = 0x28,
  GDT_CUR_TSS     = 0x30,
  GDT_OLD_TSS     = 0x38,
};

void set_sdesc( sdesc_t *desc, word32 base, word32 limit, uint type, uint dpl );
void set_tss32_sdesc( sdesc_t *desc, word32 base, word32 limit, uint dpl );
void set_gdesc( gdesc_t *desc, word16 sel, word32 offset, uint type, uint dpl );

#endif /* DESC_H */
