/*
** dma.c --- DMA controller
*/

#include <coron.h>

/* DMA controller registers */
enum {
  DMA1_BASE	= 0x00,
  DMA1_STAT	= 0x08,
  DMA1_CMD	= 0x08,
  DMA1_REQ	= 0x09,
  DMA1_MASK	= 0x0a,
  DMA1_MODE	= 0x0b,
  DMA1_CLRFF	= 0x0c,
  DMA1_RESET	= 0x0d,

  DMA2_BASE	= 0xc0,
  DMA2_STAT	= 0xd0,
  DMA2_CMD	= 0xd0,
  DMA2_REQ	= 0xd2,
  DMA2_MASK	= 0xd4,
  DMA2_MODE	= 0xd6,
  DMA2_CLRFF	= 0xd8,
  DMA2_RESET	= 0xda,

  DMA_PAGE0	= 0x87,
  DMA_PAGE1	= 0x83,
  DMA_PAGE2	= 0x81,
  DMA_PAGE3	= 0x82,
  DMA_PAGE4	= 0x8f,
  DMA_PAGE5	= 0x8b,
  DMA_PAGE6	= 0x89,
  DMA_PAGE7	= 0x8a,
};

Inline void
iodelay(void)
{
  inb(0x80);
}

Inline void
clear_dma_flipflop( byte channel )
{
  byte reg_clrff = (channel < 4) ? DMA1_CLRFF : DMA2_CLRFF;

  outb( reg_clrff, 0 );
  iodelay();
}

void
enable_dma( byte channel )
{
  byte reg_mask = (channel < 4) ? DMA1_MASK : DMA2_MASK;
  outb( reg_mask, channel );
  iodelay();
}

void
disable_dma( byte channel )
{
  byte reg_mask = (channel < 4) ? DMA1_MASK : DMA2_MASK;
  outb( reg_mask, channel | BV(2) );
  iodelay();
}

void
setup_dma(void)
{
  /* reset DMA controllers */
  outb( DMA1_RESET, 0 );
  iodelay();
  //BUG?
  //outb( DMA2_RESET, 0 );
  //iodelay();
}

/* dma memory area must be on 128KB boundary physical memory address */
void
set_dma_area( byte channel, uint dma_addr, uint dma_count )
{
  static const byte reg_page[8] = {
    DMA_PAGE0, DMA_PAGE1, DMA_PAGE2, DMA_PAGE3,
    DMA_PAGE4, DMA_PAGE5, DMA_PAGE6, DMA_PAGE7,
  };
  byte reg_addr, reg_count;

  if( channel < 4 )
    {
      reg_addr = DMA1_BASE + channel * 2;
      reg_count = reg_addr + 1;
    }
  else
    {
      reg_addr = DMA2_BASE + channel * 4;
      reg_count = reg_addr + 2;
    }

  /* set base address */
  clear_dma_flipflop(channel);
  outb( reg_addr, (dma_addr & 0x0000ff) );
  iodelay();
  outb( reg_addr, (dma_addr & 0x00ff00) >> 8 );
  iodelay();

  /* set base count */
  clear_dma_flipflop(channel);
  outb( reg_count, (dma_count & 0x00ff) );
  iodelay();
  outb( reg_count, (dma_count & 0xff00) >> 8 );
  iodelay();

  /* set dma page */
  outb( reg_page[channel], (dma_addr & 0xff0000) >> 16 );
  iodelay();
}

void
set_dma_mode( byte channel, byte trans, byte mode, bool autoinit )
{
  byte reg_mode = (channel < 4) ? DMA1_MODE : DMA2_MODE;

  outb( reg_mode, channel | (trans << 2) |
	(autoinit ? BV(4): 0) | (mode << 6) );
  iodelay();
}
