/*
** excp.c --- exception handlers
*/

#include <coron.h>
#include "debug.h"
#include "intr.h"

typedef struct {
  word32 gs, fs, es, ds, ss;		// push values of each register
  word32 edi, esi, ebp, esp;		// pushad info
  word32 ebx, edx, ecx, eax;		// ..
  word32 errcode, eip, cs, eflags;	// exception info
} Excinfo;

static void
excp_abort( Excinfo *i, const char *msg )
{
  printk("\n*** Exception: %s ***\n"
	 "Info    eflags: %08x, errcode: %08x\n"
	 "Inst.   cs:  %08x, eip: %08x\n"
	 "Stack   ss:  %08x, esp: %08x, ebp: %08x\n"
	 "Data    ds:  %08x, es:  %08x, fs:  %08x, gs:  %08x\n"
	 "        eax: %08x, ebx: %08x, ecx: %08x, edx: %08x\n"
	 "        esi: %08x, edi: %08x\n",
	 msg,
	 i->eflags, i->errcode,
	  i->cs, i->eip,
	 i->ss, i->esp +16, i->ebp,
	  i->ds, i->es, i->fs, i->gs,
	 i->eax, i->ebx, i->ecx, i->edx,
	 i->esi, i->edi );

  printk("INSTRUCTION DUMP");
  dump((void*)(i->eip), 16);

  printk("STACK DUMP");
  stack_dump((void*)(i->esp +16), 4);

  panic("*** System Abort ***");
}

#define EXCEPTION_ABORT(func, msg)\
  EXCEPTION(func); void func( Excinfo info ){ excp_abort(&info, msg); }

EXCEPTION_ABORT( excp_div, "Divide Error" );
EXCEPTION_ABORT( excp_deb, "Debug" );
EXCEPTION_ABORT( excp_nmi, "NMI Interrupt" );
EXCEPTION_ABORT( excp_brk, "Breakpoint" );
EXCEPTION_ABORT( excp_ovf, "Overflow" );
EXCEPTION_ABORT( excp_ran, "BOUND Range Exceeded" );
EXCEPTION_ABORT( excp_opc, "Undefined Opcode" );
EXCEPTION_ABORT( excp_fpu, "No Math Coprocessor" );
EXCEPTION_ABORT( excp_dbl, "Double Fault" );
EXCEPTION_ABORT( excp_cso, "Coprocessor Segment Overrun" );
EXCEPTION_ABORT( excp_tss, "Invalid TSS" );
EXCEPTION_ABORT( excp_seg, "Segment Not Present" );
EXCEPTION_ABORT( excp_stk, "Stack-Segment Fault" );
EXCEPTION_ABORT( excp_pro, "General Protection" );
EXCEPTION_ABORT( excp_pgf, "Page Fault" );
EXCEPTION_ABORT( excp_res, "Reserved" );
EXCEPTION_ABORT( excp_mth, "Math Fault" );
EXCEPTION_ABORT( excp_agc, "Alignment Check" );
EXCEPTION_ABORT( excp_mac, "Machine Check" );
EXCEPTION_ABORT( excp_sim, "Streaming SIMD Extensions" );

void
setup_excp(void)
{
  /* define exception handlers */
  define_intr(  0, EXCP_ENTRY(excp_div) );	// Divide Error
  define_intr(  1, EXCP_ENTRY(excp_deb) );	// Debug
  define_intr(  2, EXCP_ENTRY(excp_nmi) );	// NMI Interrupt
  define_intr(  3, EXCP_ENTRY(excp_brk) );	// Breakpoint
  define_intr(  4, EXCP_ENTRY(excp_ovf) );	// Overflow
  define_intr(  5, EXCP_ENTRY(excp_ran) );	// BOUND Range Exceeded
  define_intr(  6, EXCP_ENTRY(excp_opc) );	// Undefined Opcode
  define_intr(  7, EXCP_ENTRY(excp_fpu) );	// No Math Coprocessor
  define_intr(  8, EXCP_ENTRY(excp_dbl) );	// Double Fault
  define_intr(  9, EXCP_ENTRY(excp_cso) );	// Coprocessor Segment Overrun
  define_intr( 10, EXCP_ENTRY(excp_tss) );	// Invalid TSS
  define_intr( 11, EXCP_ENTRY(excp_seg) );	// Segment Not Present
  define_intr( 12, EXCP_ENTRY(excp_stk) );	// Stack-Segment Fault
  define_intr( 13, EXCP_ENTRY(excp_pro) );	// General Protection
  define_intr( 14, EXCP_ENTRY(excp_pgf) );	// Page Fault
  define_intr( 15, EXCP_ENTRY(excp_res) );	// Reserved
  define_intr( 16, EXCP_ENTRY(excp_mth) );	// Math Fault
  define_intr( 17, EXCP_ENTRY(excp_agc) );	// Alignment Check
  define_intr( 18, EXCP_ENTRY(excp_mac) );	// Machine Check
  define_intr( 19, EXCP_ENTRY(excp_sim) );	// Streaming SIMD Extensions
}
