/*
** floppy.c --- floppy disk driver
*/

#include <coron.h>
#include <string.h>
#include <irq.h>
#include <intr.h>
#include <dma.h>
#include <task.h>

enum {
  MAX_DRIVE = 2,
  MAX_HEAD = 2,
  SECSIZE = 512,
  DMA_BUFFER_ADDR = 0x20000, ///DEBUG
};

/* FDC registers */
enum {
  SRA = 0x3f0, // Status Register A
  SRB = 0x3f1, // Status Register B
  DOR = 0x3f2, // Digital Output Register
  DSR = 0x3f3, // Drive Status Register
  MSR = 0x3f4, // Main Status Register
  IOR = 0x3f5, // Data port
  DIR = 0x3f7, // Digital Input Register
  CCR = 0x3f7, // Configuration Control Register
};

/* data rate */
enum {
  RATE_500K = 0,
  RATE_300K = 1,
  RATE_250K = 2,
  RATE_1M = 3,
};

/* media number */
enum {
  MEDIA_UNDEF = 0,
  MEDIA_2M88 = 1,
  MEDIA_1M44 = 2,
  MEDIA_720K = 3,
};

/* sector size code */
enum {
  SSZ_512 = 2,
  SSZ_1024 = 3,
};

typedef struct {
  byte sector_size_code;
  byte sectors_per_track;
  byte gap_length;
  byte format_gap_length;
  byte data_rate;
} MediaInfo;

static const MediaInfo media_info[4] =
  {
    [MEDIA_UNDEF]{ SSZ_512, 18, 0x1b, 0x6c, RATE_500K },
    [MEDIA_2M88] { SSZ_512, 36, 0x1b, 0x50, RATE_1M   },
    [MEDIA_1M44] { SSZ_512, 18, 0x1b, 0x6c, RATE_500K },
    [MEDIA_720K] { SSZ_512,  9, 0x2a, 0x50, RATE_250K },
  };

typedef struct {
  bool mount;
  byte media;
  MediaInfo *media_info;
} DriveInfo;

static DriveInfo drive_info[MAX_DRIVE];
static int intr_fdc_count;
static void *dma_buffer;

Inline void
iodelay(void)
{
  inb(0x80);
}

Inline void
motor_on( byte drive )
{
  outb(DOR, (inb(DOR) | 0x0c) | (BV(4) << drive));
}

Inline void
motor_off( byte drive )
{
  outb(DOR, (inb(DOR) | 0x0c) &~ (BV(4) << drive));
}

Inline bool
check_disk_changed(void)
{
  return (inb(DIR) & BV(7)) ? 1 : 0;
}

Inline void
set_data_rate( byte rate )
{
  outb(CCR, rate);
}

/* issues an interrupt and result 2 byte */
static void
reset_fdc(void)
{
  outb(DOR, 0x08);
  iodelay();
  outb(DOR, 0x0c);
  iodelay();
  outb(CCR, 0x00);
  iodelay();
}

static void
select_drive( byte drive )
{
  outb(DOR, 0x0c | (BV(4) << drive) | drive);
}

static int
read_fdc( byte *data, int count )
{
  int i, n;
  for( i = 0 ; i < count ; i++ )
    {
      n = 999;
      for(;;)
	{
	  iodelay();
	  if( (inb(MSR) & 0xc0) == 0xc0 )
	    {
	      data[i] = inb(IOR);
	      break;
	    }
	  if( n-- < 0 )
	    return -1;
	}
    }
  return 0;
}

static int
write_fdc( byte *data, int count )
{
  int i, n;
  for( i = 0 ; i < count ; i++ )
    {
      n = 999;
      for(;;)
	{
	  iodelay();
	  if( (inb(MSR) & 0xc0) == 0x80 )
	    {
	      outb(IOR, data[i]);
	      break;
	    }
	  if( n-- < 0 )
	    return -1;
	}
    }
  return 0;
}

/* issues an interrupt and result 7 byte */
static int
cmd_standard( byte cmd_num, byte drive, byte head, byte cylinder,
	      byte first_sector, byte last_sector )
{
  byte cmd[9];
  cmd[0] = cmd_num;
  cmd[1] = (head << 2) | drive;
  cmd[2] = cylinder;
  cmd[3] = head;
  cmd[4] = first_sector;
  cmd[5] = drive_info[drive].media_info->sector_size_code;
  cmd[6] = last_sector;
  cmd[7] = drive_info[drive].media_info->gap_length;
  cmd[8] = 0xff;
  return write_fdc(cmd, 9);
}

/* no result */
static int
cmd_specify(void)
{
  byte cmd[3];
  cmd[0] = 0x03;
  //cmd[1] = 0xa1;
  //cmd[2] = 0x04;
  cmd[1] = 0xc1;
  cmd[2] = 0x10;
  return write_fdc(cmd, 3);
}

/* result 1 byte */
static int
cmd_sense_drive_status( byte drive, byte head )
{
  byte cmd[2];
  cmd[0] = 0x04;
  cmd[1] = (head << 2) | drive;
  return write_fdc(cmd, 2);
}

/* issues an interrupt */
static int
cmd_recalibrate( byte drive )
{
  byte cmd[2];
  cmd[0] = 0x07;
  cmd[1] = drive;
  return write_fdc(cmd, 2);
}

/* result 2 byte */
static int
cmd_sense_intr_stat(void)
{
  byte cmd[1];
  cmd[0] = 0x08;
  return write_fdc(cmd, 1);
}

/* result 10 byte */
static int
cmd_dump_register(void)
{
  byte cmd[1];
  cmd[0] = 0x0e;
  return write_fdc(cmd, 1);
}

/* issues an interrupt */
static int
cmd_seek( byte drive, byte head, byte cylinder )
{
  byte cmd[3];
  cmd[0] = 0x0f;
  cmd[1] = (head << 2) | drive;
  cmd[2] = cylinder;
  return write_fdc(cmd, 3);
}

/* result 1 byte */
static int
cmd_version(void)
{
  byte cmd[1];
  cmd[0] = 0x10;
  return write_fdc(cmd, 1);
}

/* no result */
static int
cmd_configure( bool implied_seek, byte fifo_size, bool polling, byte precomp )
{
  byte cmd[4];
  cmd[0] = 0x13;
  cmd[1] = 0;
  cmd[2] = (implied_seek ? BV(6) : 0) | (fifo_size ? 0 : BV(5)) |
           (polling ? 0 : BV(4)) | fifo_size;
  cmd[3] = precomp;
  return write_fdc(cmd, 4);
}

/* issues an interrupt and result 7 byte */
static int
cmd_format_track( byte drive, byte head )
{
  byte cmd[6];
  cmd[0] = 0x4d;
  cmd[1] = (head << 2) | drive;
  cmd[2] = drive_info[drive].media_info->sector_size_code;
  cmd[3] = drive_info[drive].media_info->sectors_per_track;
  cmd[4] = drive_info[drive].media_info->gap_length;
  cmd[5] = 0xf6;  // initial data byte
  return write_fdc(cmd, 6);
}

/* issues an interrupt and result 7 byte */
static int
cmd_read_id( byte drive, byte head )
{
  byte cmd[2];
  cmd[0] = 0x4a;
  cmd[1] = (head << 2) | drive;
  return write_fdc(cmd, 2);
}

/* issues an interrupt and result 7 byte */
static int
cmd_write_data( byte drive, byte head, byte cylinder,
		byte first_sector, byte last_sector )
{
  return cmd_standard(0xc5, drive, head, cylinder, first_sector, last_sector);
}

/* issues an interrupt and result 7 byte */
static int
cmd_write_deleted_data( byte drive, byte head, byte cylinder,
			byte first_sector, byte last_sector )
{
  return cmd_standard(0xc9, drive, head, cylinder, first_sector, last_sector);
}

/* issues an interrupt and result 7 byte */
static int
cmd_read_data( byte drive, byte head, byte cylinder,
	       byte first_sector, byte last_sector )
{
  return cmd_standard(0xe6, drive, head, cylinder, first_sector, last_sector);
}

INTERRUPT( intr_fdc );
Static void
intr_fdc(void)
{
  intr_fdc_count++;
  next_irq(IRQ_KBC);
}

static int
wait_for_intr(void)
{
  uint i;
  for( i = 0 ; i < 3000 ; i++ )
    {
      if( intr_fdc_count > 0 )
	{
	  intr_fdc_count--;
	  printk("wait: %dms\n", i);
	  return i;
	}
      usleep(1000);
    }
  printk("wait: %dms *** TIMEOUT ***\n", i);
  return -1;
}

static int
wait_for_intr_stat( byte *data )
{
  byte stat[2];
  return_if_neg( wait_for_intr() );
  return_if_neg( cmd_sense_intr_stat() );
  return_if_neg( read_fdc(stat,2) );
  if( data )
    memcpy( data, stat, 2 );
  printk("intr_stat: %02xh, %02xh\n", stat[0], stat[1]);
  return 0;
}

static int
wait_for_intr_result( byte *data )
{
  byte result[7];
  return_if_neg( wait_for_intr() );
  return_if_neg( read_fdc(result,7) );
  if( data )
    memcpy( data, result, 7 );
  printk("intr_res: %02xh, %02xh, %02xh, %02xh, %02xh, %02xh, %02xh\n",
	 result[0], result[1], result[2], result[3],
	 result[4], result[5], result[6]);
  return 0;
}

void
setup_floppy(void)
{
  intr_fdc_count = 0;
  memset( drive_info, 0, sizeof(drive_info) );
  dma_buffer = (void*)DMA_BUFFER_ADDR;
  CHECK_ABORT( dma_buffer, "cannot allocate floppy DMA buffer" );
  
  define_intr( IRQ_TO_INTN(IRQ_FDC), INTR_ENTRY(intr_fdc) );
  enable_irq(IRQ_FDC);
}

Err
init_floppy( void *arg )
{
  /* reset controller */
  reset_fdc();
  return_if_neg( wait_for_intr_stat(NULL) );

  /* specify */
  return_if_neg( cmd_specify() );
  return 0;
}

Eint
open_floppy( byte minor, void *arg )
{
  byte stat[2];
  byte d;

  CHECK_ARG( minor < MAX_DRIVE );
  d = minor;
  
  /* change drive */
  select_drive( d );

  /* recalibrate 2 times */
  printk("rec1\n");
  goto_if_neg( cmd_recalibrate(d), error );
  goto_if_neg( wait_for_intr_stat(NULL), error )

  printk("rec2\n");
  goto_if_neg( cmd_recalibrate(d), error );
  goto_if_neg( wait_for_intr_stat(stat), error )
  if( stat[0] != (0x20 | d) )
    goto error;  //ERROR: media not found or invalidate

  /* set media information */
  ///TODO: needs to identify media type
  drive_info[d].media = MEDIA_1M44;
  drive_info[d].media_info = (MediaInfo*)&media_info[MEDIA_1M44];
  set_data_rate( drive_info[d].media_info->data_rate );

  /* motor off */
  motor_off(d);
  return 0;

 error:
  motor_off(d);
  return -1;
}

Eint
write_floppy_sector( byte minor, void *buf, uint sector, uint count )
{
  byte stat[2], result[7];
  byte d, h, c, s, spt;
  int i;

  CHECK_ARG( minor < MAX_DRIVE );
  d = minor;
  spt = drive_info[d].media_info->sectors_per_track;
  
  /* change drive */
  select_drive( d );

  /* write sectors */
  for( i = 0 ; i < count ; i++ )
    {
      c = (sector + i) / spt / MAX_HEAD;
      h = (sector + i) / spt % MAX_HEAD;
      s = (sector + i) % spt + 1;

      /* copy to DMA buffer */
      memcpy( dma_buffer, buf + i*SECSIZE, SECSIZE );

      /* setup DMA channel 2 */
      printk("setup DMA channel 2\n");
      disable_dma(2);
      set_dma_area(2, (uint)dma_buffer, SECSIZE-1);
      set_dma_mode(2, DMA_IOREAD, DMA_SINGLE, FALSE );
      enable_dma(2);

      /* seek */
      printk("seek\n");
      goto_if_neg( cmd_seek(d, h, c), error );
      goto_if_neg( wait_for_intr_stat(stat), error );
      
      /* read id */
      printk("read_id\n");
      goto_if_neg( cmd_read_id(d, h), error );
      goto_if_neg( wait_for_intr_result(result), error );

      /* write sector */
      printk("write d:c:h:s = %d:%d:%d:%d\n", d, c, h, s );
      goto_if_neg( cmd_write_data(d, h, c, s, s), error );
      goto_if_neg( wait_for_intr_result(result), error );
    }

  motor_off(d);
  return 0;

 error:
  motor_off(d);
  return -1;
}

Eint
read_floppy_sector( byte minor, void *buf, uint sector, uint count )
{
  byte stat[2], result[7];
  byte d, h, c, s, spt;
  int i;

  CHECK_ARG( minor < MAX_DRIVE );
  d = minor;
  spt = drive_info[d].media_info->sectors_per_track;
  
  /* change drive */
  select_drive( d );

  /* read sectors */
  for( i = 0 ; i < count ; i++ )
    {
      c = (sector + i) / spt / MAX_HEAD;
      h = (sector + i) / spt % MAX_HEAD;
      s = (sector + i) % spt + 1;

      /* setup DMA channel 2 */
      printk("setup DMA channel 2\n");
      disable_dma(2);
      set_dma_mode(2, DMA_IOWRITE, DMA_SINGLE, FALSE );
      set_dma_area(2, (uint)dma_buffer, SECSIZE-1);
      enable_dma(2);

      /* seek */
      printk("seek\n");
      goto_if_neg( cmd_seek(d, h, c), error );
      goto_if_neg( wait_for_intr_stat(stat), error );
      
      /* read id */
      printk("read_id\n");
      goto_if_neg( cmd_read_id(d, h), error );
      goto_if_neg( wait_for_intr_result(result), error );

      /* read sector */
      printk("write d:c:h:s = %d:%d:%d:%d\n", d, c, h, s );
      goto_if_neg( cmd_read_data(d, h, c, s, s), error );
      goto_if_neg( wait_for_intr_result(result), error );

      /* copy from DMA buffer */
      memcpy( buf + i*SECSIZE, dma_buffer, SECSIZE );
    }

  motor_off(d);
  return 0;

 error:
  motor_off(d);
  return -1;
}
