/*
** irq.c --- interrupt request management using i8259A
*/

#include <coron.h>
#include <irq.h>

/* i8259A registers */
enum {
  M0 = 0x20,
  M1 = 0x21,
  S0 = 0xA0,
  S1 = 0xA1
};

#define IRQMASK( irq )	(1 << (irq & 7))

void
setup_irq(void)
{
  /* master */
  outb( M0, 0x11 );
  outb( M1, IRQBASE );
  outb( M1, 0x04 );
  outb( M1, 0x11 );

  /* slave */
  outb( S0, 0x11 );
  outb( S1, IRQBASE8 );
  outb( S1, 0x02 );
  outb( S1, 0x01 );

  /* mask all requests */
  outb( M1, ~(0x04) );
  outb( S1, ~(0x00) );
}

void
eoi_irq( uint irq )
{
  BEGIN_CPULOCK

  if( irq < 8 )
    {
      /* send master EOI */
      outb( M1, inb(M1) | IRQMASK(irq) );
      outb( M0, 0x20 );
      outb( M0, 0x0b );
    }
  else
    {
      /* send slave EOI */
      outb( S1, inb(S1) | IRQMASK(irq) );
      outb( S0, 0x20 );
      outb( S0, 0x0b );

      /* send master EOI when there are no remaining slave requests */
      if( inb(S0) == 0 )
	outb( M0, 0x20 );
    }

  END_CPULOCK
}

void
disable_irq( uint irq )
{
  BEGIN_CPULOCK

  if( irq < 8 )
    outb( M1, inb(M1) | IRQMASK(irq) );
  else
    outb( S1, inb(S1) | IRQMASK(irq) );

  END_CPULOCK
}

void
enable_irq( uint irq )
{
  BEGIN_CPULOCK

  if( irq < 8 )
    outb( M1, inb(M1) & ~IRQMASK(irq) );
  else
    outb( S1, inb(S1) & ~IRQMASK(irq) );

  END_CPULOCK
}
