/*
** keyboard.c --- keyboard driver
*/

#include <coron.h>
#include <string.h>
#include <buffer.h>
#include <irq.h>
#include <intr.h>
#include <keyboard.h>

enum {
/* registers */
  KBC_DAT		= 0x60,
  KBC_CTL		= 0x64,
  KBC_CMD		= 0x64,
/* data commands */
  KBC_SET_MOUSE_1_1	= 0xe6,
  KBC_SET_MOUSE_1_2	= 0xe7,
  KBC_SET_MOUSE_RESOL	= 0xe8,
  KBC_GET_MOUSE_INFO	= 0xe9,
  KBC_LED_WRITE		= 0xed,
  KBC_ECHO		= 0xee,
  KBC_ALT_SCAN		= 0xf0,
  KBC_READ_ID		= 0xf2,
  KBC_SET_RATE		= 0xf3,
  KBC_ENABLE		= 0xf4,
  KBC_DEFAULT		= 0xf6,
  KBC_RESET		= 0xff,
/* status bits */
  KBC_PARITY		= 0x80,
  KBC_TMOUT		= 0x40,
  KBC_MOUSE		= 0x20,
  KBC_LOCK		= 0x10,
  KBC_PORT		= 0x08,
  KBC_SYSTEST		= 0x04,
  KBC_IBF		= 0x02,
  KBC_OBF		= 0x01,
  KBC_BUSY		= 0x03,
/* commands */
  KBC_GETCMD		= 0x20,
  KBC_SETCMD		= 0x60,
  KBC_INSTPWD		= 0xa4,
  KBC_LOADPWD		= 0xa5,
  KBC_CHKPWD		= 0xa6,
  KBC_DIS_MOUSE		= 0xa7,
  KBC_ENA_MOUSE		= 0xa8,
  KBC_TEST_MOUSE	= 0xa9,
  KBC_TEST		= 0xaa,
  KBC_IFCTEST		= 0xab,
  KBC_DUMP		= 0xac,
  KBC_DIS_KEYBD		= 0xad,
  KBC_ENA_KEYBD		= 0xae,
  KBC_READ_P1		= 0xc0,
  KBC_POLL_P1L		= 0xc1,
  KBC_POLL_P1H		= 0xc2,
  KBC_READ_P2		= 0xd0,
  KBC_WRITE_P2		= 0xd1,
  KBC_WBUF_KEYBD	= 0xd2,
  KBC_WBUF_MOUSE	= 0xd3,
  KBC_WRITE_MOUSE	= 0xd4,
  KBC_DIS_A20		= 0xdd,
  KBC_ENA_A20		= 0xdf,
  KBC_READ_TEST		= 0xe0,
  KBC_REBOOT		= 0xfe,
  KBC_PULSE		= 0xff,
/* command bits */
  KBC_TRANS_CODE	= 0x40,
  KBC_DIS_MOUSEB	= 0x20,
  KBC_DIS_KEYBDB	= 0x10,
  KBC_FLAG		= 0x04,
  KBC_INTR_MOUSE	= 0x02,
  KBC_INTR_KEYBD	= 0x01,
};

static word16 ksym_table[256] = {
//  0     1     2     3     4     5     6     7
     0, KESC,  '1',  '2',  '3',  '4',  '5',  '6',
   '7',  '8',  '9',  '0',  '-',  '=',  KBS, KTAB,   //0f
   'q',  'w',  'e',  'r',  't',  'y',  'u',  'i',
   'o',  'p',  '[',  ']', KRET, KCTL,  'a',  's',   //1f
   'd',  'f',  'g',  'h',  'j',  'k',  'l',  ';',
  '\'',  '`', KSFT, '\\',  'z',  'x',  'c',  'v',   //2f
   'b',  'n',  'm',  ',',  '.',  '/', KSFT,  '*',
  KALT,  ' ', KCAP,  KF1,  KF2,  KF3,  KF4,  KF5,   //3f
   KF6,  KF7,  KF8,  KF9, KF10, KNUM, KSCR, KHOM,
    KU, KPGU,  '-',   KL,    0,   KR,  '+', KEND,   //4f
    KD, KPGD, KINS, KDEL, KSRQ,    0,    0, KF11,
  KF12,    0,    0,    0,    0,    0,    0,    0,   //5f
     0,    0,    0,    0,    0,    0,    0,    0,
     0,    0,    0,    0,    0,    0,    0,    0,   //6f
     0,    0,    0,    0,    0,    0,    0,    0,
     0,    0,    0,    0,    0,    0,    0,    0,   //7f

//  0     1     2     3     4     5     6     7
     0, KESC,  '!',  '@',  '#',  '$',  '%',  '^',
   '&',  '*',  '(',  ')',  '_',  '+',  KBS, KTAB,   //8f
   'Q',  'W',  'E',  'R',  'T',  'Y',  'U',  'I',
   'O',  'P',  '{',  '}', KRET, KCTL,  'A',  'S',   //9f
   'D',  'F',  'G',  'H',  'J',  'K',  'L',  ':',
   '"',  '~', KSFT,  '|',  'Z',  'X',  'C',  'V',   //af
   'B',  'N',  'M',  '<',  '>',  '?', KSFT,  '*',
  KALT,  ' ', KCAP,  KF1,  KF2,  KF3,  KF4,  KF5,   //bf
   KF6,  KF7,  KF8,  KF9, KF10, KNUM, KSCR, KHOM,
    KU, KPGU,  '-',   KL,    0,   KR,  '+', KEND,   //cf
    KD, KPGD, KINS, KDEL, KSRQ,    0,    0, KF11,
  KF12,    0,    0,    0,    0,    0,    0,    0,   //df
     0,    0,    0,    0,    0,    0,    0,    0,
     0,    0,    0,    0,    0,    0,    0,    0,   //ef
     0,    0,    0,    0,    0,    0,    0,    0,
     0,    0,    0,    0,    0,    0,    0,    0,   //ff
};

static byte kcodemap[512];
static Buffer kcode_buffer;
static byte kcodebuf[128];
static int leds, state;

Inline void
iodelay(void)
{
  int n = 10;
  while(n--) inb(0x80);
}

static int
read_kbc(void)
{
  int n = 999;
  while( n-- )
    {
      if( inb(KBC_CTL) & KBC_OBF )
	return inb(KBC_DAT);
      iodelay();
    }
  return -1;
}

static int
write_kbc( byte data )
{
  int n = 999;
  inb(KBC_DAT);
  while( n-- )
    {
      if( (inb(KBC_CTL) & KBC_BUSY) == 0 )
	{
	  outb(KBC_DAT, data);
	  return 0;
	}
      iodelay();
    }
  return -1;
}

static int
cmd_kbc( byte cmd )
{
  int n = 999;
  while( n-- )
    {
      if( (inb(KBC_CTL) & KBC_IBF) == 0 )
	{
	  outb(KBC_CMD, cmd);
	  return 0;
	}
      iodelay();
    }
  return -1;
}

INTERRUPT( scan_kbc );
Static void
scan_kbc(void)
{
  int code, kcode;
  //disable_irq(IRQ_KBC);

  code = read_kbc();
  if( code == 0xe0 )
    state = 0x80;
  ef( code == 0xe1 )
    state = 0x100;
  else
    {
      kcode = kcodemap[(code & 0x7f) | state] | (code & 0x80);
      buffer_putc( &kcode_buffer, kcode );
      state = 0;
    }

  next_irq(IRQ_KBC);
}

void
set_keyboard_leds( int mod )
{
  leds = mod & 0x07;
  write_kbc( KBC_LED_WRITE );
  write_kbc( leds );
}

int
get_keyboard_leds(void)
{
  return leds;
}

void
send_keyboard_reboot(void)
{
  cmd_kbc( KBC_REBOOT );
}

Eint
get_keyboard_kcode(void)
{
  int kcode;
  kcode = buffer_getc( &kcode_buffer );
  return kcode;
}

int
get_keyboard_ksym( int *mod )
{
  int ksym, kcode;
  byte shift;

  kcode = get_keyboard_kcode();
  if( kcode < 0 )
    return 0;

  shift = (*mod & MSFT) ? 0x80 : 0;
  ksym = ksym_table[(kcode & 0x7f) | shift];

  if( *mod & MCAP )
    {
      if( 'a' <= ksym && ksym <= 'z' )
	ksym = ksym - 'a' + 'A';
      ef( 'A' <= ksym && ksym <= 'Z' )
	ksym = ksym - 'A' + 'a';
    }

  if( kcode & 0x80 )
    {
      /* release */
      switch(ksym)
	{
	case KSCR: break;
	case KNUM: break;
	case KCAP: break;
	case KCTL: *mod &= ~MCTL; break;
	case KSFT: *mod &= ~MSFT; break;
	case KALT: *mod &= ~MALT; break;
	default: return -ksym;
	}
    }
  else
    {
      /* make */
      switch(ksym)
	{
	case KSCR: *mod ^= MSCR; set_keyboard_leds(*mod); break;
	case KNUM: *mod ^= MNUM; set_keyboard_leds(*mod); break;
	case KCAP: *mod ^= MCAP; set_keyboard_leds(*mod); break;
	case KCTL: *mod |= MCTL; break;
	case KSFT: *mod |= MSFT; break;
	case KALT: *mod |= MALT; break;
	default: return ksym;
	}
    }
  return 0;
}

void
setup_keyboard(void)
{
  int i;

  /* default kcodes map */
  memset( kcodemap, 0, sizeof(kcodemap) );
  for( i = 0 ; i < 0x80 ; i++ )
    kcodemap[i] = i;

  state = 0;
  buffer_init( &kcode_buffer, kcodebuf, sizeof(kcodebuf) );
  define_intr( IRQ_TO_INTN(IRQ_KBC), INTR_ENTRY(scan_kbc) );
  enable_irq( IRQ_KBC );
}
