/*
** memory.c --- 
*/

#include <coron.h>
#include <bootinfo.h>
#include <string.h>
#include <splist.h>
#include <memory.h>

#define SLAB_SIZE  (PAGE_SIZE - sizeof(pointer)*2)
#define SLAB_FREE_PTR(s)  ((s)+sizeof(pointer))

#define pte_entry(p)		((p).page*PAGE_SIZE)
#define pgd_index(vpage)	((vpage)>>10)
#define pte_index(vpage)	((vpage)&0x3ff)

/* free pages list */
static pointer freepage_ptr = NULL;
static int freepage_count = 0;
static int memory_pages;

/* system page directory */
PTE *system_pgd;

static void
push_freepages( uint page, uint count )
{
  uint i;
  for( i = 0 ; i < count ; i++ )
    push_splist( &freepage_ptr, (pointer)((page + i)*PAGE_SIZE) );
  freepage_count += count;
}

void *
alloc_page(void)
{
  pointer p = pop_splist( &freepage_ptr );
  if( p != NULL )
    memset( p, 0, PAGE_SIZE );
  CHECK_ABORT( p, "alloc_page miss" );
  freepage_count--;
  return p;
}

void
free_page( void *p )
{
  push_splist( &freepage_ptr, p );
  freepage_count++;
}

void *
map_vpage( PTE *pgd, uint vpage, uint page, bool writable, bool user )
{
  PTE *pte;
  uint d, t;

  d = pgd_index(vpage);
  t = pte_index(vpage);
  
  if( pgd[d].map )
    {
      /* page table ϴˤ */
      pte = (PTE*)pte_entry(pgd[d]);
    }
  else
    {
      /* Υȥˤ page table ƤƤʤΤǡ˺ */
      pte = alloc_page();
      CHECK_ABORT( pte, "cannot allocates new page for PTE" );

      pgd[d].page = (uint)pte / PAGE_SIZE;
      pgd[d].map  = 1;
      pgd[d].p    = 1;
      pgd[d].rw   = 1;
      pgd[d].us   = 1;
    }

  /* page table ˥ڡƤ */
  CHECK_ABORT( pte[t].map == 0,
	       "already mapped vpage 0x%x to page 0x%x", vpage, pte[t].page );

  pte[t].page = page;
  pte[t].map  = 1;
  pte[t].p    = 1;
  pte[t].rw   = writable;
  pte[t].us   = user;
  flush_page_cache();
  return (void*)pte_entry(pte[t]);
}

void *
unmap_vpage( PTE *pgd, uint vpage )
{
  PTE *pte;
  uint d, t;

  d = pgd_index(vpage);
  t = pte_index(vpage);

  CHECK_ABORT( pgd[d].map, "vpage (0x%x) not avail on pgd", vpage );
  pte = (PTE*)pte_entry(pgd[d]);

  CHECK_ABORT( pte[t].map, "vpage (0x%x) not avail on pte", vpage );
  pte[t].map = 0;
  pte[t].p = 0;
  flush_page_cache();
  return (void*)pte_entry(pte[t]);
}

void *
alloc_vpage( PTE *pgd, uint vpage, bool writable, bool user )
{
  void *p;
  p = alloc_page();
  if( p == NULL )
    return NULL;
  return map_vpage( pgd, vpage, (uint)p/PAGE_SIZE, writable, user );
}

void
free_vpage( PTE *pgd, uint vpage )
{
  free_page( unmap_vpage(pgd, vpage) );
}

void
set_vpage_p( PTE *pgd, uint vpage, bool p )
{
  PTE *pte;
  uint d, t;

  d = pgd_index(vpage);
  t = pte_index(vpage);

  pte = (PTE*)pte_entry(pgd[d]);
  pte[t].p = p;
  flush_page_cache();
}

bool
get_vpage_p( PTE *pgd, uint vpage )
{
  PTE *pte;
  uint d, t;

  d = pgd_index(vpage);
  t = pte_index(vpage);

  pte = (PTE*)pte_entry(pgd[d]);
  return pte[t].p;
}

void
map_vmem( PTE *pgd, uint page, uint count, bool writable, bool user )
{
  uint i;
  for( i = 0 ; i < count ; i++ )
    map_vpage( pgd, page + i, page + i, writable, user );
}

void
unmap_vmem( PTE *pgd, uint page, uint count )
{
  uint i;
  for( i = 0 ; i < count ; i++ )
    unmap_vpage( pgd, page + i );
}

void
setup_memory(void)
{
  Bootinfo *info = (void*)MEM_BOOTINFO;
  int page, count, extmem_pages, kernel_pages;

  /* setup conventional memory zone */
  page = MEM_ZONE / PAGE_SIZE;
  count = (MEM_VIDEO - MEM_ZONE) / PAGE_SIZE;
  push_freepages( page, count );

  /* setup extended memory zone */
  extmem_pages = info->extmem_size / PAGE_SIZE;
  kernel_pages = info->kernel_size / PAGE_SIZE + 1;
  page = MEM_KERNEL/PAGE_SIZE + kernel_pages;
  count = extmem_pages - kernel_pages;
  push_freepages( page, count );
  memory_pages = 256 + extmem_pages;

  /* setup virtual memory */
  system_pgd = alloc_page();
  CHECK_ABORT( system_pgd, "allocating system PGD" );
  map_vmem( system_pgd, 0, 256 + extmem_pages, 1, 0 );

  /* begin the paging mode */
  set_cr3(system_pgd);
  paging_on();
}

int
get_freepage_count(void)
{
  return freepage_count;
}

int
get_memory_pages(void)
{
  return memory_pages;
}

void
create_cache( Cache *cache, int objsize )
{
  CHECK_ABORT( objsize < SLAB_SIZE,
	       "objsize must be small than SLAB_SIZE (%d bytes)", SLAB_SIZE );

  if( objsize < sizeof(pointer) )
    objsize = sizeof(pointer);

  cache->objsize = objsize;
  cache->objcount = SLAB_SIZE / objsize;
  cache->slab_ptr = NULL;
}

void *
alloc_cache( Cache *cache )
{
  pointer s, free_ptr;
  int i;

  foreach_splist( &(cache->slab_ptr), s )
    {
      free_ptr = SLAB_FREE_PTR(s);
      if( next_splist(free_ptr) )
	return pop_splist(free_ptr);
    }

  s = alloc_page();
  if(s == NULL)
    return NULL;

  push_splist( &(cache->slab_ptr), s );
  free_ptr = SLAB_FREE_PTR(s);
  for( i = 0 ; i < cache->objcount ; i++ )
    push_splist( free_ptr, free_ptr + i*cache->objsize );

  return pop_splist(free_ptr);
}

void
free_cache( Cache *cache, void *obj )
{
  pointer slab = (pointer)((uint)obj & (-PAGE_SIZE));
  pointer s, free_ptr;

  foreach_splist( &(cache->slab_ptr), s )
    {
      if( slab == s )
	break;
    }
  if( s == NULL )
    return;

  free_ptr = SLAB_FREE_PTR(s);
  push_splist( free_ptr, obj );
}
