/*
** setup.c --- kernel loader
*/

/* startup code */
asm(
".text				\n"
".code16gcc			\n"
"	cli			\n"
"	mov	%cs, %ax	\n"
"	mov	%ax, %ds	\n"
"	mov	%ax, %es	\n"
"	mov	%ax, %ss	\n"
"	mov	$0x3f00, %sp	\n" // mov $(MEM_ZONE-0x100), %sp
"	sti			\n"
"	call	setup		\n"
"dead:	jmp	dead		\n"
);

#include <coron.h>
#include <bootinfo.h>

/* including source code indirect */
#include "desc.c"

/* boot information */
static Bootinfo *info = (void*)MEM_BOOTINFO;
static Bootinfo *ipl_info = (void*)MEM_IPLINFO;	// inner IPL bootinfo

void *
memcpy( void *dest, const void *src, uint n )
{
  char *d = (char*)dest;
  const char *s = (const char*)src;

  while( n-- )
    *d++ = *s++;

  return dest;
}

void *
memset( void *s, int c, uint n )
{
  char *p = (char*)s;

  while( n-- )
    *p++ = (char)c;

  return s;
}

void
puts( char *msg )
{
  char c;
  while((c = *msg++))
    Asm("int $0x10":: "a"(0x0e00|c), "b"(7));
}

void
setup_gdt(void)
{
  sdesc_t *gdt = (void*)MEM_GDT;
  descptr_t gdtr;

  /* clear GDT */
  memset( gdt, 0, sizeof(sdesc_t)*GDTNUM );

  // 0x00: empty
  set_sdesc( &gdt[0], 0, 0, 0, 0 );
  // 0x08: reserved
  // 0x10: kernel code segment
  set_sdesc( &gdt[2], 0, 0xfffff, DESC_CODE | DESC_READABLE, 0 );
  // 0x18: kernel data segment
  set_sdesc( &gdt[3], 0, 0xfffff, DESC_DATA | DESC_WRITABLE, 0 );
  // 0x20: user code segment
  set_sdesc( &gdt[4], 0, 0xfffff, DESC_CODE | DESC_READABLE, 3 );
  // 0x28: user data segment
  set_sdesc( &gdt[5], 0, 0xfffff, DESC_DATA | DESC_WRITABLE, 3 );
  // 0x30: current TSS descriptor
  // 0x38: old TSS descriptor

  /* load GDTR */
  gdtr.limit = sizeof(sdesc_t)*GDTNUM -1;
  gdtr.base = MEM_GDT;
  Asm("lgdt %0"::"m"(gdtr));
}

void
setup_idt(void)
{
  gdesc_t *idt = (void*)MEM_IDT;
  descptr_t idtr;

  /* clear IDT */
  memset( idt, 0, sizeof(gdesc_t)*IDTNUM );

  /* load IDTR */
  idtr.limit = sizeof(gdesc_t)*IDTNUM -1;
  idtr.base = MEM_IDT;
  Asm("lidt %0"::"m"(idtr));
}

/* get extended memory size in Kilobytes (64MB limits) */
Inline word16
get_memsize(void)
{
  word16 size;
  Asm("int $0x15": "=a"(size): "a"(0x8800));
  return size;
}

/* extended memory block copy (16MB limits) */
Inline int
embcpy( uint dest, uint src, uint count )
{
  sdesc_t dt[6];
  int i, ret;

  for( i = 0 ; i < sizeof(dt) ; i++ )
    ((byte*)dt)[i] = 0;

  set_sdesc( &dt[2], src, 0xffff, DESC_DATA | DESC_WRITABLE, 0 );
  set_sdesc( &dt[3], dest, 0xffff, DESC_DATA | DESC_WRITABLE, 0 );
  Asm("int $0x15":"=a"(ret):"a"(0x8700),"c"(count),"S"(dt));
  return (ret>>8);
}

Inline int
reset_drive( uint drive )
{
  int ret;
  Asm("int $0x13": "=a"(ret): "a"(0), "d"(drive));
  return ((ret >> 8) & 0xff);
}

Inline int
read_drive_track( uint drive, void* buf, uint track )
{
  int ret;
  Asm("int $0x13": "=a"(ret): 
      "a"(0x200|SECPTRK), "b"(buf),
      "c"( ((track/2)<<8) | 1 ),
      "d"( ((track%2)<<8) | drive ));
  return ((ret>>8) & 0xff);
}

int
load_kernel( uint drive, uint head, uint count )
{
  static byte trackbuf[TRKSIZE];
  uint first, f, last, l, tracks, t, skip, seek, cnt, i;

  if( count == 0 )
    return 0;

  first  = head / SECPTRK;
  f      = head % SECPTRK;
  last   = (head + count-1) / SECPTRK + 1;
  l      = (head + count-1) % SECPTRK + 1;
  tracks = last - first;

  skip = seek = cnt = 0;
  for( t = 0 ; t < tracks ; t++ )
    {
      if( read_drive_track(drive, trackbuf, first + t) )
	return -1;
      if( tracks == 1 )
	{
	  /* only one track */
	  skip = 0;
	  seek = f;
	  cnt  = l - f;
	}
      ef( t == 0 )
	{
	  /* first */
	  skip = 0;
	  seek = f;
	  cnt  = SECPTRK - f;
	}
      ef( t == tracks - 1 )
	{
	  /* last */
	  skip += cnt;
	  seek = 0;
	  cnt  = l;
	}
      else
	{
	  /* middle */
	  skip += cnt;
	  seek = 0;
	  cnt  = SECPTRK;
	}

      for( i = 0 ; i < cnt ; i++ )
	puts(".");

      embcpy( MEM_KERNEL + skip * SECSIZE,
	      (uint)trackbuf + seek * SECSIZE, cnt * SECSIZE );
    }
  return 0;
}

Inline void
stop_floppy_motor(void)
{
  outb( 0x3f2, 0x0c );
  inb( 0x80 );
}

Inline void
enable_a20(void)
{
  while(inb(0x64) & 2);
  outb(0x64, 0xd1);
  while(inb(0x64) & 2);
  outb(0x60, 0xdf);
  while(inb(0x64) & 2);
  outb(0x64, 0xff);
  while(inb(0x64) & 2);
}

void
setup(void)
{
  puts("setup:");

  /* get boot information from IPL */
  memcpy( info, ipl_info, sizeof(Bootinfo) );
  
  /* load kernel */
  if( reset_drive(info->boot_drive) )
    goto load_error;
  if( load_kernel(info->boot_drive, info->kernel_head, info->kernel_count ) )
    goto load_error;
  puts("ok\r\n");

  /* get extended memory size */
  info->extmem_size = get_memsize() * 1024;

  /* enable protect mode */
  stop_floppy_motor();
  enable_a20();
  disable_nmi();
  lock_cpu();
  setup_gdt();
  setup_idt();

  /* turn on PE(Protection Enable) bit */
  Asm("	mov	%cr0, %eax	\n"
      "	or	$1, %eax	\n"
      "	mov	%eax, %cr0	\n"
      /* flush out the pipeline */
      "	jmp	1f		\n"
      "1:			\n");

  /* activate kernel */
  Asm("ljmpl %0, %1"::"i"(GDT_KERNEL_CODE),"i"(MEM_KERNEL));

 load_error:
  puts("error");
  return;
}
