/*
** task.c --- task scheduler
*/

#include <coron.h>
#include <clock.h>
#include <desc.h>
#include <memory.h>
#include <list.h>
#include <string.h>
#include <task.h>

static byte esp0_stack[PAGE_SIZE];
static Cache task_cache;
static Task ready_queue;
static uint ready_count;
Task *running;
bool enable_dispatch;

void
setup_task(void)
{
  sdesc_t *gdt = (sdesc_t*)MEM_GDT;
  Task *idle = &ready_queue;

  /* setup idle task */
  memset( idle, 0, sizeof(Task) );
  init_list( idle );
  idle->state = TASK_IDLE;

  /* setup idle task TSS */
  idle->tss.cs = GDT_KERNEL_CODE;
  idle->tss.ds = GDT_KERNEL_DATA;
  idle->tss.es = GDT_KERNEL_DATA;
  idle->tss.fs = GDT_KERNEL_DATA;
  idle->tss.gs = GDT_KERNEL_DATA;
  idle->tss.ss = GDT_KERNEL_DATA;
  idle->tss.cr3 = (uint)system_pgd;

  /* setup TSS descriptor for the idle task */
  set_tss32_sdesc(&(idle->tss_desc),(uint)&(idle->tss),sizeof(idle->tss)-1,0);
  memcpy( &gdt[6], &(idle->tss_desc), sizeof(sdesc_t) );
  Asm("ltr %%ax"::"a"(GDT_CUR_TSS));

  /* initialize task management variables */
  create_cache( &task_cache, sizeof(Task) );
  enable_dispatch = FALSE;
  ready_count = 0;
  running = idle;
}

Task *
create_task( Taskinfo *info )
{
  Task *task;

  /* allocates new Task */
  task = alloc_cache( &task_cache );
  if( task == NULL )
    return NULL;

  /* setup new task */
  memset( task, 0, sizeof(Task) );
  init_list( task );
  memcpy( &(task->info), info, sizeof(Taskinfo) );
  task->state = TASK_READY;

  /* setup TSS */
  if( task->info.user )
    {
      /* user type TSS */
      task->tss.cs = GDT_USER_CODE | 3;
      task->tss.ds = GDT_USER_DATA | 3;
      task->tss.es = GDT_USER_DATA | 3;
      task->tss.fs = GDT_USER_DATA | 3;
      task->tss.gs = GDT_USER_DATA | 3;
      task->tss.ss = GDT_USER_DATA | 3;
      task->tss.eip = task->info.code_head;
      task->tss.esp = task->info.stack_tail;      
      task->tss.cr3 = task->info.pgd;
      task->tss.ss0 = GDT_KERNEL_DATA;
      task->tss.esp0 = (uint)esp0_stack + PAGE_SIZE;
      task->tss.iobase = IOBASE;
      memset( &(task->tss.ioenable), 0xff, IOENABLE_SIZE );
    }
  else
    {
      /* system type TSS */
      task->tss.cs = GDT_KERNEL_CODE;
      task->tss.ds = GDT_KERNEL_DATA;
      task->tss.es = GDT_KERNEL_DATA;
      task->tss.fs = GDT_KERNEL_DATA;
      task->tss.gs = GDT_KERNEL_DATA;
      task->tss.ss = GDT_KERNEL_DATA;
      task->tss.eip = task->info.code_head;
      task->tss.esp = task->info.stack_tail;
      task->tss.cr3 = task->info.pgd;
    }
      
  /* setup TSS descriptor */
  set_tss32_sdesc(&(task->tss_desc),(uint)&(task->tss),sizeof(task->tss)-1,0);

  /* append new task to the last of ready queue */
  BEGIN_CPULOCK;
  add_list_tail( running, task );
  ready_count++;
  END_CPULOCK;

  return task;
}

void
destroy_task( Task *task )
{
  BEGIN_CPULOCK;

  del_list( task );
  free_cache( &task_cache, task );

  END_CPULOCK;
}

void
wait_task( Task *task, void (*wait_poll)(Task*, void*), void *wait_info )
{
  BEGIN_CPULOCK;

  task->state = TASK_WAIT;
  task->wait_poll = wait_poll;
  task->wait_info = wait_info;
  ready_count--;
  
  END_CPULOCK;
}

void
weakup_task( Task *task )
{
  BEGIN_CPULOCK;

  task->state = TASK_READY;
  ready_count++;

  END_CPULOCK;
}

void
exit_task( Task *task )
{
  BEGIN_CPULOCK;

  if( task->state != TASK_EXIT )
    {
      task->state = TASK_EXIT;
      ready_count--;
    }

  END_CPULOCK;
}

Err
brk_task( Task *task, uint size )
{
  BEGIN_CPULOCK

  uint i, pages;
  void *p;

  pages = size / PAGE_SIZE + 1;
  if( pages > task->brk_pages )
    {
      for( i = task->brk_pages ; i < pages ; i++ )
	{
	  p = alloc_vpage( (PTE*)task->info.pgd, task->brk_start_page + i,
			   1, task->info.user );
	  CHECK_ABORT( p, "allocating vpage=0x%x", i );
	}
    }
  ef( pages < task->brk_pages )
    {
      for( i = pages ; i < task->brk_pages ; i++ )
	free_vpage( (PTE*)task->info.pgd, task->brk_start_page + i );
    }
  task->brk_pages = pages;

  END_CPULOCK
  return 0;
}

/*
 * ؤ(ǥѥå)Ԥ
 * ߶ػ߾֤ǸƤӽФɬפ롣
 */
void
dispatch(void)
{
  sdesc_t *gdt = (sdesc_t*)MEM_GDT;
  Task *ready = running;

  /* dispatching loop */
  enable_dispatch = FALSE;
  for(;;)
    {
      unlock_cpu();
      Asm("nop");  // ǥѥåȯߤ򡢤ǽ
      lock_cpu();

      ready = ready->next;
      switch( ready->state )
	{
	case TASK_READY:
	  goto do_dispatch;

	case TASK_WAIT:
	  /* Ԥ */
	  ready->wait_poll( ready, ready->wait_info );
	  break;
	  
	case TASK_EXIT:
	  /* νλ */
	  ready = ready->prev;
	  destroy_task( ready->next );
	  break;

	case TASK_IDLE:
	  /*
	   * ¹Բǽʥ̵С
	   * dispatch ǽ˸ƤӽФؿ start() 
	   * ɥ󥰤Ԥ
	   */
	  if( ready_count == 0 )
	    goto do_dispatch;
	  else
	    break;

	default:
	  ABORT("unexpected state 0x%x, in 0x%x (id:%s)", ready->state, ready, ready->info.id);
	}
    }

 do_dispatch:
  enable_dispatch = TRUE;

  if( ready != running )
    {
      /* ¹楿TSSǥץư */
      memcpy( &gdt[7], &gdt[6], sizeof(sdesc_t) );
      gdt[7].type = GATE_TSS32;
      Asm("ltr %%ax"::"a"(GDT_OLD_TSS));

      /* ˼¹Ԥ륿TSSǥץѰդ */
      memcpy( &gdt[6], &(ready->tss_desc), sizeof(sdesc_t) );

      /* Υפǥڤؤ */
      running = ready;
      Asm("ljmpl %0, $0"::"i"(GDT_CUR_TSS));
    }
}

static void
sleep_poll( Task *task, void *wait_info )
{
  uint until = (uint)wait_info;
  
  //FIXME: ޤȤΤȤθƤʤ
  if( until <= get_clock() )
    weakup_task( task );
}

void
sleep( uint seconds )
{
  BEGIN_CPULOCK

  wait_task( running, sleep_poll, (void*)get_clock() + seconds * 1000 );
  dispatch();

  END_CPULOCK
}

void
usleep( uint usec )
{
  BEGIN_CPULOCK

  wait_task( running, sleep_poll, (void*)get_clock() + usec / 1000 );
  dispatch();

  END_CPULOCK
}

void
exit(void)
{
  lock_cpu();
  exit_task( running );
  dispatch();
}

Err
brk( uint size )
{
  return brk_task( running, size );
}
