/*
** vga.c --- Video Graphics Array driver
*/

#include <coron.h>
#include <string.h>
#include <vga.h>
 
/* VGA registers */
enum {
  SEQ_ADDR	= 0x3c4, // Sequencer Address
  SEQ_DATA	= 0x3c5, //     ..    Data
  CRT_ADDR	= 0x3d4, // CRT Controller Address
  CRT_DATA	= 0x3d5, //     ..         Data
  GRAP_ADDR	= 0x3ce, // Graphics  Address
  GRAP_DATA	= 0x3cf, //     ..    Data
  ATTR_ADDR	= 0x3c0, // Attribute Address
  ATTR_READ	= 0x3c1, //     ..    Read data
  ATTR_WRITE	= 0x3c0, //     ..    Write data
  DAC_MASK	= 0x3c6, // DAC PEL Mask
  DAC_READ_ADDR	= 0x3c7, //  .. Read Address
  DAC_WRITE_ADDR= 0x3c8, //  .. Write Address
  DAC_DATA	= 0x3c9, //  .. Data
  MISC_READ	= 0x3cc, // Miscellaneous Output Read
  MISC_WRITE	= 0x3c2, //     ..               Write
  INPUT_STAT0	= 0x3c2, // Input Status #0 Read
  INPUT_STAT1	= 0x3da, //     ..       #1 Read
  FEATURE_READ	= 0x3ca, // Feature Control Read
  FEATURE_WRITE	= 0x3da, //     ..          Write
  SUBSYS_DATA	= 0x3c3, // Video Subsystem Enable Data
};

/* VGA informations */
enum {
  VGA_ADDR	= 0xa0000,
  VGA_SIZE	= 0x10000, // 64KB
  FONT_ADDR	= 0xa0000,
  FONT_SIZE	= 0x08000, // 32KB
  TEXT_ADDR	= 0xb8000,
  TEXT_SIZE	= 0x08000, // 32KB
};

/* VGA registers */
typedef struct {
  byte seq[5];
  byte crt[25];
  byte grap[9];
  byte attr[21];
  byte misc[1];
} VGAREGS;

/*
** Sequencer
*/

/* reset and halt sequencer */
Inline void
reset_seq(void)
{
  outb( SEQ_ADDR, 0 );
  outb( SEQ_DATA, 1 );
}

Inline void
start_seq(void)
{
  outb( SEQ_ADDR, 0 );
  outb( SEQ_DATA, 3 );
}

void
enable_screen( bool ena )
{
  outb( SEQ_ADDR, 1 );
  if( ena )
    outb( SEQ_DATA, inb(SEQ_DATA) & ~BV(5) );
  else
    outb( SEQ_DATA, inb(SEQ_DATA) | BV(5) );
}

Inline void
set_map_mask( byte mask )
{
  outb( SEQ_ADDR, 2 );
  outb( SEQ_DATA, mask );
}

Inline void
select_char_map( byte a, byte b )
{
  byte o;
  outb( SEQ_ADDR, 3 );
  o  = (a & 3) << 2;
  o |= (a & 4) << 3;
  o |= (b & 3);
  o |= (b & 4) << 2;
  outb( SEQ_DATA, o );
}

/* set character width */
Inline void
set_char_width( bool eight )
{
  int clock_mode;
  outb( SEQ_ADDR, 1 );
  clock_mode = inb(SEQ_DATA) & ~BV(0);
  clock_mode |= eight ? BV(0) : 0;
  outb( SEQ_DATA, clock_mode );
}

static void
read_seq( byte seq[5] )
{
  int i;
  for( i = 0 ; i < 5 ; i++ )
    {
      outb( SEQ_ADDR, i );
      seq[i] = inb( SEQ_DATA );
    }
}

static void
write_seq( byte seq[5] )
{
  int i;
  reset_seq();
  for( i = 0 ; i < 5 ; i++ )
    {
      outb( SEQ_ADDR, i );
      outb( SEQ_DATA, seq[i] );
    }
  start_seq();
}

/*
** CRT Controller
*/

Inline void
disable_crt_protect(void)
{
  outb( CRT_ADDR, 17 );
  outb( CRT_DATA, inb(CRT_DATA) & ~BV(8) );
}

Inline void
enable_crt_protect(void)
{
  outb( CRT_ADDR, 17 );
  outb( CRT_DATA, inb(CRT_DATA) | BV(8) );
}

Inline void
set_char_height( int height )
{
  outb( CRT_ADDR, 9 );
  outb( CRT_DATA, (inb(CRT_DATA) & 0xe0) | ((height - 1) & 0x1f) );
}

void
enable_cursor( bool ena )
{
  byte d;
  outb( CRT_ADDR, 10 );
  d = inb(CRT_DATA);
  if(ena)
    d &= ~BV(5);
  else
    d |= BV(5);
  outb( CRT_DATA, d );
}

void
set_cursor_size( int start, int end )
{
  outb( CRT_ADDR, 10 );
  outb( CRT_DATA, (inb(CRT_DATA) & 0xe0) | (start & 0x1f) );
  outb( CRT_ADDR, 11 );
  outb( CRT_DATA, (inb(CRT_DATA) & 0xe0) | (end & 0x1f) );
}

void
set_cursor_addr( int addr )
{
  BEGIN_CPULOCK

  outb( CRT_ADDR, 14 );
  outb( CRT_DATA, (addr >> 8) & 0xff );
  outb( CRT_ADDR, 15 );
  outb( CRT_DATA, addr & 0xff );

  END_CPULOCK
}

void
set_start_addr( int addr )
{
  outb( CRT_ADDR, 12 );
  outb( CRT_DATA, (addr >> 8) & 0xff );
  outb( CRT_ADDR, 13 );
  outb( CRT_DATA, addr & 0xff );
}

void
set_crt_timings( int hdisp, int hsyncstart, int hsyncend, int htotal,
		int vdisp, int vsyncstart, int vsyncend, int vtotal,
		bool hpole_n, bool vpole_n )
{
  int start_hblank, end_hblank;
  int start_hretrace, end_hretrace;
  int start_vblank, end_vblank;
  int start_vretrace, end_vretrace;
  int total, disp, ov, vbs9, misc;

  disable_crt_protect();

  total = htotal - 5;
  disp = hdisp - 1;
  start_hblank = hdisp;
  start_hretrace = hsyncstart;
  end_hretrace = hsyncend;
  end_hblank = htotal - 1;

  // Horizontal Total
  outb( CRT_ADDR, 0 );
  outb( CRT_DATA, total );
  // Horizontal Display Enable End
  outb( CRT_ADDR, 1 );
  outb( CRT_DATA, disp );
  // Start Horizontal Blanking
  outb( CRT_ADDR, 2 );
  outb( CRT_DATA, start_hblank );
  // End Horizontal Blanking
  outb( CRT_ADDR, 3 );
  outb( CRT_DATA, (end_hblank & 0x1f) | BV(7) );
  // Start Horizontal Retrace Pulse
  outb( CRT_ADDR, 4 );
  outb( CRT_DATA, start_hretrace );
  // End Horizontal Retrace 
  outb( CRT_ADDR, 5 );
  outb( CRT_DATA, (end_hretrace & 0x1f) | ((end_hblank & BV(5)) << 2) );
  // Offset
  outb( CRT_ADDR, 19 );
  outb( CRT_DATA, hdisp / 2 );

  total = vtotal - 2;
  disp = vdisp - 1;
  start_vblank = vdisp;
  start_vretrace = vsyncstart;
  end_vretrace = vsyncend;
  end_vblank = vtotal - 1;
  
  // Read Overflow Register
  outb( CRT_ADDR, 7 );
  ov = inb(CRT_DATA) & BV(4);
  // Vertical Total
  outb( CRT_ADDR, 6 );
  outb( CRT_DATA, total & 0xff );
  ov |= (total & BV(9)) ? BV(5) : 0;
  ov |= (total & BV(8)) ? BV(0) : 0;
  // Vertical Display Enable End
  outb( CRT_ADDR, 18 );
  outb( CRT_DATA, disp & 0xff );
  ov |= (disp & BV(9)) ? BV(6) : 0;
  ov |= (disp & BV(8)) ? BV(1) : 0;
  // Start Vertical Blanking
  outb( CRT_ADDR, 21 );
  outb( CRT_DATA, start_vblank & 0xff );
  vbs9 = (start_vblank & BV(9)) ? BV(5) : 0;
  ov |= (start_vblank & BV(8)) ? BV(3) : 0;
  // End Vertical Blanking
  outb( CRT_ADDR, 22 );
  outb( CRT_DATA, end_vblank & 0xff );
  // Vertical Retrace Start
  outb( CRT_ADDR, 16 );
  outb( CRT_DATA, start_vretrace & 0xff );
  ov |= (start_vretrace & BV(9)) ? BV(7) : 0;
  ov |= (start_vretrace & BV(8)) ? BV(2) : 0;
  // Vertical Retrace End
  outb( CRT_ADDR, 17 );
  outb( CRT_DATA, (inb(CRT_DATA) & 0xf0) | (end_vretrace & 0x0f) );
  // Write Maximum Scan Line Register
  outb( CRT_ADDR, 9 );
  outb( CRT_DATA, (inb(CRT_DATA) & ~BV(5)) | vbs9 );
  // Write Overflow Register
  outb( CRT_ADDR, 7 );
  outb( CRT_DATA, ov );
  // Miscellaneous Output Register
  misc = inb( MISC_READ ) & 0x3f;
  misc |= vpole_n ? BV(7) : 0;
  misc |= hpole_n ? BV(6) : 0;
  outb( MISC_WRITE, misc );

  enable_crt_protect();
}

static void
read_crt( byte crt[25] )
{
  int i;
  for( i = 0 ; i < 25 ; i++ )
    {
      outb( CRT_ADDR, i );
      crt[i] = inb( CRT_DATA );
    }
}

static void
write_crt( byte crt[25] )
{
  int i;
  disable_crt_protect();
  for( i = 0 ; i < 25 ; i++ )
    {
      outb( CRT_ADDR, i );
      outb( CRT_DATA, crt[i] );
    }
  enable_crt_protect();
}

/*
** Graphics Controller
*/

Inline void
set_write_operation( int opr )
{
  outb( GRAP_ADDR, 3 );
  outb( GRAP_DATA, opr << 3 );
}

Inline void
set_rotate_count( int count )
{
  outb( GRAP_ADDR, 3 );
  outb( GRAP_DATA, count );
}

Inline void
set_read_map( int select )
{
  outb( GRAP_ADDR, 4 );
  outb( GRAP_DATA, select );
}

Inline void
set_graphics_mode( int mode )
{
  outb( GRAP_ADDR, 5 );
  outb( GRAP_DATA, mode );
}

Inline void
set_bit_mask( int mask )
{
  outb( GRAP_ADDR, 8 );
  outb( GRAP_DATA, mask );
}

Inline void
enable_reset( int map )
{
  outb( GRAP_ADDR, 1 );
  outb( GRAP_DATA, map );
}

Inline void
set_reset( int map )
{
  outb( GRAP_ADDR, 0 );
  outb( GRAP_DATA, map );
}

Inline void
enable_graphics( bool enable )
{
  int am, gm;

  inb( INPUT_STAT1 ); // reset addr/write selection flip-flop
  outb( ATTR_ADDR, 16 );
  am = inb( ATTR_READ ) & ~BV(0);
  am |= enable ? BV(0) : 0;
  outb( ATTR_WRITE, am );

  outb( GRAP_ADDR, 6 );
  gm = inb( GRAP_DATA ) & ~BV(0);
  gm |= enable ? BV(0) : 0;
  outb( GRAP_DATA, gm );
}

static void
read_grap( byte grap[9] )
{
  int i;
  for( i = 0 ; i < 9 ; i++ )
    {
      outb( GRAP_ADDR, i );
      grap[i] = inb( GRAP_DATA );
    }
}

static void
write_grap( byte grap[9] )
{
  int i;
  for( i = 0 ; i < 9 ; i++ )
    {
      outb( GRAP_ADDR, i );
      outb( GRAP_DATA, grap[i] );
    }
}

/*
 * Attribute Controller
 */

Inline void
set_palette_source( bool external )
{
  inb( INPUT_STAT1 ); // reset addr/write selection flip-flop
  if( external )
    outb( ATTR_ADDR, 0x20 );
  else
    outb( ATTR_ADDR, 0x00 );
}

static void
read_attr( byte attr[21] )
{
  int i;
  for( i = 0 ; i < 21 ; i++ )
    {
      inb( INPUT_STAT1 ); // reset addr/write selection flip-flop
      outb( ATTR_ADDR, i );
      attr[i] = inb( ATTR_READ );
    }
  set_palette_source(1);
}

static void
write_attr( byte attr[21] )
{
  int i;
  inb( INPUT_STAT1 ); // reset addr/write selection flip-flop
  for( i = 0 ; i < 21 ; i++ )
    {
      outb( ATTR_ADDR, i );
      outb( ATTR_WRITE, attr[i] );
    }
  set_palette_source(1);
}

/*
 * Miscellaneous
 */

static void
read_misc( byte misc[1] )
{
  misc[0] = inb( MISC_READ );
}

static void
write_misc( byte misc[1] )
{
  reset_seq();
  outb( MISC_WRITE, misc[0] );
  start_seq();
}

/*
** Digital-to-Analog Converter
*/

void
read_dac( word32 dac[256] )
{
  int i;
  outb( DAC_READ_ADDR, 0 );
  for( i = 0 ; i < 256 ; i++ )
    {
      dac[i]  = inb(DAC_DATA) << 2;
      dac[i] |= inb(DAC_DATA) << 10;
      dac[i] |= inb(DAC_DATA) << 18;
    }
}

void
write_dac( word32 dac[256] )
{
  int i;
  outb( DAC_WRITE_ADDR, 0 );
  for( i = 0 ; i < 256 ; i++ )
    {
      outb( DAC_DATA, (dac[i] >> 2) & 0xff );
      outb( DAC_DATA, (dac[i] >> 10) & 0xff );
      outb( DAC_DATA, (dac[i] >> 18) & 0xff );
    }
}

/*
** Read/Write All Registers
*/

void
read_vgaregs( VGAREGS *regs )
{
  BEGIN_CPULOCK

  read_attr( regs->attr );
  read_seq( regs->seq );
  read_grap( regs->grap );
  read_crt( regs->crt );
  read_misc( regs->misc );

  END_CPULOCK
}

void
write_vgaregs( VGAREGS *regs )
{
  BEGIN_CPULOCK

  write_attr( regs->attr );
  write_misc( regs->misc );
  write_seq( regs->seq );
  write_grap( regs->grap );
  write_crt( regs->crt );

  END_CPULOCK
}

#include <debug.h>
void
dump_vgaregs( VGAREGS *regs )
{
  printk("\nseq[5]");
  dump( regs->seq, 5 );
  printk("\ncrt[25]");
  dump( regs->crt, 25 );
  printk("\ngrap[9]");
  dump( regs->grap, 9 );
  printk("\nattr[21]");
  dump( regs->attr, 21 );
  printk("\nmisc[1]");
  dump( regs->misc, 1 );
}

/*
** VGA Device Interface
*/

static VGAREGS vgaregs = {
  seq: { 0x03, 0x00, 0x03, 0x00, 0x02 },
  crt: { 0x5f, 0x4f, 0x50, 0x82, 0x55, 0x81, 0xbf, 0x1f,
	 0x00, 0x4f, 0x0d, 0x0e, 0x00, 0x00, 0x00, 0x50,
	 0x9c, 0x8e, 0x8f, 0x28, 0x1f, 0x96, 0xb9, 0xa3,
	 0xff },
  grap: { 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x0e, 0x00, 0xff },
  attr: { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	  0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	  0x0c, 0x00, 0x0f, 0x08, 0x00 },
  misc: { 0x67 },
};

static word32 pallets[256] = {
  0x000000, 0xff6b65, 0x63ff6b, 0xffc465,
  0x6565ff, 0xf465ff, 0x60a5ff, 0xa8a8a8,
  0x545454, 0xce9a94, 0x92dd99, 0xd3c494,
  0x9191e2, 0xb196cc, 0x92e5df, 0xfcfcfc,
};

void
setup_vga(void)
{
  write_vgaregs( &vgaregs );
  write_dac( pallets );
  set_cursor_size( 13, 14 );
  enable_cursor( TRUE );
}
