/*
** vgafb.c --- VGA frame buffer device
**
** Bibliography:
**  - FreeVGA Project
**     http://web.inter.nl.net/hcc/S.Weijgers/FreeVGA/home.htm
**  - Online Book Initiative VGA Standards
**     http://ftp.std.com/obi/Standards/VGA/
*/

#include <coron.h>
#include <string.h>
#include <vgafb.h>

/* VGA Registers */
enum {
  VGA_ATTAW = 0x3c0,	// Attribute Address and Write data
  VGA_ATTRD = 0x3c1,	//    ..     Read data
  VGA_SEQA  = 0x3c4,	// Sequencer Address
  VGA_SEQD  = 0x3c5,	//    ..     Data
  VGA_GRAA  = 0x3ce,	// Graphics  Address
  VGA_GRAD  = 0x3cf,	//    ..     Data
  VGA_CRTA  = 0x3d4,	// CRT Controller Address
  VGA_CRTD  = 0x3d5,	//    ..          Data
  VGA_MISCR = 0x3cc,	// External Read
  VGA_MISCW = 0x3c2,	//    ..    Write
  VGA_DACRA = 0x3c7,	// DAC Read Address
  VGA_DACWA = 0x3c8,	//  .. Write Address
  VGA_DACD  = 0x3c9,	//  .. Data
  VGA_STAT  = 0x3da,	// Input status #1 read
};

/* VGA Address */
enum {
  VGA_ADDR	= 0xa0000,
  VGA_SIZE	= 0x10000, // 64KB
  TEXT_ADDR	= 0xb8000,
  TEXT_SIZE	= 0x00fa0, // 4000 Bytes
  FONT_ADDR	= 0xa0000,
  FONT_SIZE	= 0x04000, // 16KB
};

/* VGA register data */
typedef struct {
  byte att[21];
  byte seq[5];
  byte gra[9];
  byte crt[25];
  byte misc;
} VGAREG;

/* VGA text-mode data */
typedef struct {
  VGAREG reg;
  Pallet pals[16];
  byte	 data[TEXT_SIZE];
  byte	 font[FONT_SIZE];
} VGADATA;
static VGADATA text_mode;

/* register setting for 640x480 16colors graphics mode */
static VGAREG vgareg = {
  att: { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
	 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
	 0x01, 0x00, 0x0f, 0x00, 0x00 },
  seq: { 0x00, 0x01, 0x0f, 0x00, 0x06 },
  gra: { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x05, 0x0f, 0xff },
  crt: { 0x5f, 0x4f, 0x50, 0x82, 0x54, 0x80, 0x0b, 0x3e,
	 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	 0xea, 0x8c, 0xdf, 0x28, 0x00, 0xe7, 0x04, 0xe3, 0xff },
  misc:  0xe3,
};

/* 16colors pallet (pastel coron) */
static Pallet default_pals[VGAFB_PALNUM] = {
  { 0x00, 0x00, 0x00, 0x00 },
  { 0x65, 0x6b, 0xff, 0x00 },
  { 0x6b, 0xff, 0x63, 0x00 },
  { 0x65, 0xc4, 0xff, 0x00 },
  { 0xff, 0x65, 0x65, 0x00 },
  { 0xff, 0x65, 0xf4, 0x00 },
  { 0xff, 0xa5, 0x60, 0x00 },
  { 0xa8, 0xa8, 0xa8, 0x00 },
  { 0x54, 0x54, 0x54, 0x00 },
  { 0x94, 0x9a, 0xce, 0x00 },
  { 0x99, 0xdd, 0x92, 0x00 },
  { 0x94, 0xc4, 0xd3, 0x00 },
  { 0xe2, 0x91, 0x91, 0x00 },
  { 0xcc, 0x96, 0xb1, 0x00 },
  { 0xdf, 0xe5, 0x92, 0x00 },
  { 0xff, 0xff, 0xff, 0x00 },
};

static uint fb_plane, fb_pos;

/* pallet address source select ( 1: normal, 0: internal ) */
Inline void
vga_set_pas( uint b )
{
  inb( VGA_STAT );
  if( b )
    outb( VGA_ATTAW, 0x20 );
  else
    outb( VGA_ATTAW, 0 );
}

/* set read map */
Inline void
vga_set_rmap( uint plane )
{
  outw( VGA_GRAA, 4 | (plane << 8) );
}

/* set write mode */
Inline void
vga_set_mode( uint mode )
{
  outw( VGA_GRAA, 5 | (mode << 8) );
}

/* set write enable maps */
Inline void
vga_set_wena( uint plane_bmp )
{
  outw( VGA_SEQA, 2 | (plane_bmp << 8) );
}

/* set write map */
Inline void
vga_set_wmap( uint plane )
{
  outw( VGA_SEQA, 2 | (1 << (plane + 8)) );
}

/* set write operation */
Inline void
vga_set_wopr( uint opr )
{
#ifndef VGA_BOCHS_HACK	// VRAM$B=q$-9~$_;~$N(B and/or/xor $B1i;;$r;HMQ$G$-$J$$(B
  outw( VGA_GRAA, 3 | (opr << 11) );
#endif
}

/* set rotate count */
Inline void
vga_set_wrot( uint count )
{
  outw( VGA_GRAA, 3 | (count << 8) );
}

/* set write mask */
Inline void
vga_set_wmask( uint mask )
{
  outw( VGA_GRAA, 8 | (mask << 8) );
}

/* enable set/reset */
Inline void
vga_ena_reset( uint plane_bmp )
{
  outw( VGA_GRAA, 1 | (plane_bmp << 8) );
}

/* set set/reset */
Inline void
vga_set_reset( uint plane_bmp )
{
  outw( VGA_GRAA, 0 | (plane_bmp << 8) );
}

static void
vga_read_reg( VGAREG *reg )
{
  int i;

  lock_cpu();
  vga_set_pas( 0 );

  /* attribute register */
  for( i = 0 ; i < 21 ; i++ )
    {
      inb( VGA_STAT );
      outb( VGA_ATTAW, i );
      reg->att[i] = inb( VGA_ATTRD );
    }

  /* sequencer register */
  for( i = 0 ; i < 5 ; i++ )
    {
      outb( VGA_SEQA, i );
      reg->seq[i] = inb( VGA_SEQD );
    }

  /* graphics register */
  for( i = 0 ; i < 9 ; i++ )
    {
      outb( VGA_GRAA, i );
      reg->gra[i] = inb( VGA_GRAD );
    }

  /* CRT controller */
  for( i = 0 ; i < 25 ; i++ )
    {
      outb( VGA_CRTA, i );
      reg->crt[i] = inb( VGA_CRTD );
    }

  /* miscellaneous register */
  reg->misc = inb( VGA_MISCR );

  inb( VGA_STAT );
  unlock_cpu();
}

static void
vga_write_reg( VGAREG *reg )
{
  int i;

  lock_cpu();

  /* attribute register */
  inb( VGA_STAT );
  for( i = 0 ; i < 21 ; i++ )
    {
      outb( VGA_ATTAW, i );
      outb( VGA_ATTAW, reg->att[i] );
    }
  vga_set_pas( 1 );

  /* miscellaneous register */
  outb( VGA_MISCW, reg->misc );

  /* synchronous reset on */
  outw( VGA_SEQA, 0x0100 );

  /* sequencer register */
  for( i = 1 ; i < 5 ; i++ )
    {
      outb( VGA_SEQA, i );
      outb( VGA_SEQD, reg->seq[i] );
    }

  /* synchronous reset off */
  outw( VGA_SEQA, 0x0300 );

  /* graphics register */
  for( i = 0 ; i < 9 ; i++ )
    {
      outb( VGA_GRAA, i );
      outb( VGA_GRAD, reg->gra[i] );
    }

  /* disable CRTC protect */
  outw( VGA_CRTA, 0x0011 );

  /* CRT controller */
  for( i = 0 ; i < 25 ; i++ )
    {
      outb( VGA_CRTA, i );
      outb( VGA_CRTD, reg->crt[i] );
    }

  vga_set_pas( 1 );
  unlock_cpu();
}

void
vga_set_pallets( Pallet *pals, uint first, uint count )
{
  uint i;
  outb( VGA_DACWA, first );
  for( i = 0 ; i < count ; i++ )
    {
      outb( VGA_DACD, pals[i].r >> 2 );
      outb( VGA_DACD, pals[i].g >> 2 );
      outb( VGA_DACD, pals[i].b >> 2 );
    }
}

void
vga_get_pallets( Pallet *pals, uint first, uint count )
{
  uint i;
  outb( VGA_DACRA, first );
  for( i = 0 ; i < count ; i++ )
    {
      pals[i].r = inb( VGA_DACD ) << 2;
      pals[i].g = inb( VGA_DACD ) << 2;
      pals[i].b = inb( VGA_DACD ) << 2;
    }
}

void
vga_clear(void)
{
  vga_set_wena( 0xf );
  vga_ena_reset( 0xf );
  vga_set_reset( 0 );
  memset( (void*)VGA_ADDR, 0, 640/8 * VGAFB_WIDTH );
  vga_ena_reset( 0 );
}

Err
setup_vgafb( void *arg )
{
  /* backup text buffer */
  vga_read_reg( &text_mode.reg );
  memcpy( text_mode.data, (void*)TEXT_ADDR, TEXT_SIZE );
  vga_get_pallets( text_mode.pals, 0, VGAFB_PALNUM );
  
  /* shift to graphics mode */
  vga_set_pallets( default_pals, 0, VGAFB_PALNUM );
  vga_write_reg( &vgareg );
  
  /* read font map */
  vga_set_rmap( 2 );
  memcpy( text_mode.font, (void*)FONT_ADDR, FONT_SIZE );
  
  /* setup default setting */
  vga_set_mode( 0 );
  vga_set_wmask( 0xff );
  vga_set_wopr( 0 );
  vga_set_wrot( 0 );
  vga_ena_reset( 0 );

  /* setup variables */
  fb_plane = 0xf;
  fb_pos = 0;

  /* clear all pixels */
  vga_clear();
  return 0;
}

Err
release_vgafb(void)
{
  /* clear all */
  vga_clear();
  vga_set_pas( 0 );
  vga_ena_reset( 0 );
  
  /* restore font map */
  vga_set_wmap( 2 );
  memcpy( (void*)FONT_ADDR, text_mode.font, FONT_SIZE );
  
  /* restore text mode */
  vga_set_pallets( text_mode.pals, 0, VGAFB_PALNUM );
  vga_write_reg( &text_mode.reg );
  vga_set_pas( 1 );
  
  /* restore text data */
  memcpy( (void*)TEXT_ADDR, text_mode.data, TEXT_SIZE );
  return 0;
}

Err
open_vgafb( int id )
{
  return 0;
}

Err
close_vgafb( int id )
{
  return 0;
}

Eint
read_vgafb( int id, void *buf, uint count )
{
  int map;

  if( fb_pos + count > VGA_SIZE )
    return -1;

  if( fb_plane & 1 )
    map = 0;
  ef( fb_plane & 2 )
    map = 1;
  ef( fb_plane & 4 )
    map = 2;
  ef( fb_plane & 8 )
    map = 3;
  else
    return -1;
  
  vga_set_rmap(map);
  memcpy( buf, (void*)VGA_ADDR + fb_pos, count );
  fb_pos += count;
  return count;
}

Eint
write_vgafb( int id, void *buf, uint count )
{
  if( fb_pos + count > VGA_SIZE )
    return -1;
  
  vga_set_wena( fb_plane );
  memcpy( (void*)VGA_ADDR + fb_pos, buf, count );
  fb_pos += count;
  return count;
}

Eint
lseek_vgafb( int id, int offset, int whence )
{
  uint pos = fb_pos;

  switch(whence)
    {
    case SEEK_SET:
      pos = offset;
      break;
    case SEEK_CUR:
      pos += offset;
      break;
    case SEEK_END:
      pos = VGA_SIZE + offset;
      break;
    default:
      return -1;
    }

  if( pos > VGA_SIZE )
    return -1;

  fb_pos = pos;
  return pos;
}

Eint
set_vgafb_plane( uint plane )
{
  fb_plane = plane;
  return 0;
}

Eint
get_vgafb_plane(void)
{
  return fb_plane;
}

Eint
get_vgafb_width(void)
{
  return VGAFB_WIDTH;
}

Eint
get_vgafb_height(void)
{
  return VGAFB_HEIGHT;
}

Eint
get_vgafb_depth(void)
{
  return VGAFB_DEPTH;
}

Eint
set_vgafb_oldfont( void *buf )
{
  memcpy( text_mode.font, buf, FONT_SIZE );
  return 0;
}

Eint
get_vgafb_oldfont( void *buf )
{
  memcpy( buf, text_mode.font, FONT_SIZE );
  return 0;
}

Eint
set_vgafb_oldtext( void *buf )
{
  memcpy( text_mode.data, buf, TEXT_SIZE );
  return 0;
}

Eint
get_vgafb_oldtext( void *buf )
{
  memcpy( buf, text_mode.data, TEXT_SIZE );
  return 0;
}

Eint
get_vgafb_palnum(void)
{
  return VGAFB_PALNUM;
}

Eint
set_vgafb_pallets( Pallet *pals )
{
  vga_set_pallets( pals, 0, VGAFB_PALNUM );
  return 0;
}

Eint
get_vgafb_pallets( Pallet *pals )
{
  vga_get_pallets( pals, 0, VGAFB_PALNUM );
  return 0;
}

Eint
ioctl_vgafb( int id, int request, void *arg )
{
  switch(request)
    {
    case SET_PLANE: return set_vgafb_plane( (uint)arg );
    case GET_PLANE: return get_vgafb_plane();
    case GET_WIDTH: return get_vgafb_width();
    case GET_HEIGHT: return get_vgafb_height();
    case GET_DEPTH: return get_vgafb_depth();
    case SET_OLDFONT: return set_vgafb_oldfont(arg);
    case GET_OLDFONT: return get_vgafb_oldfont(arg);
    case SET_OLDTEXT: return set_vgafb_oldtext(arg);
    case GET_OLDTEXT: return get_vgafb_oldtext(arg);
    case GET_PALNUM: return get_vgafb_palnum();
    case SET_PALLETS: return set_vgafb_pallets( (Pallet*)arg );
    case GET_PALLETS: return get_vgafb_pallets( (Pallet*)arg );
    }
  return -ENOSPT;
}

Device vgafb_device = {
  setup:   setup_vgafb,
  release: release_vgafb,
  open:    open_vgafb,
  close:   close_vgafb,
  read:    read_vgafb,
  write:   write_vgafb,
  lseek:   lseek_vgafb,
  ioctl:   ioctl_vgafb,
};
