/*
 * Copyright 2004-2005 The Trix Development Team.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.trix.cuery.property;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.trix.cuery.util.CSSUtil;
import org.trix.cuery.value.CSSValue;

import org.w3c.css.sac.LexicalUnit;

/**
 * DOCUMENT.
 * 
 * @author <a href="mailto:Teletha.T@gmail.com">Teletha Testarossa</a>
 * @version $ Id: ShorthandPropertyDefinition.java,v 1.01 2005/09/01 12:23:41 Teletha Exp $
 */
public class ShorthandPropertyDefinition extends PropertyDefinition {

    /** The property definition list. */
    protected final List definitions = new ArrayList(4);

    /**
     * @see org.trix.cuery.property.PropertyDefinition#addAcceptable(java.lang.Object)
     */
    protected void addAcceptable(Object acceptable) {
        if (acceptable instanceof PropertyDefinition) {
            definitions.add(acceptable);
        } else {
            super.addAcceptable(acceptable);
        }
    }

    /**
     * @see org.trix.cuery.property.PropertyDefinition#parse(org.trix.cuery.value.CSSValue)
     */
    public Map parse(CSSValue value) {
        // check inherit
        if (value.getLength() == 1 && value.getLexicalUnitType() == LexicalUnit.SAC_INHERIT) {
            // inherit identifier is not acceptable
            if (!acceptInherit()) {
                return null;
            }

            Map map = new HashMap(definitions.size());

            for (int i = 0; i < definitions.size(); i++) {
                PropertyDefinition definition = (PropertyDefinition) definitions.get(i);
                Map result = definition.parse(value);

                // check result
                if (result == null) {
                    continue;
                }

                map.putAll(result);
                map.put(definition.getName(), value);
            }
            return map;
        }

        Map map = new HashMap(definitions.size());
        CSSValue current = value;

        boolean[] flags = new boolean[definitions.size()];
        Arrays.fill(flags, false);

        while (current != null) {
            // inherit identifier is forbidden in the shorthand property
            int type = current.getLexicalUnitType();

            if (type == LexicalUnit.SAC_INHERIT) {
                return null;
            }

            // skip slash and comma
            if (type == LexicalUnit.SAC_OPERATOR_SLASH) {
                current = current.getNextValue();
                continue;
            }

            if (type == LexicalUnit.SAC_OPERATOR_COMMA) {
                break;
            }

            for (int i = 0; i < definitions.size(); i++) {
                // check used definition
                if (flags[i]) {
                    // not found a suitable definition
                    if (i == definitions.size() - 1) {
                        return null;
                    }
                    continue;
                }

                Map result = null;
                PropertyDefinition definition = (PropertyDefinition) definitions.get(i);

                if (definition instanceof ShorthandPropertyDefinition) {
                    result = definition.parse(CSSUtil.singleValue(current));
                } else {
                    result = definition.parse(current);
                }

                // this definition is unsuitable
                if (result == null) {
                    // not found a suitable definition
                    if (i == definitions.size() - 1) {
                        return null;
                    }
                    continue;
                }

                // store
                map.putAll(result);
                map.put(definition.getName(), current);
                flags[i] = true;

                // next
                current = current.getNextValue();
                break;
            }
        }
        return validate(map);
    }

    /**
     * This method is executed after the method 'parse'. The purpose of this method are validating
     * value and filling ommitted properties.
     * 
     * @param result A parsing result.
     * @return A validating and filling result.
     */
    protected Map validate(Map result) {
        return result;
    }

    /**
     * Helper method to check this definition accepts the inherit identifier.
     * 
     * @return A result.
     */
    private boolean acceptInherit() {
        for (int i = 0; i < acceptables.size(); i++) {
            if (acceptables.get(i).equals(CSSValue.INHERIT)) {
                return true;
            }
        }
        return false;
    }

}
