#include <iostream>
#include <fstream>
#include <string>
#include <list>
#include <set>
#include <exception>
#include <cstdio>

#include "cuppa/cocuppa.h"

// system does NOT have getopt
#define NO_GETOPT

/*
 * getopt
 */
#ifdef NO_GETOPT
#ifndef __STDC__
#  define __STDC__ 1
#  include "getopt.h" // getopt_long
#  undef __STDC__
#else
#  include "getopt.h" // getopt_long
#endif
#else
#include <getopt.h>
#endif

using namespace std;

bool exist(const string& path) {
  ifstream stream(path.c_str());
  return stream.is_open();
}

void version() {
  cout << "couma $Date: 2002/08/27 08:38:41 $" << endl;
}

void usage() {
  cout <<
"--help,      -h        : display help\n"
"--include,   -i <file> : insert #include \"file\"\n"
"--prefix,    -P        : prepend 'test' to methods\n"
"--skeleton,  -s <Name> : create skeleton Name" IMPL_SUFFIX "\n"
"--version,   -v        : print version\n"
;
}

struct method {
  bool   noregist;
  string name;
  bool   outofline;
  method(bool no, const string& na, bool ou) : noregist(no), name(na), outofline(ou) {}
  method() : noregist(false), outofline(false) {}
  explicit method(const string& na) : noregist(false), name(na), outofline(false) {}
  bool operator==(const method& rhs) const { return name == rhs.name; }
  string suite() const;
  string decl() const;
  string impl(const string& suite) const;
};

string method::suite() const {
  return (noregist ? "//" : "  ") + string("CPPUNIT_TEST(") + name + ");";
}

string method::decl() const {
  string result = "  void " + name + "()";
  if ( outofline ) {
    result += ';';
  } else {
    result += " {\n    CPPUNIT_FAIL(\"no implementation\");\n  }";
  }
  return result;
}

string method::impl(const string& suite) const {
  string result;
  if ( outofline ) {
    result += "void " + suite + "::" + name +
              "() {\n  CPPUNIT_FAIL(\"no implementation\");\n}\n";
  }
  return result;
}

struct parse_result {
  list<string> lines;
  set<string>  cur_includes;
  set<string>  cur_methods;

  list<string>::iterator include_pos;
  list<string>::iterator suite_pos;
  list<string>::iterator decl_pos;
  list<string>::iterator impl_pos;

  bool separate;
};

bool extract_marker(const string& str, string& key, string& value) {
  char buf[128];
  strncpy(buf, str.c_str(), 128)[127] = '\0';
  char* m = strtok(buf, "*/: \t");
  if ( !m || strcmp(m,"CUPPA") ) return false;
  char* k = strtok( 0 , "= \t");
  if ( !k ) return false;
  key = k;
  char* v = strtok( 0 , " \t\n");
  value = v ? v : "";
  return true;
}

bool extract_include(const string& str, string& include) {
  char buf[128];
  strncpy(buf, str.c_str(), 128)[127] = '\0';
  char* m = strtok(buf, "# \t");
  if ( !m || strcmp(m,"include") ) return false;
  char* i = strtok( 0 , "\"<>");
  if ( !i ) return false;
  include = i;
  return true;
}

bool extract_suite(const string& str, string& suite) {
  char buf[128];
  strncpy(buf, str.c_str(), 128)[127] = '\0';
  char* m = strtok(buf, "*/( \t");
  if ( !m || strcmp(m,"CPPUNIT_TEST") ) return false;
  char* s = strtok( 0 , ") \t");
  if ( !s ) return false;
  suite = s;
  return true;
}

void parse(istream& strm, parse_result& pr) {
  string       line;

  while ( getline(strm, line) ) {
    pr.lines.push_back(line);
  }

  pr.include_pos = pr.lines.end();
  pr.suite_pos   = pr.lines.end();
  pr.decl_pos    = pr.lines.end();
  pr.impl_pos    = pr.lines.end();

  bool in_include = false;
  bool in_suite   = false;

  for ( list<string>::iterator iter = pr.lines.begin();
        iter != pr.lines.end(); ++iter ) {
    line = *iter;
    string key;
    string value;
    if ( extract_marker(line, key, value) ) {
      bool is_begin = value == "+";
      bool is_end   = value == "-";
      if ( key == "include" ) {
        in_include = is_begin;
        if ( is_end ) {
          pr.include_pos = iter;
        }
      } else if ( key == "suite" ) {
        in_suite = is_begin;
        if ( is_end ) {
          pr.suite_pos = iter;
        }
      } else if ( key == "decl" ) {
        if ( is_end ) {
          pr.decl_pos = iter;
        }
      } else if ( key == "impl" ) {
        if ( is_end ) {
          pr.impl_pos = iter;
        }
      } else if ( key == "separate" ) {
        pr.separate = value == "true";
      }
    } else {
      string token;
      if ( in_include && extract_include(line, token) ) {
        pr.cur_includes.insert(token);
      }
      if ( in_suite && extract_suite(line, token) ) {
        pr.cur_methods.insert(token);
      }
    }
  }
}

void insert(parse_result& pr, list<string>& includes,
            list<method>& methods, const string& suite) {

  if ( pr.include_pos != pr.lines.end() ) {
    for ( list<string>::iterator iter = includes.begin();
          iter != includes.end(); ++iter ) {
      pr.lines.insert(pr.include_pos, "#include \"" + *iter +"\"");
    }
  }

  if ( pr.suite_pos != pr.lines.end() ) {
    for ( list<method>::iterator iter = methods.begin();
          iter != methods.end(); ++iter ) {
      pr.lines.insert(pr.suite_pos, iter->suite());
    }
  }

  if ( pr.decl_pos != pr.lines.end() ) {
    for ( list<method>::iterator iter = methods.begin();
          iter != methods.end(); ++iter ) {
      pr.lines.insert(pr.decl_pos, iter->decl());
    }
  }

  if ( pr.impl_pos != pr.lines.end() ) {
    for ( list<method>::iterator iter = methods.begin();
          iter != methods.end(); ++iter ) {
      if ( iter->outofline ) {
        pr.lines.insert(pr.impl_pos, iter->impl(suite));
      }
    }
  }

}

int comad(int argc, char* argv[]) {

  bool         prefix = false;
  list<string> includes;
  bool         standalone = true;
  string       suite;
  list<method> methods;

  while ( true ) {
    int option_index = 0;
    static struct option long_options[] = {
      { "help",       0, 0, 'h'},
      { "include",    1, 0, 'i'},
      { "prefix",     0, 0, 'P'},
      { "skeleton",   1, 0, 's'},
      { "version",    0, 0, 'v'},
      { 0, 0, 0, 0 }
    };

    int c = getopt_long(argc, argv, "Phi:s:v",
                        long_options, &option_index);
    if ( c == -1 ) break;

    switch ( c ) {
    case 'P' : prefix = true;              break;
    case 'h' : usage();                    return 0;
    case 'i' : includes.push_back(optarg); break;
    case 's' : suite = optarg;             break;
    case 'v' : version();                 return 0;
    }
  }

  if ( !suite.empty() ) {
    while ( optind < argc ) {
      string meth = argv[optind++];
      string::size_type methlen = meth.size();
      string::size_type methpos = 0;
      method m;
      m.outofline = false;
      m.noregist  = false;
      if ( meth[methlen-1] == '.' ) {
        m.outofline = true;
        --methlen;
      }
      if ( meth[0] == '.' ) {
        m.noregist = true;
        --methlen;
        ++methpos;
      }
      m.name = meth.substr(methpos, methlen);
      if ( prefix ) {
        m.name = "test" + m.name;
      }
      methods.push_back(m);
    }
  } else {
    return 1;
  }

  if ( !exist( suite + IMPL_SUFFIX ) ) {
    return 1;
  }

  if ( exist( suite + DECL_SUFFIX ) ) {
    standalone = false;
  }

  string path = suite + (standalone ? IMPL_SUFFIX : DECL_SUFFIX);

  {
    ifstream istrm(path.c_str());
    parse_result pr;
    parse(istrm, pr);
    istrm.close();

    set<string>::iterator siter;
    for ( siter = pr.cur_includes.begin();
          siter != pr.cur_includes.end(); ++siter ) {
      includes.remove(*siter);
    }

    for ( siter = pr.cur_methods.begin();
          siter != pr.cur_methods.end(); ++siter ) {
      methods.remove(method(*siter));
    }

    insert(pr, includes, methods, suite);

    remove((path + '~').c_str());
    rename(path.c_str(), (path + '~').c_str());
    ofstream ostrm(path.c_str());

    for ( list<string>::iterator iter = pr.lines.begin();
          iter != pr.lines.end(); ++iter ) {
      ostrm << *iter << endl;
    }
    ostrm.close();
  }

  if ( standalone ) {
    return 0;
  }

  path = suite + IMPL_SUFFIX;

  {
    ifstream istrm(path.c_str());
    parse_result pr;
    parse(istrm, pr);
    istrm.close();

    insert(pr, includes, methods, suite);

    remove((path + '~').c_str());
    rename(path.c_str(), (path + '~').c_str());
    ofstream ostrm(path.c_str());

    for ( list<string>::iterator iter = pr.lines.begin();
          iter != pr.lines.end(); ++iter ) {
      ostrm << *iter << endl;
    }
    ostrm.close();

  }

  return 0;
}

int main(int argc, char* argv[]) {
  try {
    return comad(argc, argv);
  } catch ( exception& ex ) {
    cerr << ex.what() << endl;
  }
  return 1;
}

