require 'umigame/xunit'
require 'umigame/creator'
require 'umigame/adder'
require 'umigame/sourcecode'
require 'umigame/method'
require 'test/unit/assertions'

module Umigame

  class JUnit < XUnit
    def JUnit.version
      "JUnit 3.7"
    end
    
    def JUnit.usage
      print <<"EOB"

#{JUnit.version}(--lang java)
  support option: -a, -i, -P, -r, -s
EOB
    end
    
    def JUnit.reference
      print <<"EOB"
Assertion Quick Reference
  for #{JUnit.version}

assertEquals(expected, actual)
assertEquals(java.lang.String message, expected, actual)
   (argument type:boolean, byte char, double, float, int, long ,java.lang.Object)

assertNotNull(object)
assertNotNull(java.lang.String message, object)
assertNull(object)
assertNull(java.lang.String message, object)
assertSame(expected, actual)
assertSame(java.lang.String message, expected, actual)
   (argument type:java.lang.Object)

assert(condition)
assert(java.lang.String message, condition)
assertTrue(condition)
assertTrue(java.lang.String message, condition)
   (argument type:boolean)

fail()
fail(java.lang.String message)
EOB
    end
  end
  
  class JUnitMethod < Method
    def initialize(class_name, method_name, statement)
      super(class_name, method_name, "//", statement)
    end
    
    def name
      if @prefix
        "test" + @name.capitalize
      else
        @name
      end
    end
    
    def decl
     ret = []
      ret.push "public void #{name}() {"
      ret.push @statement
      ret.push "}"
      return prefix + ret.join("\n" + prefix) + "\n"
    end
    
    def suite
      return "#{prefix}  suite.addTest(new #{@classname}(\"#{name}\"));\n"
    end
  end
  
  class JUnitSourceCode < SourceCode
    def add_method(methodname, prefix=false, statement=nil)
      if statement
        statement =~ /^(.*)\}/m
        statement = $1
      else
        statement = "  fail(\"not implement yet\");"
      end
      method = JUnitMethod.new(@class_name, methodname, statement)
      super(method, prefix)
    end
    
    def print
      include = ""
      decl = ""
      suite = ""
      @include.each do |file|
        include +="\nimport #{file};"
      end
      @method.each do |method|
        decl += "\n" + method.decl
        suite += method.suite
      end
      if has_suite?
        suite = "  public static Test suite() {\n    TestSuite suite = new TestSuite();\n//CUPPA:suite=+\n#{suite}//CUPPA:suite=-\n    return suite;\n  }\n"
      else
        suite = ""
      end
      class_end = @class_end.join
      return <<"EOB"
import junit.framework.*;
//CUPPA:include=+#{include}
//CUPPA:include=-
#{@class_begin.join}//CUPPA:decl=+#{decl}
//CUPPA:decl=-

#{suite}#{class_end}
EOB
    end
  end
  
  class JUnitAdder < Adder
    def initialize
      super("//", JUnitSourceCode.new)
    end
    
    protected
    
    def parse_include
      do_parse_include(/import (.+);/)
    end
    
    def parse_class_begin
      do_parse_class_begin(/public class (.+) extends .* \{/)
    end
    
    def parse_method
      do_parse_method(/^(..)public void (.+)\(\) \{/)
    end
    
    def parse_suite
      do_parse_suite(/public static Test suite\(\)/, 
        /(\/\/|\s\s)\s*suite.addTest\(new #{@code.class_name}\(\"(.+)\"\)\)/, 0, 2)
    end
  end
  
  class JUnitCreator < Creator
    attr_reader :code
    
    def filename(name)
      name + ".java"
    end
    
    def eval_argv
      assert_equal(false, @parser['noregist'], "not support --noregist option")
      assert_equal(false, @parser['main'], "not support --main option")
      @include.collect! do |file|
        'import #{file};'
      end
    end
    
    def write_skeleton
      code = JUnitSourceCode.new
      code.class_name = @skeleton
      @include.each do |includefile|
        code.add_include includefile
      end
      code.class_begin.push <<"EOB"

public class #{@skeleton} extends #{@superclass ? @superclass : 'TestCase'} {

  public #{@skeleton}(String name) {
    super(name);
  }
  
  //your stuff...
  
  protected void setUp() {
    // initialize
  }
  
  protected void tearDown() {
    // terminate
  }

EOB
      code.has_suite = !@prefix

      @testmethod.each do | testcase |
        code.add_new_method testcase
      end
      
      code.class_end.push <<"EOB"
  public static void main(String args[]) {
    if ( args.length > 0 ) {
      if ( args[0].charAt(0) == 's' ) {
        junit.swingui.TestRunner.run(#{@skeleton}.class);
      } else if ( args[0].charAt(0) == 'a' ) {
        junit.awtui.TestRunner.run(#{@skeleton}.class);
      } else {
        junit.textui.TestRunner.run(#{@skeleton}.class);
      }
    } else {
      junit.textui.TestRunner.run(#{@skeleton}.class);
    }
  }
}
EOB
      return code.print
    end
  end
end
