/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security;

import java.util.Iterator;
import java.util.HashMap;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.om.security.Group;
import org.apache.jetspeed.om.security.JetspeedGroupFactory;

/**
 * Unit test for GroupManagement interface
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestGroupManagement.java,v 1.1 2004/04/07 22:02:43 jford Exp $
 */

public class TestGroupManagement extends JetspeedTestCase {    
    
    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestGroupManagement( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestGroupManagement.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Group Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestGroupManagement.class );
    }

    /**
     * Tests getGroups method
     * @throws Exception
     */

    public void testGetGroups() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;
        HashMap map = new HashMap();

        try
        {
            Iterator groups = service.getGroups();
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
            }
            assertTrue(map.get("apache") != null);
            assertTrue(map.get("Jetspeed") != null);
            assertTrue(map.get("bogusGroup") == null);
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroups Test OK ");

    }

    /**
     * Tests getGroups method
     * @throws Exception
     */

    public void testGetGroupsForUser() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;
        HashMap map = new HashMap();

        try
        {
            Iterator groups = service.getGroups("turbine");
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
                System.out.println("[turbine] group = " + group.getName());
            }
            assertTrue(map.get("Jetspeed") != null);
            assertTrue(map.get("apache") == null);

            map.clear();
            groups = service.getGroups("admin");            
            while (groups.hasNext())
            {
                group = (Group)groups.next();
                map.put(group.getName(), group);
                System.out.println("[admin] group = " + group.getName());
            }
            assertTrue(map.get("Jetspeed") != null);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroups Test OK ");

    }

    /**
     * Tests addGroup method 
     * @throws Exception
     */

    public void testAddGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            group = JetspeedGroupFactory.getInstance();
            group.setName("bogus");
            service.addGroup(group);
            System.out.println("new group id = " + group.getId());
            assertTrue(group.getId() != null);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            group = JetspeedGroupFactory.getInstance();
            group.setName("bogus");
            service.addGroup(group);
            fail("Should've thrown a dup key exception on group");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed addGroup Test OK ");

    }

    /**
     * Tests getRemoveGroup method 
     * @throws Exception
     */

    public void testRemoveGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.removeGroup("bogus");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.removeGroup("catchmeifyoucan");
            fail("Should've thrown a not found exception on group");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);
        }

        System.out.println("Completed addGroup Test OK ");

    }

    /**
     * Tests getGroup method
     * @throws Exception
     */

    public void testGetGroup() throws Exception 
    {
        GroupManagement service = getService();

        try
        {
            Group group = service.getGroup("Jetspeed");
            System.out.println("*** group nm = " + group.getName());
            System.out.println("*** group id = " + group.getId());
            assertTrue(group.getName().equals("Jetspeed"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed getGroup Test OK ");

    }

    /**
     * Tests saveGroup method 
     * @throws Exception
     */

    public void testSaveGroup() throws Exception 
    {
        GroupManagement service = getService();

        try
        {
            Group group = service.getGroup("apache");
            service.saveGroup(group);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed saveGroup Test OK ");

    }

    /**
     * Tests joinGroup method 
     * @throws Exception
     */
    public void testJoinGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.joinGroup("turbine", "apache");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.joinGroup("baduser", "apache");
            fail("Should've thrown a bad user exception on join");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed joinGroup Test OK ");

    }

    /**
     * Tests unjoinGroup method 
     * @throws Exception
     */
    public void testUnjoinGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            service.unjoinGroup("turbine", "apache");
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            service.unjoinGroup("baduser", "apache");
            fail("Should've thrown a bad user exception on unjoin");
        }
        catch(Exception e)
        {
            assertTrue(e instanceof GroupException);           
        }

        System.out.println("Completed unjoinGroup Test OK ");

    }

    /**
     * Tests inGroup method 
     * @throws Exception
     */
    public void testInGroup() throws Exception 
    {
        GroupManagement service = getService();
        Group group = null;

        try
        {
            boolean in = service.inGroup("admin", "Jetspeed");
            assertTrue(true == in);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
        try
        {
            boolean in = service.inGroup("turbine", "apache");
            assertTrue(false == in);
        }
        catch(Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed inGroup Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static GroupManagement getService()
    {
        return (GroupManagement)TurbineServices
                .getInstance()
                .getService(GroupManagement.SERVICE_NAME);
    }

}






