/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and
 *     "Apache Jetspeed" must not be used to endorse or promote products
 *    derived from this software without prior written permission. For
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache" or
 *    "Apache Jetspeed", nor may "Apache" appear in their name, without
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

package org.apache.jetspeed.portlet;

import org.apache.jetspeed.portlet.event.WindowListener;

/**
 ** The <CODE>PortletWindow</CODE> represents the window that encloses
 ** a portlet. The portlet window can send events on manipulation of its
 ** various window controls, like the "minimize" or "close" buttons.
 ** But a portlet also interrogate the portlet window about its current
 ** visibility state. For example, a portlet may render its content
 ** differently depending on whether its window is maximized or not.
 **
 ** @author   <A HREF="mailto:tboehme@us.ibm.com">Thomas F. Boehme</A>
 **/

public interface PortletWindow
{
    /**
     ** The <CODE>State</CODE> class is a finite enumeration of
     ** the possible states that a window can assume.
     **/

    public static class State implements java.io.Serializable
    {
        /**
         ** The standard "one-of many" window state on a page.
         **/

        public final static State NORMAL        = new State ("Normal", 0);

        /**
         ** In this window state the portlet is displayed maximized
         ** which means that it is the only portlet shown on the page.
         **/

        public final static State MAXIMIZED     = new State ("Maximized", 1);

        /**
         ** In this window state the portlet is displayed minimzed
         ** which means that only the portlet title is shown
         **/

        public final static State MINIMIZED     = new State ("Minimized", 2);

        /**
         ** Allows the portlet window to be detached of the normal
         ** content view of the portal and thus be shown in a separate
         ** window.
         **/

        public final static State DETACHED      = new State ("Detached", 3);

        /**
         ** Allows the portlet window to be moved
         **/

        public final static State MOVING        = new State ("Moving", 4);

        /**
         ** Allows the portlet window to be resized
         **/

        public final static State RESIZING      = new State ("Resizing", 5);

        /**
         ** The window is or will be closed and thus is not shown on the
         ** portal page anymore.
         **/

        public final static State CLOSED        = new State ("Closed", 6);

        private String identifier;

        private int value;

        private State (String identifier, int value)
        {
            this.identifier = identifier;
            this.value = value;
        }

        public int getId()
        {
            return value;
        }

        public String toString ()
        {
            return (identifier);
        }
    }

    /**
     ** Returns the title of this window. This method returns
     ** either a dynamic title or a static title
     ** from the portlet configuration.
     **
     ** @return  the portlet title
     **/

    public String getTitle ();

    /**
     ** Returns whether this portlet window is currently detached.
     **
     ** @return   <CODE>true</CODE> if this window is detached, <BR>
     **           <CODE>false</CODE> otherwise
     **/

    public boolean isDetached ();

    /**
     ** Returns whether this portlet window is currently maximized.
     **
     ** @return   <CODE>true</CODE> if this window is maximized, <BR>
     **           <CODE>false</CODE> otherwise
     **/

    public boolean isMaximized ();

    /**
     ** Returns whether this portlet window is currently minimized.
     **
     ** @return   <CODE>true</CODE> if this window is minimized, <BR>
     **           <CODE>false</CODE> otherwise
     **/

    public boolean isMinimized ();

}
