/*
 * Copyright 2000-2001,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.jetspeed.services.security;

// Junit imports
import junit.framework.Test;
import junit.framework.TestSuite;

import org.apache.turbine.services.TurbineServices;
import org.apache.turbine.util.TurbineConfig;
import org.apache.turbine.util.StringUtils;

// Jetspeed imports
import org.apache.jetspeed.test.JetspeedTestCase;
import org.apache.jetspeed.services.JetspeedSecurity;
import org.apache.jetspeed.om.security.Role;
import org.apache.jetspeed.om.security.Permission;
import org.apache.jetspeed.om.security.JetspeedUser;

/**
 * Unit test for SecurityCacheService
 * 
 * @author <a href="mailto:david@bluesunrise.com">David Sean Taylor</a>
 * @version $Id: TestSecurityCache.java,v 1.1 2004/04/07 22:02:43 jford Exp $
 */

public class TestSecurityCache extends JetspeedTestCase {    

    /**
     * Defines the testcase name for JUnit.
     *
     * @param name the testcase's name.
     */
    public TestSecurityCache( String name ) {
        super( name );
    }
    
    /**
     * Start the tests.
     *
     * @param args the arguments. Not used
     */
    public static void main(String args[]) 
    {
        junit.awtui.TestRunner.main( new String[] { TestSecurityCache.class.getName() } );
    }
 
    public void setup() 
    {
        //System.out.println("Setup: Testing Turbine Role Management");         
    }

    /**
     * Creates the test suite.
     *
     * @return a test suite (<code>TestSuite</code>) that includes all methods
     *         starting with "test"
     */
    public static Test suite() 
    {
        // All methods starting with "test" will be executed in the test suite.
        return new TestSuite( TestSecurityCache.class );
    }

    /**
     * Tests getRoles method
     * @throws Exception
     */

    public void testLoadCache() throws Exception 
    {
        SecurityCacheService service = getService();

        try
        {
            JetspeedUser user = JetspeedSecurity.getUser("turbine");
            service.load(user.getUserName());
            Role role = service.getRole(user.getUserName(), "user");
            assertTrue(role.getName().equals("user"));
            assertTrue(service.hasRole(user.getUserName(), "user"));
            assertTrue(service.hasPermission("user", "view"));
            assertTrue(service.hasPermission("user", "customize"));
            assertTrue(service.hasPermission("user", "maximize"));
            assertTrue(!service.hasPermission("user", "failure"));
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed loadCache Test OK ");

    }


    public void testAddRemoveFromCache() throws Exception 
    {
        SecurityCacheService service = getService();

        try
        {
            JetspeedUser user = JetspeedSecurity.getUser("anon");
            service.load(user.getUserName());

            Role role1 = service.getRole(user.getUserName(), "guest");
            assertTrue(role1.getName().equals("guest"));
            assertTrue(service.hasPermission("guest", "view"));

            // add role
            Role role2 = JetspeedSecurity.getRole("user");
            service.addRole(user.getUserName(), role2);
            assertTrue(service.hasRole(user.getUserName(), "user"));
            assertTrue(service.getRole(user.getUserName(),"user").getName().equals("user"));

            // remove role
            service.removeRole(user.getUserName(), "user");
            assertTrue(!service.hasRole(user.getUserName(), "user"));
            Role role3 = service.getRole(user.getUserName(),"user");
            assertTrue(null == role3);

            // add permission
            Permission perm1 = JetspeedSecurity.getPermission("info");
            assertTrue(null != perm1);
            service.addPermission("guest", perm1);
            Permission permission = service.getPermission("guest", "info");
            assertTrue(permission.getName().equals("info"));
            assertTrue(service.hasPermission("guest", "info"));

            // remove permission
            service.removePermission("guest", "info");
            assertTrue(!service.hasPermission( "guest", "info"));
            Permission perm2 = service.getPermission( "guest", "info");
            assertTrue(null == perm2);

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed addRemoveFromCache Test OK ");

    }

    public void testRemoveAll() throws Exception 
    {
        SecurityCacheService service = getService();
        try
        {
            Role role = JetspeedSecurity.getRole("admin");
            JetspeedUser anon = JetspeedSecurity.getUser("anon");
            service.load(anon.getUserName());
            JetspeedUser turbine = JetspeedSecurity.getUser("turbine");
            service.load(turbine.getUserName());

            service.addRole(anon.getUserName(), role);
            service.addRole(turbine.getUserName(), role);

            assertTrue(service.hasRole(anon.getUserName(), role.getName()));
            assertTrue(service.hasRole(turbine.getUserName(), role.getName()));

            service.removeAllRoles("admin");

            assertTrue(!service.hasRole(anon.getUserName(), role.getName()));
            assertTrue(!service.hasRole(turbine.getUserName(), role.getName()));

        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }

        System.out.println("Completed removeAll Test OK ");

    }

  /*
    Configuration object to run Turbine outside a servlet container
    ( uses turbine.properties )
    */
    private static TurbineConfig config = null;
    
    /**
    Sets up TurbineConfig using the system property:
    <pre>turbine.properties</pre>
    */
    static
    {
        try
        {
            config = new TurbineConfig( "webapp", "/WEB-INF/conf/TurbineResources.properties");
            config.init();
        }
        catch (Exception e)
        {
            fail(StringUtils.stackTrace(e));
        }
    }

    private static SecurityCacheService getService()
    {
        return (SecurityCacheService)TurbineServices
                .getInstance()
                .getService(SecurityCacheService.SERVICE_NAME);
    }

}






