#include  "d2_polygon.h"
#include  "test_environment.h"
#include  <vector>

int    main( void )
{
	Test_Environment	t;

	//
	// empty
	//
	{
		const D2_Polygon	empty_polygon;

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    empty_polygon.intersect( D2_Vector::origin() ) ) );
	}


	//
	// point polygon
	//
	{
		const D2_Vector		p( +100.0 , +100.0 );

		std::vector<D2_Vector>	v;
		v.push_back( p );


		const D2_Polygon	point_polygon( v );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    point_polygon.intersect( D2_Vector::origin() ) ) );

#ifdef STRICT_FLOAT_CHECK
		ASSERT
		( t.Assert_Equals
		  ( true ,
		    point_polygon.intersect( p ) ) );
		ASSERT
		( t.Assert_Equals
		  ( false ,
		    point_polygon.intersect( p , false ) ) );
#endif
	}


	std::vector<D2_Vector>	rect;
	rect.push_back( D2_Vector( +200.0 , +100.0 ) );
	rect.push_back( D2_Vector( -200.0 , +100.0 ) );
	rect.push_back( D2_Vector( -200.0 , -100.0 ) );
	rect.push_back( D2_Vector( +200.0 , -100.0 ) );

	const D2_Polygon	rectangle( rect );

	//                                   //
	//  +------------------------------+ //
	//  |                              | //
	//  |                              | //
	//  |                              | //
	//  |                              | //
	//  |                              | //
	//  |                              | //
	//  +------------------------------+ //
	//                                   //

	//
	// get_bounding_box()
	//
	{
		D2_Rectangle_Region	r;
		rectangle.get_bounding_box( &r );

		ASSERT
		( t.Assert_Nearly_Equals( +200.0 , r.max_x() ) );
		ASSERT
		( t.Assert_Nearly_Equals( -200.0 , r.min_x() ) );
		ASSERT
		( t.Assert_Nearly_Equals( +100.0 , r.max_y() ) );
		ASSERT
		( t.Assert_Nearly_Equals( -100.0 , r.min_y() ) );
	}


	//
	// intersect
	//
	{
		ASSERT
		( t.Assert_Equals
		  ( true ,
		    rectangle.intersect( D2_Vector( 0.0 , 0.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( true ,
		    rectangle.intersect( D2_Vector( 50.0 , 50.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( true ,
		    rectangle.intersect( D2_Vector( 199.9 , 99.9 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( true ,
		    rectangle.intersect( D2_Vector( -199.9 , -99.9 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    rectangle.intersect( D2_Vector( 200.1 , 100.1 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    rectangle.intersect( D2_Vector( -200.1 , -100.1 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    rectangle.intersect( D2_Vector( +500.0 , +500.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    rectangle.intersect( D2_Vector( +0.0 , +500.0 ) ) ) );
	}


	//
	// intersect 2
	//
	{
		std::vector<D2_Vector>	tri;
		tri.push_back( D2_Vector( -200.0 , -100.0 ) );
		tri.push_back( D2_Vector(    0.0 , +100.0 ) );
		tri.push_back( D2_Vector( +200.0 , -100.0 ) );

		const D2_Polygon	triangle( tri );


		ASSERT
		( t.Assert_Equals
		  ( true ,
		    triangle.intersect( D2_Vector( 0.0 , 0.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    triangle.intersect( D2_Vector( 0.0 , -300.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    triangle.intersect( D2_Vector( 0.1 , -300.0 ) ) ) );
	}


	//
	// intersect 3
	//
	{
		std::vector<D2_Vector>	tri2;
		tri2.push_back( D2_Vector(   0.0 ,   0.0 ) );
		tri2.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri2.push_back( D2_Vector(   0.0 , 200.0 ) );

		const D2_Polygon	triangle2( tri2 );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    triangle2.intersect( D2_Vector( -100.0 , 100.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( true ,
		    triangle2.intersect( D2_Vector( 50 , 100.0 ) ) ) );
	}


	//
	// intersect 4
	//
	{
		std::vector<D2_Vector>	tri3;
		tri3.push_back( D2_Vector(   0.0 ,   0.0 ) );
		tri3.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri3.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri3.push_back( D2_Vector(   0.0 , 200.0 ) );

		const D2_Polygon	triangle3( tri3 );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    triangle3.intersect( D2_Vector( -100.0 , 100.0 ) ) ) );
	}

	//
	// intersect 5
	//
	{
		std::vector<D2_Vector>	tri4;
		tri4.push_back( D2_Vector(   0.0 ,   0.0 ) );
		tri4.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri4.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri4.push_back( D2_Vector( 100.0 , 100.0 ) );
		tri4.push_back( D2_Vector(   0.0 , 200.0 ) );

		const D2_Polygon	triangle4( tri4 );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    triangle4.intersect( D2_Vector( -100.0 , 100.0 ) ) ) );
	}


	//
	// intersect 6
	//
	{
		std::vector<D2_Vector>	rect;
		rect.push_back( D2_Vector(  0 ,  0 ) );
		rect.push_back( D2_Vector( 10 ,  0 ) );
		rect.push_back( D2_Vector( 10 , 10 ) );
		rect.push_back( D2_Vector(  0 , 10 ) );

		const D2_Polygon	r( rect );

		ASSERT
		( t.Assert_Equals( false ,
				   r.intersect( D2_Vector( -100 , 0 ) ) ) );
	}


	//
	// intersect (grid)
	//
	{
		std::vector<D2_Vector>	rect;
		rect.push_back( D2_Vector(  0 ,  0 ) );
		rect.push_back( D2_Vector( 10 ,  0 ) );
		rect.push_back( D2_Vector( 10 , 10 ) );
		rect.push_back( D2_Vector(  0 , 10 ) );

		const D2_Polygon	r( rect );

		int	count = 0;

		for ( int  x = -100  ;  x <= +100  ;  x ++ )
		{
			for ( int  y = -100  ;  y <= +100  ;  y ++ )
			{
				if ( 0 <= x && x <= 10
				     && 0 <= y && y <= 10 )
				{
					continue;
				}

				if ( r.intersect( D2_Vector( x , y ) ) )
				{
					std::cerr << "!["
						  << x << "," << y
						  << "]" << std::endl;
					count ++;
				}
			}
		}

		ASSERT
		( t.Assert_Equals( 0 , count ) );
	}


	//
	// intersect
	//
	{
		std::vector<D2_Vector>	v;
		v.push_back( D2_Vector( 100 , 100 ) );
		v.push_back( D2_Vector( 200 , 100 ) );
		v.push_back( D2_Vector( 200 , 500 ) );

		const D2_Polygon	tri( v );

		//                    //
		//  po1               //
		//                    //
		//  po2          p5   //
		//              /|    //
		//             / |    //
		//            /  |    //
		//           /   |    //
		//          /    |    //
		//         /     |    //
		//  po3  p7  p1  p6   //
		//       /       |    //
		//  po4 p4---p2--p3   //
		//                    //
		//  po5               //

		D2_Vector	p1( 150 , 150 );
		D2_Vector	p2( 150 , 100 );
		D2_Vector	p3( 200 , 100 );
		D2_Vector	p4( 100 , 100 );
		D2_Vector	p5( 200 , 500 );
		D2_Vector	p6( 200 , 150 );
		D2_Vector	p7( 200 , 150 );

		D2_Vector	po1( 50 , 600 );
		D2_Vector	po2( 50 , 500 );
		D2_Vector	po3( 50 , 150 );
		D2_Vector	po4( 50 , 100 );
		D2_Vector	po5( 50 ,   0 );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p1 ) ) );
		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p1 , false ) ) );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p2 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p2 , false ) ) );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p3 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p3 , false ) ) );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p4 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p4 , false ) ) );

		ASSERT
		( t.Assert_Equals( true , tri.intersect( p5 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p5 , false ) ) );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p6 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p6 , false ) ) );

		ASSERT
		( t.Assert_Equals( true  , tri.intersect( p7 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( p7 , false ) ) );


		ASSERT
		( t.Assert_Equals( false , tri.intersect( po1 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( po1 , false ) ) );

		ASSERT
		( t.Assert_Equals( false , tri.intersect( po2 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( po2 , false ) ) );

		ASSERT
		( t.Assert_Equals( false , tri.intersect( po3 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( po3 , false ) ) );

		ASSERT
		( t.Assert_Equals( false , tri.intersect( po4 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( po4 , false ) ) );

		ASSERT
		( t.Assert_Equals( false , tri.intersect( po5 ) ) );
		ASSERT
		( t.Assert_Equals( false , tri.intersect( po5 , false ) ) );
	}


	//
	// empty area
	//
	{
		std::vector<D2_Vector>	a0;
		a0.push_back( D2_Vector( 100.0 , 100.0 ) );
		a0.push_back( D2_Vector( 100.0 , 100.0 ) );
		a0.push_back( D2_Vector( 100.0 , 100.0 ) );
		a0.push_back( D2_Vector( 100.0 , 100.0 ) );
		a0.push_back( D2_Vector( 100.0 , 100.0 ) );

		const D2_Polygon	area_1( a0 );

		a0.push_back( D2_Vector( 100.0 , 100.0 ) );
		const D2_Polygon	area_2( a0 );


		ASSERT
		( t.Assert_Equals
		  ( false ,
		    area_1.intersect( D2_Vector( 0.0 , 0.0 ) ) ) );

		ASSERT
		( t.Assert_Equals
		  ( false ,
		    area_2.intersect( D2_Vector( 0.0 , 0.0 ) ) ) );

#ifdef STRICT_FLOAT_CHECK
		ASSERT
		( t.Assert_Equals
		  ( true ,
		    area_1.intersect( D2_Vector( 100.0 , 100.0 ) ) ) );
		ASSERT
		( t.Assert_Equals
		  ( false ,
		    area_1.intersect( D2_Vector( 100.0 , 100.0 ) , false ) ) );

		ASSERT
		( t.Assert_Equals
		  ( true ,
		    area_2.intersect( D2_Vector( 100.0 , 100.0 ) ) ) );
		ASSERT
		( t.Assert_Equals
		  ( false ,
		    area_2.intersect( D2_Vector( 100.0 , 100.0 ) , false ) ) );
#endif
	}


	//
	// scissoring
	//
	{
		const D2_Rectangle_Region	rectangle( -100 , +100 ,
							   -100 , +100 );

		//                         //
		//              (200,200)  //
		//           +---------+   //
		//           |         |   //
		//    -100   |         |   //
		// +100 +----|----+    |   //
		//      |    |    |    |   //
		//      |    |    |    |   //
		//      |    +---------+   //
		//      |   (0,0) |        //
		//      |         |        //
		// -100 +---------+        //
		//                         //

		std::vector<D2_Vector>	v;
		v.push_back( D2_Vector( 0   ,   0 ) );
		v.push_back( D2_Vector( 200 ,   0 ) );
		v.push_back( D2_Vector( 200 , 200 ) );
		v.push_back( D2_Vector( 0   , 200 ) );
		v.push_back( D2_Vector( 0   ,   0 ) );

		const D2_Polygon	polygon( v );

		D2_Polygon	result;

		polygon.get_scissored_connected_polygon( &result , rectangle );

		ASSERT( t.Assert_Nearly_Equals
			( 10000.0, result.area() ) );

		D2_Rectangle_Region	bbox;
		result.get_bounding_box( &bbox );
		ASSERT( t.Assert_Nearly_Equals
			(   0.0, bbox.min_x() , 1.0e-5 ) );
		ASSERT( t.Assert_Nearly_Equals
			( 100.0, bbox.max_x() , 1.0e-5 ) );
		ASSERT( t.Assert_Nearly_Equals
			(   0.0, bbox.min_y() , 1.0e-5 ) );
		ASSERT( t.Assert_Nearly_Equals
			( 100.0, bbox.max_y() , 1.0e-5 ) );

		ASSERT( t.Assert_Equals
			( size_t(5) , result.points().size() ) );

	}


#if 0
	//
	// overlap
	//
	{
		// XXX: not completed yet

		std::vector<D2_Vector>	rect;
		rect.push_back( D2_Vector( +100.0 , +100.0 ) );
		rect.push_back( D2_Vector( -100.0 , +100.0 ) );
		rect.push_back( D2_Vector( -100.0 , -100.0 ) );
		rect.push_back( D2_Vector( +100.0 , -100.0 ) );

		const D2_Polygon	rectangle( rect );

		// +200          +             //
		//               |\            //
		//               | \           //
		//               |  \          //
		//               |   \         //
		//               |    \        //
		//               |     \       //
		//    -100       |      \ +100 //
		// +100 +--------|-------\+    //
		//      |        |        \    //
		//      |        |        |\   //
		//      |        +--------+-+  //
		//      |                 |    //
		//      |                 |    //
		// -100 +-----------------+    //
		//                             //

		std::vector<D2_Vector>	v;
		v.push_back( D2_Vector( 0   ,   0 ) );
		v.push_back( D2_Vector( 200 ,   0 ) );
		v.push_back( D2_Vector( 0   , 200 ) );

		const D2_Polygon	triangle( v );

		D2_Polygon	result;

		bool	succeeded = rectangle.get_overlap_polygon
					      ( &result , triangle );

		ASSERT( t.Assert_Equals( true , succeeded ) );
	}
#endif

	//
	// get_distance
	//
	{
		std::vector<D2_Vector>	rect;
		rect.push_back( D2_Vector(  0 ,  0 ) );
		rect.push_back( D2_Vector( 10 ,  0 ) );
		rect.push_back( D2_Vector( 10 , 10 ) );
		rect.push_back( D2_Vector(  0 , 10 ) );

		const D2_Polygon	r( rect );

		ASSERT( t.Assert_Nearly_Equals
			( 1.0 , r.get_distance( D2_Vector( 11.0 , 10.0 ) ) ) );

		ASSERT( t.Assert_Nearly_Equals
			( 0.0 , r.get_distance( D2_Vector( 5.0 , 5.0 ) ) ) );

		ASSERT( t.Assert_Nearly_Equals
			( 5.0 , r.get_distance( D2_Vector( 5.0 , 5.0 ) ,
						false ) ) );
	}


	//
	// area
	//
	{
		std::vector<D2_Vector>	rect;
		rect.push_back( D2_Vector( 10 , 10 ) );
		rect.push_back( D2_Vector( 20 , 10 ) );
		rect.push_back( D2_Vector( 20 , 20 ) );
		rect.push_back( D2_Vector( 10 , 20 ) );

		const D2_Polygon	r( rect );

		ASSERT( t.Assert_Nearly_Equals(  100.0 , r.area() ) );
		ASSERT( t.Assert_Nearly_Equals( +200.0 , r.signed_area_2() ) );
	}


	//
	// reverse_clockwise/clockwise, signed_area_2
	//
	{
		std::vector<D2_Vector>	points;
		const D2_Polygon	empty(points);

		points.push_back( D2_Vector( 10 , 10 ) );
		const D2_Polygon	point(points);

		points.push_back( D2_Vector( 20 , 10 ) );
		const D2_Polygon	line(points);

		points.push_back( D2_Vector( 20 , 20 ) );
		const D2_Polygon	triangle(points);

		points.push_back( D2_Vector( 10 , 20 ) );
		const D2_Polygon	rectangle(points);


		ASSERT( t.Assert_Nearly_Equals
			(    0.0 , empty.signed_area_2() ) );
		ASSERT( t.Assert_Nearly_Equals
			(    0.0 , point.signed_area_2() ) );
		ASSERT( t.Assert_Nearly_Equals
			(    0.0 , line.signed_area_2() ) );
		ASSERT( t.Assert_Nearly_Equals
			( +100.0 , triangle.signed_area_2() ) );
		ASSERT( t.Assert_Nearly_Equals
			( +200.0 , rectangle.signed_area_2() ) );


		ASSERT( t.Assert_Equals
			( false , empty.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( false , empty.is_clockwise() ) );

		ASSERT( t.Assert_Equals
			( false , point.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( false , point.is_clockwise() ) );

		ASSERT( t.Assert_Equals
			( false , line.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( false , line.is_clockwise() ) );

		ASSERT( t.Assert_Equals
			( true  , triangle.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( false , triangle.is_clockwise() ) );

		ASSERT( t.Assert_Equals
			( true  , triangle.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( false , triangle.is_clockwise() ) );



		std::vector<D2_Vector>	r_points;
		r_points.push_back( D2_Vector( 10 , 20 ) );
		r_points.push_back( D2_Vector( 20 , 20 ) );
		r_points.push_back( D2_Vector( 20 , 10 ) );
		const D2_Polygon	r_triangle(r_points);

		r_points.push_back( D2_Vector( 10 , 10 ) );
		const D2_Polygon	r_rectangle(r_points);


		ASSERT( t.Assert_Nearly_Equals
			( -100.0 , r_triangle.signed_area_2() ) );
		ASSERT( t.Assert_Nearly_Equals
			( -200.0 , r_rectangle.signed_area_2() ) );

		ASSERT( t.Assert_Equals
			( false , r_triangle.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( true  , r_triangle.is_clockwise() ) );

		ASSERT( t.Assert_Equals
			( false , r_rectangle.is_reverse_clockwise() ) );
		ASSERT( t.Assert_Equals
			( true  , r_rectangle.is_clockwise() ) );
	}


	return( t.exit_status() );
}
