#ifndef	   FDSTREAMBUF_H_INCLUDED
#define	   FDSTREAMBUF_H_INCLUDED

#ifdef HAVE_CONFIG_H
# include  "config.h"
#endif

#include  "streambuf_extension.h"
#include  <sys/types.h>
#include  <unistd.h>

#include  <cstdio>
#include  <string>
#include  <vector>
#include  <csignal>
#include  <cstdlib>
#include  <errno.h>


class  fdstreambuf : public streambuf_extension
{
public:
	enum { BAD_FD = -1 };

protected:
	int	fd_;
	bool	auto_close_;

private:
	char	buffer[BUFSIZ];
	char	output_buffer[BUFSIZ];
	char *	output_buffer_ptr;

	void	initialize_buffer()
	{
		this -> setg( buffer , buffer , buffer );

		this -> output_buffer_ptr = output_buffer;
	}

private:
	 fdstreambuf( const fdstreambuf & );			// not allowed.
	 fdstreambuf &	operator= ( const fdstreambuf & );	// not allowed.

public:
	fdstreambuf()
		: fd_( BAD_FD ) , auto_close_( false )
	{
		this -> initialize_buffer();
	}

	fdstreambuf( int  fd ,  bool  auto_close = false )
		: fd_( fd ) , auto_close_( auto_close )
	{
		this -> initialize_buffer();
	}

	~fdstreambuf()
	{
		this -> flush_output_buffer();

		if ( this -> auto_close_ )
		{
			this -> close();
		}
	}

	void  set_fd( int  fd ,  bool  auto_close = false )
	{
		if ( this -> auto_close_ )
		{
			this -> close();
		}

		this -> fd_ = fd;
		this -> auto_close_ = auto_close;
	}


protected:
	int  underflow()
	{
		int	ch = this -> uflow();

		this -> pbackfail( ch );

		return( ch );
	}

	int  uflow()
	{
		if ( fd_ == BAD_FD )
		{
			return( EOF );
		}

		if ( this -> input_buffer_current()
		     >= this -> buffer + sizeof(this -> buffer) )
		{
			this -> setg( this -> input_buffer_begin() ,
				      this -> input_buffer_begin() ,
				      this -> input_buffer_begin() );
		}

		const size_t	free_buf_size
				= (this -> buffer + sizeof(this -> buffer))
				  - this -> input_buffer_current();

		int	n;

		switch( (n = ::read( fd_ ,
				     this -> input_buffer_current() ,
				     free_buf_size )) )
		{
		case -1:
			std::perror( "read" );
			return( EOF );
			break;

		case  0:
			return( EOF );
			break;

		default:
		      {
			unsigned char	ch;
			ch = *(this -> input_buffer_current());

			this -> setg( this -> input_buffer_begin() ,
				      this -> input_buffer_current() ,
				      this -> input_buffer_current() + n );
			this -> input_buffer_move_current( 1 );

			return( ch );
			break;
		      }
		}
	}

	int	pbackfail( int  ch )
	{
		if ( this -> input_buffer_current()
		     > this -> input_buffer_begin() )
		{
			this -> input_buffer_move_current( -1 );

			return( ch );
		}
		else
		{
			return( EOF );
		}
	}


protected:
	ssize_t	write_all( const char *  buf , size_t  size )
	{
		size_t	n_wrote = 0;

		while( n_wrote < size )
		{
			ssize_t	ret = ::write( this -> fd_ ,
					       buf + n_wrote ,
					       size - n_wrote );

			if ( ret == -1 )
			{
				std::perror( "write" );
				return( -1 );
			}

			n_wrote += ret;
		}

		return( n_wrote );
	}

	ssize_t	flush_output_buffer()
	{
		size_t	ret = this -> write_all
				( this -> output_buffer ,
				  this -> output_buffer_ptr - output_buffer );

		this -> output_buffer_ptr = this -> output_buffer;

		return( ret );
	}


protected:
	int  overflow( int  ch = EOF )
	{
		if ( ch == EOF || fd_ == BAD_FD )
		{
			return( EOF );
		}

		std::streamsize	buf_space = sizeof( this -> output_buffer )
					    - (this -> output_buffer_ptr
					       - this -> output_buffer );

		if ( buf_space >= 1 )
		{
			*output_buffer_ptr = ch;

			output_buffer_ptr ++;
		}

		return( ch );
	}

	std::streamsize	xsputn( const char *  str ,  std::streamsize  n )
	{
		std::streamsize	buf_space = sizeof( this -> output_buffer )
					    - (this -> output_buffer_ptr
					       - this -> output_buffer );

		if ( n < buf_space )
		{
			std::memcpy( output_buffer_ptr , str , n );

			output_buffer_ptr += n;
		}
		else if ( n < buf_space
			  + static_cast<std::streamsize>
			    ( sizeof( this -> output_buffer ) ) )
		{
			std::streamsize	write_size = n - buf_space;

			std::memcpy( this -> output_buffer_ptr ,
				     str , write_size );

			output_buffer_ptr += n;

			if ( this -> flush_output_buffer() == -1 )
			{
				return( -1 );
			}

			std::memcpy( this -> output_buffer_ptr ,
				     str + write_size , n - write_size );
			output_buffer_ptr += n;
		}
		else
		{
			if ( this -> write_all( str , n ) == -1 )
			{
				return( -1 );
			}

			this -> output_buffer_ptr = this -> output_buffer;
		}

		return( n );
	}

	int	sync()
	{
		return( this -> flush_output_buffer() );
	}

public:
	void  close()
	{
		if ( fd_ != BAD_FD && ::close( fd_ ) == -1 )
		{
			std::perror( "close" );
		}

		fd_ = BAD_FD;

		this -> initialize_buffer();
	}

	int	fd() const
	{
		return( fd_ );
	}

	bool	poll() const;
};

#include  "system_call_wrapper.h"

inline
bool   fdstreambuf::poll() const
{
	if ( (this -> input_buffer_end()
	      - this -> input_buffer_current()) > 0 )
	{
		return( 1 );
	}

	return( System_Call_Wrapper::poll_fd( this -> fd_ ) );
}


#endif	/* FDSTREAMBUF_H_INCLUDED */
