package aist.daruma.client.darumaearth;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.Stack;
import java.util.Iterator;
import java.util.HashMap;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import javax.xml.xpath.XPathFactoryConfigurationException;
import javax.xml.xpath.XPathConstants;
import javax.xml.XMLConstants;

import org.w3c.dom.Document;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.Text;
import org.w3c.dom.Attr;
import org.w3c.dom.NamedNodeMap;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.DefaultHandler;

import org.apache.commons.codec.binary.Base64;

public class KMLConverter  extends DefaultHandler {
	final static public String KMLROOT_NODE_STR = "<kml xmlns='http://earth.google.com/kml/2.1'>";
	private DarumaEarthConfig conf;
	private String featureName;
	private String locationXPath;
	private MispRequestBuilder mispReqBuilder;
	
	private DocumentBuilderFactory factory;
	private DocumentBuilder builder;
	private Document doc;
	private Node lastSibling = null;
	private Stack<Node> nodeStack;
	
	private DarumaEarthlNamespaceContext nsContext;
        /* !!! [[07/12/04 11:36 I.Noda]] !!! 
	 * nsContext is for XPath to access scanned XML DOM.
	 * nsContextForSax is for scanning response XML in Sax.
	 */
	private DarumaEarthlNamespaceContext nsContextForSax;

	private boolean inFeatureTagFlag = false;
	private FileOutputStream outStream;
	
	private Node placemarkStyle = null;
	private String defaultNsUri;
	
	private void pushNode(Node newNode) {
//		System.out.println("Push("+newNode.getLocalName()+")");
		nodeStack.push(newNode);
	}
	
	private Node popNode() {
		Node n = nodeStack.pop();
//System.out.println("Pop("+n.getLocalName()+")");		
		return n;
	}
	
	public void startDocument() throws SAXException {
		// just ignore
	}
	public void endDocument() throws SAXException {
		// just ignore
	}
	
	public void characters(char[] ch, int start, int length) {
		if (inFeatureTagFlag) {
			if ( length == 0 ) {
				return;
			}
			
			Node last = nodeStack.peek();
			if (last != doc ) {
				String text = new String(ch, start, length);
				if (lastSibling != null && lastSibling.getNodeType() == Node.TEXT_NODE) {
					((Text)lastSibling).appendData(text);
				} else {
					lastSibling = last.appendChild(doc.createTextNode(text));
				}
			}
		}
	}
	
	public void startPrefixMapping(String prefix, String uri) {
	    /* <<< [[07/12/04 11:36 I.Noda]] <<< */
	    // nsContext.registerNamespace(prefix, uri);
	    /* === [[07/12/04 11:36 I.Noda]] === */
	    nsContextForSax.registerNamespace(prefix, uri);
	    /* >>> [[07/12/04 11:36 I.Noda]] >>> */
	}
	
	public void startElement(String namespaceURI,	 String sName, String qName,Attributes attrs) 
		throws SAXException {

		String tagName = qName;// sName
		String nsUri = namespaceURI;
		if (!inFeatureTagFlag && tagName.equals(featureName)) { 
			inFeatureTagFlag = true;
			this.doc = builder.newDocument();
			this.defaultNsUri = nsUri;
			pushNode(doc);
		}

		if (inFeatureTagFlag) {
			Element elem = null;			
			if ((conf.prefixNs() == null || conf.prefixNs().equals("")) && nsUri.equals(defaultNsUri)) {
				// TODO: THIS IS UGRY WORKAROUND FOR DEFAULT NAMESPACE
				// 接頭辞無しのデフォルト名前空間を XPathで検索するために、
				// XMLの デフォルト名前空間URIに置き換えてしまう
			        elem = doc.createElementNS(XMLConstants.NULL_NS_URI, tagName);				
			} else {
				 elem = doc.createElementNS(nsUri, tagName);
			}
			if (attrs != null) {
				for (int i = 0; i < attrs.getLength(); i++) {
					String ns = attrs.getURI(i);
					String qn = attrs.getQName(i);

					if (qn.startsWith("xmlns:") || qn.equals("xmlns")) {
						ns = XMLConstants.XMLNS_ATTRIBUTE_NS_URI;	
					}
					
					elem.setAttributeNS(ns, qn, attrs.getValue(i));
				}
				
				Node last = nodeStack.peek();
				last.appendChild(elem);
				pushNode(elem);

			}
		} else {
			// do nothing other tag
		}
		lastSibling = null;		
	}

	public void endElement(String namespaceURI,String sName, String qName) throws SAXException {
		if (inFeatureTagFlag && qName.equals(featureName)) { // sName.equals(featureName)
			inFeatureTagFlag = false;
			Node n = popNode();
			this.processToKML(n);
		}

		if (inFeatureTagFlag) {
			popNode();
		}

		lastSibling = null;		
	}
	
	
	/**
	 * create simple node as <tagName>value</tagName>
	 * @param tagName
	 * @param value (String)
	 * @return
	 */
	private Node genSimpleNode(String tagName,String value) {
		Node node = doc.createElement(tagName);
		node.appendChild(doc.createTextNode(value));
		return node;
	}
	
	/**
	 * create simple node as <tagName>double_value</tagName>
	 * @param tagName
	 * @param value (double)
	 * @return
	 */
	private Node genSimpleNode(String tagName,double value) {
		return genSimpleNode(tagName,Double.toString(value));
	}

	/**
	 * create simple node as <tagName>int_value</tagName>
	 * @param tagName
	 * @param value (int)
	 * @return
	 */
	private Node genSimpleNode(String tagName,int value) {
		return genSimpleNode(tagName,Integer.toString(value));
	}

	/**
	 * create <Style> node for Placemark from configuration
	 * @return
	 */
	private Node genPlaceMarkStyle() {
		Node nodeStyle = doc.createElement("Style");
		
		Node nodeIconStyle = doc.createElement("IconStyle");
		nodeIconStyle.appendChild(genSimpleNode("color",conf.iconColor()));
		nodeIconStyle.appendChild(genSimpleNode("scale",conf.iconScale()));
		
		Node nodeIcon = doc.createElement("Icon");
		nodeIcon.appendChild(genSimpleNode("href",conf.iconHref()));
		nodeIcon.appendChild(genSimpleNode("x",conf.iconX()));
		nodeIcon.appendChild(genSimpleNode("y",conf.iconY()));
		nodeIcon.appendChild(genSimpleNode("w",conf.iconW()));
		nodeIcon.appendChild(genSimpleNode("h",conf.iconH()));
		
		nodeIconStyle.appendChild(nodeIcon);
		nodeStyle.appendChild(nodeIconStyle);
		
		Node nodeLabelStyle = doc.createElement("LabelStyle");
		nodeLabelStyle.appendChild(genSimpleNode("color",conf.labelColor()));
		nodeLabelStyle.appendChild(genSimpleNode("scale",conf.labelScale()));
		nodeStyle.appendChild(nodeLabelStyle);
		
		Node nodeLineStyle = doc.createElement("LineStyle");
		nodeLineStyle.appendChild(genSimpleNode("color",conf.lineColor()));
		nodeLineStyle.appendChild(genSimpleNode("width",conf.lineWidth()));
		nodeStyle.appendChild(nodeLineStyle);
		
		placemarkStyle = nodeStyle;

		return placemarkStyle;
	}
	
	/**
	 *  generate <Placemark> node
	 * @param name
	 * @param x
	 * @param y
	 * @param geoNode
	 * @param descNode
	 * @param timeStamp
	 * @return
	 */
	private Node genPlaceMarkBody(
					String name,
					double x,
					double y,
					Node geoNode,
					Node descNode,
					String timeStamp,
					Node nodePlacemarkRoot) {
		
		Node nodePlacemark = doc.createElement("Placemark");

		nodePlacemark.appendChild(genSimpleNode("name",name));
		
		if (conf.timestamp() && timeStamp !=null && !timeStamp.equals("")) {
			// date time の秒数の小数点以下部分を削除 (SS.SSS -> SS)
			String kmlDateTime = timeStamp.replaceFirst("(:\\d{2})\\.\\d{3}", "$1");
			Node timestampNode = doc.createElement("TimeStamp");
			Element whenElem = doc.createElement("when");
			whenElem.setTextContent(kmlDateTime);
			timestampNode.appendChild(whenElem);
			nodePlacemark.appendChild(timestampNode);
		}
		
		if (descNode != null) {
			nodePlacemark.appendChild(descNode);
		}
		
		Node nodeLookAt = doc.createElement("LookAt");
		nodeLookAt.appendChild(genSimpleNode("longitude",x));
		nodeLookAt.appendChild(genSimpleNode("latitude", y));
		nodePlacemark.appendChild(nodeLookAt);
		
		nodePlacemark.appendChild(genPlaceMarkStyle());
		nodePlacemark.appendChild(geoNode);
		
		nodePlacemarkRoot.appendChild(nodePlacemark);
		
		return nodePlacemark;
	}



	private Node genDescTableRow(String label, String value) {
		Node row = doc.createElement("TR");
		Node labelNode = doc.createElement("TD");
		labelNode.setTextContent(label);
		Node valueNode = doc.createElement("TD");
		valueNode.setTextContent(value);
		
		row.appendChild(labelNode);
		row.appendChild(valueNode);
		
		return row;
	}

	private void getDescTableBody(Node featureNode, String xpath, Node descTableNode) {
		//System.out.println(xpath);
		NamedNodeMap attrMapList = featureNode.getAttributes();
		
		if (attrMapList != null) {
			for (int i=0; i < attrMapList.getLength(); i++) {
				Node attr = attrMapList.item(i);
				String attrName = attr.getNodeName();
				if (!(attrName.equals("xmlns") || attrName.startsWith("xmlns:"))) {
					String attrXPath = xpath+"@"+attrName;
					String attrValue = attr.getNodeValue();										
					Node node = genDescTableRow(attrXPath, attrValue);
					descTableNode.appendChild(node);
				}

			}
		}
	
		Node elem = XmlUtil.getFirstChildElement(featureNode);
		while (elem != null) {
			String childXPath = xpath + elem.getLocalName() +"/";
			getDescTableBody(elem, childXPath, descTableNode);
			elem = XmlUtil.getNextSiblingElement(elem);
		}
	
		if (XmlUtil.getFirstChildElement(featureNode) == null ) {
			String value = featureNode.getTextContent();

			value.replaceFirst("/^[ \t\n\r]+/", "");
			value.replaceFirst("/[ \t\n\r]+$/", "");
			if (value != null && !value.equals("")) {
				if (value.length() > conf.valueMaxLen()) {
					value = value.substring(0,conf.valueMaxLen()) + " ...";
				}

				Node node = genDescTableRow(xpath, value);
				descTableNode.appendChild(node);			
			}
		}
		
	}

	

	private Node transformDescTree(Node featureNode,String name, String imageFeature) {

		Node descNode = doc.createElement("description");
		Node descTableNode = doc.createElement("TABLE");
		getDescTableBody(featureNode,"./",descTableNode);

		StringWriter sw = new StringWriter();
		sw.write(name);
		XmlUtil.domToWriter(descTableNode, sw);
		//System.out.println("table="+sw.toString());

		Node descCDATA = doc.createCDATASection(sw.toString());
		descNode.appendChild(descCDATA);
		
		Iterator i = imageFileNameList.keySet().iterator();
		while (i.hasNext()){
			String imageId = (String)i.next();
			try {
				Node n = getImageData(imageId, imageFeature);
				if ( n != null ) {
					descNode.appendChild(n);
				}
			} catch (Exception ex) {
				// ignore. no image Node add on exception;
				if (conf.debug()) {
					ex.printStackTrace();
				}
			}
		}
		return descNode;
	}

	private String getTimestamp(Node featureNode) throws Exception {
		String timestampXPath = conf.timestampXPath();
		Node timestampNode = XmlUtil.getNodeByXPath(featureNode, timestampXPath,nsContext);
		if (timestampNode != null) {
			return timestampNode.getNodeValue();
		} else {
			return null;
		}
	}
	
	private Node getPositionNode(Node featureNode) throws Exception {
		String posXPath = this.locationXPath;
		featureNode.getPrefix();
		Node posNode = XmlUtil.getNodeByXPath(featureNode, posXPath, nsContext);

		return posNode;
	}
	
	
	private double [] convertToLatLon(double x, double y) {
		double [] result = new double[2];
		if (conf.flipXY()) {
			double t = x;
			x = y;
			y = t;
		}
		
		double offsetX = conf.offsetX();
		double offsetY = conf.offsetY();
		double scaleX = conf.scaleX();
		double scaleY = conf.scaleY();
		
		result[0] = offsetX + (scaleX * x);
		result[1] = offsetY + (scaleY * y);		
		return result;
	}

	private void genPlacemarkPointBody(
			String name,
			GmlPoint gmlPoint,
			Node descNode,
			String timeStamp,
			Node placemarkRootNode) {
		StringBuilder sb = new StringBuilder();		
		String altModeStr = "relativeToGround";
		
		double x = gmlPoint.getX();
		double y = gmlPoint.getY();
		double [] latlon = convertToLatLon(x, y);
		double lat = latlon[0];
		double lon = latlon[1];

		sb.append(lat).append(',').append(lon);

		if (gmlPoint.hasZ()) {
			sb.append(',').append(gmlPoint.getZ());
			altModeStr = "absolute";			
		}
		
		String coordStr = sb.toString();
		
		Node pointNode = doc.createElement("Point");
		Node altNode = doc.createElement("altitudeMode");
		altNode.appendChild(doc.createTextNode(altModeStr));
		Node coordNode = doc.createElement("coordinates");

		coordNode.appendChild(doc.createTextNode(coordStr));
		pointNode.appendChild(altNode);
		pointNode.appendChild(coordNode);
		
		genPlaceMarkBody(name, lat, lon, pointNode, descNode, timeStamp, placemarkRootNode);
	}

	private void genPlacemarkLineBody(
			String name,
			GmlLineString gmlLineString,
			Node descNode,
			String timeStamp,
			Node placemarkRootNode) {

		GmlPoint point0 = gmlLineString.getPointAt(0);
		genPlacemarkPointBody(name,point0,descNode,timeStamp,placemarkRootNode);
		
		double [] latlon0 = convertToLatLon(point0.getX(), point0.getY());
		double lat0 = latlon0[0];
		double lon0 = latlon0[1];

		Node lsNode = doc.createElement("LineString");
		Node altNode = doc.createElement("altitudeMode");
		String altMode = "relativeToGround";
		if (point0.hasZ()) {
			altMode = "absolute";
		}
		altNode.appendChild(doc.createTextNode(altMode));				
		
		Node coordNode = doc.createElement("coordinates");
		
		StringBuilder sb = new StringBuilder();
		
		Iterator i = gmlLineString.iterator();
		boolean first = true;
		while ( i.hasNext() ) {
			if (first) {
				first = false;
			} else {
				sb.append(' ');
			}
			
			GmlPoint point = (GmlPoint) i.next();
			double [] latlon = convertToLatLon(point.getX(), point.getY());
			double lat = latlon[0];
			double lon = latlon[1];
			sb.append(lat).append(',').append(lon);
						
			if (point.hasZ()) {
				sb.append(',').append(point.getZ());
			} else {
				sb.append(',').append(conf.offsetZ());
			}

		}

		String coordStr = sb.toString();
		coordNode.appendChild(doc.createTextNode(coordStr));
		lsNode.appendChild(altNode);
		lsNode.appendChild(coordNode);
		
		genPlaceMarkBody(name, lat0, lon0, lsNode, null, timeStamp, placemarkRootNode);
	}

	private void genPlacemarkPolygonBody(
			String name,
			GmlPolygon gmlPolygon,
			Node descNode,
			String timeStamp,
			Node placemarkRootNode) {

		GmlPoint point0 = gmlPolygon.getExterior().getPointAt(0);
		genPlacemarkPointBody(name,point0,descNode,timeStamp,placemarkRootNode);
		
		double [] latlon0 = convertToLatLon(point0.getX(), point0.getY());
		double lat0 = latlon0[0];
		double lon0 = latlon0[1];

		Node polyNode = doc.createElement("Polygon");
		Node altNode = doc.createElement("altitudeMode");
		String altMode = "relativeToGround";
		if (point0.hasZ()) {
			altMode = "absolute";
		}

		altNode.appendChild(doc.createTextNode(altMode));

		Node altNode2 = altNode.cloneNode(true);		
		polyNode.appendChild(altNode);
		

		Node lrNode = doc.createElement("LinearRing");

		
		Node coordNode = doc.createElement("coordinates");
		
		StringBuilder sb = new StringBuilder();
		
		Iterator i = gmlPolygon.getExterior().iterator();
		while ( i.hasNext() ) {
			GmlPoint point = (GmlPoint) i.next();
			double [] latlon = convertToLatLon(point.getX(), point.getY());
			double lat = latlon[0];
			double lon = latlon[1];
			sb.append(lat).append(',').append(lon);
			
			if (point.hasZ()) {
				sb.append(',').append(point.getZ());
			} else {
				sb.append(',').append(conf.offsetZ());
			}
			sb.append(' ');
		}

		String coordStr = sb.toString();
		lrNode.appendChild(coordNode);
		
		coordNode.appendChild(doc.createTextNode(coordStr));
		Node outerBoundaryNode = doc.createElement("outerBoundaryIs");
		outerBoundaryNode.appendChild(altNode2);
		outerBoundaryNode.appendChild(lrNode);
		polyNode.appendChild(outerBoundaryNode);		
		
		genPlaceMarkBody(name, lat0, lon0, polyNode, null, timeStamp, placemarkRootNode);
	}

	
	private Node transformGeoNodes(String name, Node posNode, Node descNode, String timeStamp) 
		throws Exception {
		Node geoNode = XmlUtil.getFirstChildElement(posNode);
		if (geoNode == null){
			return null;
		}

		Node placeMarks = doc.createDocumentFragment();
		try {
		    /* <<< [[07/12/04 11:36 I.Noda]] <<< */
		    //GmlGeometry gmlGeom = GmlGeometry.scanGml(geoNode, nsContext, null);
		    /* === [[07/12/04 11:36 I.Noda]] === */
		    GmlGeometry gmlGeom = 
			GmlGeometry.scanGml(geoNode, nsContextForSax, null);
		    /* >>> [[07/12/04 11:36 I.Noda]] >>> */
			if (gmlGeom.gmlTagName().equals("Point")) {
				GmlPoint gmlPoint = (GmlPoint)gmlGeom;
				genPlacemarkPointBody(name, gmlPoint, descNode,timeStamp, placeMarks);
			} else if (gmlGeom.gmlTagName().equals("LineString")) {
				GmlLineString gmlLineString = (GmlLineString)gmlGeom;
				genPlacemarkLineBody(name, gmlLineString, descNode,timeStamp, placeMarks);			
			} else if (gmlGeom.gmlTagName().equals("Polygon")) {
				GmlPolygon gmlPolygon = (GmlPolygon)gmlGeom;
				genPlacemarkPolygonBody(name, gmlPolygon, descNode,timeStamp, placeMarks);			
			}
		} catch (ScanGmlException sge) {
			if (conf.debug()) {
				System.out.println(sge.getMessage());
			}
		}
		
		return placeMarks;
	}


	/**
	 *  list of <imageId,imageFileName>
	 */
	private HashMap<String,String> imageFileNameList; 
	private HashMap<String,String> imageFileCacheList; 
	
	private String genImageFileName(String imageId) {
		StringBuffer sb = new StringBuffer();

		File imageIdFile = new File(imageId);
		String imageDir = conf.imageDir();
		sb.append(imageDir);
		if (!imageDir.endsWith(File.separator)) {
			sb.append(File.separator);
		}
		
		String basename = imageIdFile.getName();
		sb.append(basename);

		return sb.toString();
	}
	
	/**
	 *  imageノードをとりだす
	 * @param node
	 */
	private void scanImageNode(Node node) throws Exception {
		String imageRefXPath= conf.dataRefXPath();

		String imageId = XmlUtil.getNodeValueByXPath(node, imageRefXPath, nsContext);

		if (imageId != null) {
			String imageFileName = genImageFileName(imageId);
			imageFileNameList.put(imageId, imageFileName);
		}
	}
	
	/**
	 *  image data を サーバから取得し、相当のタグ(node)を作って返す
	 *  
	 */
	private Node getImageData(String imageId, String featureName) throws Exception {
		String imageIdXPath = conf.dataIdXPath();
		String imageDataXPath = conf.dataXPath();
		String encodeXPath = conf.encodeXPath();
		Node resultImageNode = null;
		if (imageFileCacheList.containsKey(imageId)) {
			// got already...
			return null;
		}
		
		String imageFileName = imageFileNameList.get(imageId);
		// request to daruma server

		// "リクエスト"を作ってなげる MispClient
		Node requestNode = mispReqBuilder.genGetFeatureGetImageData(featureName, imageId, imageIdXPath, conf.prefixNs(), conf.prefixUri());
		String reqFileName = "";
		String resFileName = "";
		Node imageDataNode = null;
		try {
			File workDir = new File(conf.workDir()); 
			File reqFile = File.createTempFile("imgreq", ".xml", workDir);
			reqFile.deleteOnExit();
			reqFileName = reqFile.getCanonicalPath();

			File resFile = File.createTempFile("imgres", ".xml", workDir);
			resFile.deleteOnExit();
			resFileName = resFile.getCanonicalPath();

			mispReqBuilder.requestToFile(reqFileName, requestNode);
			MispSOAPClient client = new MispSOAPClient();
			FileInputStream request;
			FileOutputStream response;

			request =  new FileInputStream(reqFileName);
			response = new FileOutputStream(resFileName); //"response_test.xml";
			client.sendRequest(conf.darumaHost(), conf.darumaPort(), request, response);
			response.close();
			Node responseTree  = XmlUtil.fileToDom(resFile, builder);

			/* <<< [[07/12/04 11:36 I.Noda]] <<< 
			 * now, getNodeByXPath can not handle well 
			 * for default namespace.  
			 * so, get the node by xpath="gml:featureMember/*"
			 * Note: `{nsURI}localname' style also do not work well.
			 */
			//imageDataNode = XmlUtil.getNodeByXPath(responseTree, "//"+featureName, nsContext);
			/* === [[07/12/04 11:36 I.Noda]] === */
			imageDataNode = 
			    XmlUtil.getNodeByXPath(responseTree, 
						   "//gml:featureMember/*", 
						   nsContext);
			/* >>> [[07/12/04 11:36 I.Noda]] >>> */

		} catch (FileNotFoundException nfe) {
//			 TODO:something error occured on create request/response file
			nfe.printStackTrace();
		} catch (IOException ioe) {
//			 TODO:something error occured on making/closing request/response file
			ioe.printStackTrace();
		}
		
		// "レスポンス"をみてデコード等する


		if (imageDataNode!=null) {

			Node dataNode = XmlUtil.getNodeByXPath(imageDataNode, imageDataXPath, nsContext);

			/* !!! [[07/12/04 11:36 I.Noda]] !!! */
			if(dataNode == null) {
			    throw new Exception("data Node not found.") ;
			}

			Node encodingNode = XmlUtil.getNodeByXPath(imageDataNode, encodeXPath, nsContext);

			/* !!! [[07/12/04 11:36 I.Noda]] !!! */
			String encoding = "base64" ;
			if(encodingNode != null) {
			    encoding = encodingNode.getTextContent();
			}

			if (encoding.equals("base64")) {
				// TODO:base64 decode! and write out to imageFileName
				String base64Data = dataNode.getTextContent();
				byte[] imageBinary = Base64.decodeBase64(base64Data.getBytes()); 
				try {
				    /* !!! [[07/12/04 11:36 I.Noda]] !!! 
				     * to fix real imageFileName
				     */
				    String realImageFileName = 
					conf.kmlOutputDir() + "/" +
					imageFileName ;
				    //FileOutputStream imgFileStream = new FileOutputStream(imageFileName);
				    FileOutputStream imgFileStream = new FileOutputStream(realImageFileName);
					imgFileStream.write(imageBinary);
					imgFileStream.close();
				} catch (FileNotFoundException fne) {
					// TODO: image file open error
					fne.printStackTrace();
				} catch (IOException ioe) {
					// TODO: image file write error
					ioe.printStackTrace();
				}
				
				Element elem = doc.createElement("img");
				elem.setAttribute("src", imageFileName);
				resultImageNode = elem;

			} else if (encoding.equals("xlink:href")) {
				Element elem = doc.createElement("a");
				String rawdata = dataNode.getTextContent();
				elem.setAttribute("href", rawdata);
				Node clickHereNode = doc.createTextNode("Click Here");
				elem.appendChild(clickHereNode);
				resultImageNode = elem;
			} else {
				// unknown encoding
			}
		}
		
		imageFileCacheList.put(imageId,imageFileName);
		return resultImageNode;


		//if data is base64 encoded, return <img> node 		// 
//		Element resultImageNode = doc.createElement("img");
//		resultImageNode.setAttribute("src", imageFileName);
//		return resultImageNode;

		//if data is xlink:href, return <a href=rawdata>Click Here</a> node
//		Element resultImageNode = doc.createElement("a");
//		resultImageNode.setAttribute("href", rawdata);
//		Node clickHereNode = doc.createTextNode("Click Here");
//		resultImageNode.appendChild(clickHereNode);
//		return resultImageNode;

	}
	
	private int featureCount = 0;


	
	private void processToKML(Node featureNode) {
		featureCount++;
		if (featureCount > 10) {
			featureCount = 0;
			if (conf.debug()) {
				System.out.print(".");
			}
		}
		
		try {
			Node posNode = getPositionNode(featureNode);
			if (posNode == null ) { //
				return;
			}
		
			String imageDataFeature = conf.dataFeature();

			/* !!! [[07/12/04 11:36 I.Noda]] !!! 
			 * add "!" (not) in the if-condition.
			 * this seems bug.
			 */
			if (imageDataFeature != null && 
					!(imageDataFeature.equals("") || imageDataFeature.equals("-"))) {
				scanImageNode(featureNode);
			}

			String timeStamp = getTimestamp(featureNode);		
			String name = conf.defaultName();		

			//	description をつくる -> descNode
			Node descNode = this.transformDescTree(featureNode, name, imageDataFeature);

			Node placeMarks = transformGeoNodes(name, posNode, descNode,timeStamp);
			XmlUtil.domToStream(placeMarks, outStream);
		} catch (Exception ex) {
			// ignores
			if (conf.debug()) {
				System.out.println(ex.getMessage());
				ex.printStackTrace();
			}
		}

	}
	
	public void convert(String featureName, String locXPath, FileInputStream src, FileOutputStream dst)
			throws IOException {
		this.outStream = dst;
		this.featureName = featureName;
		this.locationXPath = locXPath;
		this.nodeStack = new Stack<Node>();
		
		String prefixNs = conf.prefixNs();
		String prefixUri = conf.prefixUri();
		if (prefixUri != null && !prefixUri.equals("") ) {
			if (prefixNs == null) {
				nsContext.registerNamespace("", prefixUri);
			} else if (prefixUri != null && !prefixUri.equals("")) {
				nsContext.registerNamespace(prefixNs, conf.prefixUri());
			}
		}

		imageFileNameList.clear();
		imageFileCacheList.clear();
		
		try {
			OutputStreamWriter out = new OutputStreamWriter(dst,"UTF8");
			out.write(KMLROOT_NODE_STR);
			out.write("<Document><Folder>\n");
			out.write("<name>" + featureName + "</name>\n");
			out.flush();
		} catch (UnsupportedEncodingException usex) {
			// may not be occures
			System.out.print("Internal error for output kml file...");
			System.out.println(usex.getMessage());
		} catch (IOException ioe) {
			System.out.print("Error for write kml file...");
			System.out.println(ioe.getMessage());
			throw ioe;
		}

		SAXParserFactory factory = SAXParserFactory.newInstance();
		
		try {
			SAXParser saxParser = factory.newSAXParser();
			XMLReader reader = saxParser.getXMLReader();
			String id="http://xml.org/sax/features/namespaces";
			reader.setFeature(id,true); // turn on namespace handling            
			saxParser.parse(src, this); // pickup featureName tag and convert them to KML
		} catch (Exception e) {
			System.out.print("Error on parsing response from DaRuMa server...");
			System.out.println(e.getMessage());
		} finally {
			try {
				OutputStreamWriter out = new OutputStreamWriter(dst,"UTF8");			
				out.write("</Folder></Document></kml>");
				out.flush();
			} catch (UnsupportedEncodingException usex) {
				// may not be occures
				System.out.print("Internal Error for output kml file...");
				System.out.println(usex.getMessage());
			} catch (IOException ioe) {
				System.out.print("Error on closing kml file...");
				System.out.println(ioe.getMessage());
				throw ioe;
			}
		}
	}

	public KMLConverter(MispRequestBuilder reqBuilder) throws ParserConfigurationException {
		conf = DarumaEarthConfig.getConfig();
		factory = DocumentBuilderFactory.newInstance();
		factory.setNamespaceAware(true);

		/* !!! [[07/12/04 11:36 I.Noda]] !!! */
		nsContextForSax = new DarumaEarthlNamespaceContext();

		// xpath related
		nsContext = new DarumaEarthlNamespaceContext();
		imageFileNameList = new HashMap<String,String>();
		imageFileCacheList = new HashMap<String,String>();		

		this.mispReqBuilder = reqBuilder;
		
		try {
			this.builder = factory.newDocumentBuilder();
		} catch (ParserConfigurationException pce ) {
			// 
			System.out.print("Internal error. can't get document builder...");
			System.out.println(pce.getMessage());
			throw pce;
		}
	}	

	/**
	 * for test 
	 * @param args
	 */
	static public void main(String[] args) {
		if (args.length < 4) {
			System.out.println("Usage: KMLconverter conv <featureName> <locXpath> <srcfilename> <dstfilename>");
			System.exit(-1);
		}

		try {
			KMLConverter kmlconv = new KMLConverter(new MispRequestBuilder());
			System.out.println("get "+ args[1]+ "/ \""+args[2]+"\" from "+args[3]+" to "+ args[4]);
			String featureName = args[0];
			String locXPath = args[1];
			//locXPath = "./*[namespace-uri()=\"http://www.infosharp.org/schemas/dml\" and local-name()=\"location\"]";
			//locXPath = "./*[local-name()=\"location\"]";			
			FileInputStream src = new FileInputStream(args[2]);
			FileOutputStream dst = new FileOutputStream(args[3]);
			kmlconv.convert(featureName,locXPath, src,dst);
			System.out.println("Done.");
		} catch (Exception e){
			System.out.println("Exception occured "
					+ e.getMessage());
			e.printStackTrace();
		}
	}
}
