#!/usr/bin/env ruby

require 'getoptlong'
require 'socket'

#
# handle command line options
#
def printUsage
  STDERR << "Usage: " + $0 + " --id OBJECT_ID " \
	 << "--location Point(X,Y) --timestamp TIMESTAMP ...\n" \
	 << "\n" \
	 << "Possible options are:\n" \
	 << "  --help\n" \
	 << "  --verbose|-v\n" \
	 << "  --host|-h HOST\n" \
	 << "  --port|-p PORT\n" \
	 << "  --id OBJECT_ID\n" \
	 << "  --location Point(X,Y)\n" \
	 << "  --srsName COORDINATE_SYSTEM_NAME\n" \
	 << "  --timestamp TIMESTAMP\n" \
	 << "  --note ANY_STRING\n" \
	 << "  --add\n" \
	 << "  --replace\n" \
	 << "  --delete\n" \
	 << "  --stdout\n"
end


opt = GetoptLong.new(['--help'         , GetoptLong::NO_ARGUMENT],
		     ['--verbose', '-v', GetoptLong::NO_ARGUMENT],
		     ['--host', '-h'   , GetoptLong::REQUIRED_ARGUMENT],
		     ['--port', '-p'   , GetoptLong::REQUIRED_ARGUMENT],
		     ['--id'           , GetoptLong::REQUIRED_ARGUMENT],
		     ['--location'     , GetoptLong::REQUIRED_ARGUMENT],
		     ['--srsName'      , GetoptLong::REQUIRED_ARGUMENT],
		     ['--timestamp'    , GetoptLong::REQUIRED_ARGUMENT],
		     ['--note'         , GetoptLong::REQUIRED_ARGUMENT],
		     ['--add'          , GetoptLong::NO_ARGUMENT],
		     ['--replace'      , GetoptLong::NO_ARGUMENT],
		     ['--delete'       , GetoptLong::NO_ARGUMENT],
		     ['--stdout'       , GetoptLong::NO_ARGUMENT])

host='localhost'
port='5050'
verbose = false
id = nil
location = nil
srsName = nil
timestamp = nil
note = ''
printRequestOnly = false
doDeleteFlag = false
doAddFlag = true

begin
  opt.each_option do | name, arg |
    case name
    when '--help'
      printUsage
      exit
    when '--verbose'
      verbose = true
    when '--host'
      host = arg
    when '--port'
      port = arg.to_i
    when '--id'
      id = arg
    when '--location'
      location = arg
    when '--srsName'
      srsName = arg
    when '--timestamp'
      timestamp = arg
    when '--note'
      note = arg
    when '--add'
      doDeleteFlag = false
      doAddFlag = true
    when '--replace'
      doDeleteFlag = true
      doAddFlag = true
    when '--delete'
      doDeleteFlag = true
      doAddFlag = false
    when '--stdout'
      printRequestOnly = true
    else
      STDERR.printf( "Illegal option: %s\n", name )
      exit 1
    end
  end
rescue
  printUsage
  exit 1
end

if doDeleteFlag
# check mandatory options
  if id == nil
    printUsage
    exit 1
  end
end

if doAddFlag
# check mandatory options
  if id == nil || location == nil || timestamp == nil
    printUsage
    exit 1
  end

  # check timestamp
  # XXX: brief checking, doesn't detail check such as month >= 13
  if timestamp !~ /^\d\d\d\d-\d\d-\d\dT\d\d:\d\d:\d\d(Z|[-+]\d\d:\d\d)$/
    STDERR << "invalid time format [" + timestamp + "]\n" \
	   << "expected was YYYY-MM-DDTHH:MM:SS[.SUBSEC](Z|[+-]HH:MM)\n" \
	   << "  (e.g. \"2000-01-01T00:00:00Z\")\n"
    exit 1
  end

  # check location
  if location =~ /^\s*Point\s*\(([-+]?\s*\d+(\.\d+)?)\s*,\s*([-+]?\s*\d+(\.\d+)?)\s*\)\s*$/
    x = $1
    y = $3
    locationString =<<"END"
	<gml:Point srsName="#{srsName}">
	  <gml:coordinates>#{x},#{y}</gml:coordinates>
	</gml:Point>
END
  else
    STDERR << "invalid location format [" + location + "]\n" \
	   << "expected was Point(X,Y)\n" \
	   << "  (e.g. \"Point(1.2,-3.4)\")\n"
    exit 1
  end
end


def sendMessage( message, host, port, verbose )
  if verbose
    STDERR.printf( "host = [%s]\n", host )
    STDERR.printf( "port = [%s]\n", port )
  end

  socket = TCPSocket.open( host, port )
  socket << message
  socket.flush()

  while line = socket.gets()
    print line
  end
end


#
# delete
#
if ( doDeleteFlag )
  #
  # prepare message for send
  #
  message =<<"END"
<?xml version="1.0"?>

<misp:Transaction xmlns:misp="http://www.infosharp.org/misp"
		  xmlns="http://www.race.u-tokyo.ac.jp/~asama/IntelligentRobot">
  <misp:Delete typeName="ObjectInfo">
    <misp:Filter>
      <misp:PropertyIsEqualTo>
	<misp:PropertyName>objectID</misp:PropertyName>
	<misp:Literal>#{id}</misp:Literal>
      </misp:PropertyIsEqualTo>
    </misp:Filter>
  </misp:Delete>
</misp:Transaction>
END
  if verbose
    STDERR.printf( "message = [%s]\n", message )
  end

  #
  # send message to delete object
  #
  if printRequestOnly
    STDOUT << message
  else
    sendMessage( message, host, port, verbose )
  end
end


#
# add object
#
if ( doAddFlag )
  #
  # prepare message for send
  #
  message =<<"END"
<misp:Transaction xmlns:misp="http://www.infosharp.org/misp">
  <misp:Insert>

    <ObjectInfo xmlns="http://www.race.u-tokyo.ac.jp/~asama/IntelligentRobot"
		xmlns:gml="http://www.opengis.net/gml">

      <time>#{timestamp}</time>
      <objectID>#{id}</objectID>
      <location>
#{locationString}
      </location>
      <locationNoise>
	<gml:Point srsName="urn:IntelliRobotProject:Noise">
	  <gml:coordinates>0,0</gml:coordinates>
	</gml:Point>
      </locationNoise>
      <angle>
	<gml:Point srsName="urn:IntelliRobotProject:Angle">
	  <gml:coordinates>0,0</gml:coordinates>
	</gml:Point>
      </angle>
      <angleNoise>
	<gml:Point srsName="urn:IntelliRobotProject:AngleNoise">
	  <gml:coordinates>0,0</gml:coordinates>
	</gml:Point>
      </angleNoise>
      <note>#{note}</note>
    </ObjectInfo>

  </misp:Insert>
</misp:Transaction>
END
  if verbose
    STDERR.printf( "message = [%s]\n", message )
  end

  #
  # send message to add object
  #
  if printRequestOnly
    STDOUT << message
  else
    sendMessage( message, host, port, verbose )
  end
end
