package zigen.plugin.db.ui.jobs;

import java.sql.SQLException;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.ide.IDE;

import zigen.plugin.db.DbPlugin;
import zigen.plugin.db.DbPluginConstant;
import zigen.plugin.db.StatusChangeListener;
import zigen.plugin.db.core.IDBConfig;
import zigen.plugin.db.core.SQLHistory;
import zigen.plugin.db.core.SQLHistoryManager;
import zigen.plugin.db.core.SQLInvoker;
import zigen.plugin.db.core.SQLTokenizer;
import zigen.plugin.db.core.SQLUtil;
import zigen.plugin.db.core.SelectIntoChecker;
import zigen.plugin.db.core.StringUtil;
import zigen.plugin.db.core.TableElement;
import zigen.plugin.db.core.TimeWatcher;
import zigen.plugin.db.core.Transaction;
import zigen.plugin.db.ext.oracle.internal.OracleSourceErrorInfo;
import zigen.plugin.db.ext.oracle.internal.OracleSourceErrorSearcher;
import zigen.plugin.db.preference.SQLEditorPreferencePage;
import zigen.plugin.db.ui.actions.ConfirmConnectDBAction;
import zigen.plugin.db.ui.actions.MaxRecordException;
import zigen.plugin.db.ui.editors.QueryViewEditor2;
import zigen.plugin.db.ui.editors.QueryViewEditorInput;
import zigen.plugin.db.ui.views.HistoryView;
import zigen.plugin.db.ui.views.SQLExecuteView;
import zigen.sql.parser.INode;
import zigen.sql.parser.Node;
import zigen.sql.parser.ast.ASTCreateStatement;
import zigen.sql.parser.ast.ASTTarget;
import zigen.sql.parser.ast.ASTType;

public class SqlExecJob extends AbstractJob {
	protected SQLHistoryManager mgr = DbPlugin.getDefault().getHistoryManager();

	protected Transaction trans;

	protected String sqlString;

	protected String secondarlyId;

	protected int executeCount;

	protected OracleSourceErrorInfo[] oracleSourceErrorInfos;

	SQLHistory fSQLHistory;
	
	boolean isSuccess = false;

	public SqlExecJob(Transaction trans, String sqlString, String secondarlyId) {
		super(Messages.getString("SqlExecJob.0")); //$NON-NLS-1$
		this.trans = trans;
		this.sqlString = sqlString;
		this.secondarlyId = secondarlyId;
	}

	protected IStatus run(IProgressMonitor monitor) {
		try {
			start();
			String demiliter = DbPlugin.getDefault().getPreferenceStore().getString(SQLEditorPreferencePage.P_SQL_DEMILITER);

			if (!trans.isConneting()) {
				Display.getDefault().syncExec(new ConfirmConnectDBAction(trans));
				if (!trans.isConneting()) {
					showWarningMessage(DbPluginConstant.MSG_NO_CONNECTED_DB);
					return Status.CANCEL_STATUS;
				}
			}

			if (monitor.isCanceled()) {
				return Status.CANCEL_STATUS;
			}

			SQLTokenizer tokenizer = new SQLTokenizer(sqlString, demiliter);
			while (tokenizer.hasMoreElements()) {
				String sql = tokenizer.nextToken();
				if (sql != null && sql.length() > 0) {
					executeSingleSQL(trans, sql);
					executeCount++; // ssJEg
				}
				if (monitor.isCanceled()) {
					return Status.CANCEL_STATUS;
				}

			}

			// SQLr[XV
			addHistory(trans.getConfig(), sqlString);
			showResults(new UpdateSQLHistoryAction());
			
			return Status.OK_STATUS;

		} catch (SQLException e) {
			showWarningMessage(e.getMessage());

		} catch (Exception e) {
			showErrorMessage(Messages.getString("SqlExecJob.2"), e); //$NON-NLS-1$

		} finally {
			stop();
			// Ԃ̕\
			// updateResponseTime(getTotalTime());

		}
		return Status.OK_STATUS; // G[_CAO\邽߂OKŕԂ

	}

	protected ASTCreateStatement findASTCreateStatement(INode node) {
		int size = node.getChildrenSize();
		for (int i = 0; i < node.getChildrenSize(); i++) {
			INode n = node.getChild(i);
			if (n instanceof ASTCreateStatement) {
				return (ASTCreateStatement) n;
			}
		}
		return null;
	}

	protected ASTType findASTType(ASTCreateStatement cs) {
		int size = cs.getChildrenSize();
		for (int i = 0; i < cs.getChildrenSize(); i++) {
			INode n = cs.getChild(i);
			if (n instanceof ASTType) {
				return (ASTType) n;
			}
		}
		return null;
	}

	protected ASTTarget findASTTarget(ASTCreateStatement cs) {
		int size = cs.getChildrenSize();
		for (int i = 0; i < cs.getChildrenSize(); i++) {
			INode n = cs.getChild(i);
			if (n instanceof ASTTarget) {
				return (ASTTarget) n;
			}
		}
		return null;
	}

	protected void executeSingleSQL(Transaction trans, String sql) throws Exception {
		if (SelectIntoChecker.check(sql)) {
			execute(sql);
		} else {
			if (SQLUtil.isSelect(sql)) {
				showDBEditor(sql);

			} else {
				// PLSQLpSQL̉sȂ悤ɁAătH[}bg
				if (sql.toLowerCase().startsWith("create")) { //$NON-NLS-1$
					// String unsql = sql.replaceAll("\r", "\n"); //$NON-NLS-1$
					// //$NON-NLS-2$

					String sql2 = StringUtil.convertLineSep(sql, "\n");

					// ----------------------------------------------
					// TODO:CREATE JAVA SOURCE ͎słȂ悤ɂ
					// ----------------------------------------------
					INode node = parseSql(sql2);
					ASTCreateStatement cs = findASTCreateStatement(node);
					if (cs != null) {

						execute(sql2); // ͎SQLs

						ASTType astType = findASTType(cs);
						ASTTarget astName = findASTTarget(cs);

						if (astType != null && astName != null) {
							String type = astType.getId().toUpperCase();
							String schema = astName.getSchemaName();
							if (schema == null) {
								schema = trans.getConfig().getSchema().toUpperCase();
							} else {
								schema = schema.toUpperCase();
							}
							String name = astName.getId().toUpperCase();
							oracleSourceErrorInfos = OracleSourceErrorSearcher.execute(trans.getConnection(), schema, name, type);
							if (oracleSourceErrorInfos != null && oracleSourceErrorInfos.length > 0) {

								StringBuffer sb = new StringBuffer();
								for (int i = 0; i < oracleSourceErrorInfos.length; i++) {
									OracleSourceErrorInfo info = oracleSourceErrorInfos[i];
									sb.append(info.getErrorText() + "\n");
								}
								showResults(new ShowWarningMessageAction(sb.toString()));
							} else {
							}
						} else {
							showResults(new ShowWarningMessageAction("PLSQLsSł"));
						}

					} else {

						execute(sql); // ͎SQLs
					}
				} else {
					executeUpdate(sql);
				}
			}
		}

	}

	protected INode parseSql(String sql) {
		INode node = new Node("root"); //$NON-NLS-1$
		zigen.sql.parser.ISqlParser parser = null;
		zigen.sql.parser.IVisitor visitor = null;
		try {
			parser = new zigen.sql.parser.SqlParser(sql, DbPlugin.getSqlFormatRult());
			visitor = new zigen.sql.parser.ASTVisitor();
			parser.parse(node);
			node.accept(visitor, null);
			// for debug
			parser.dump(node);
		} catch (Exception e) {
			DbPlugin.log(e);
		} finally {
			if (parser != null)
				parser = null;
			if (visitor != null)
				visitor = null;
		}
		return node;
	}

	protected void showDBEditor(String query) throws Exception {
		TableElement[] elements = null;
		TimeWatcher time = new TimeWatcher();
		time.start();
		IDBConfig config = trans.getConfig();
		try {
			elements = SQLInvoker.executeQuery(trans.getConnection(), query, config.isConvertUnicode(), config.isNoLockMode());
			time.stop();
			// addHistory(config, query);
			showResults(new ShowResultAction(config, query, elements, time.getTotalTime()));

		} catch (MaxRecordException e) {
			time.stop();
			// addHistory(config, query);
			elements = e.getTableElements();
			showResults(new ShowResultAction(config, query, elements, time.getTotalTime(), e.getMessage()));
			// showWarningMessage(e.getMessage());
		} catch (Exception e) {
			throw e;
		}
		// DbPlugin.log("SQLsʂN鎞ԁF" + time.getTotalTime());
	}

	void addHistory(IDBConfig config, String sql) {
		try {
			// SQLɕۑ
			SQLHistory history = new SQLHistory(config, sql);
			this.fSQLHistory = history;
			this.isSuccess = mgr.addHistory(history);

		} catch (Exception e) {
			DbPlugin.log(e);
		}
	}

	void executeUpdate(String sql) throws Exception {
		try {
			String message = null;
			if (sql.toLowerCase().equals("commit")) { //$NON-NLS-1$
				message = createCommitMessage(trans.getTransactionCount());
				trans.commit();

			} else if (sql.toLowerCase().equals("rollback")) { //$NON-NLS-1$
				message = createRollbackMessage(trans.getTransactionCount());
				trans.rollback();

			} else {
				int rowAffected = SQLInvoker.executeUpdate(trans.getConnection(), sql);
				trans.addCount(rowAffected); // gUNVNXɍXVǉ
				message = createQueryMessage(trans.getTransactionCount());

				if (trans.getConfig().isAutoCommit()) {
					trans.resetCount(); // XVNA
				}
			}
			updateMessage(trans.getConfig(), message, secondarlyId);

		} catch (Exception e) {
			throw e;
		}
	}

	void execute(String sql) throws Exception {
		try {
			String message = null;
			boolean b = SQLInvoker.execute(trans.getConnection(), sql);
			if (b) {
				message = Messages.getString("SqlExecJob.8"); //$NON-NLS-1$
			}
			updateMessage(trans.getConfig(), message, secondarlyId);

		} catch (Exception e) {
			throw e;
		}
	}

	String createQueryMessage(int count) {
		StringBuffer sb = new StringBuffer();
		sb.append(Messages.getString("SqlExecJob.9")); //$NON-NLS-1$
		sb.append(Messages.getString("SqlExecJob.10")); //$NON-NLS-1$
		sb.append(count);
		sb.append(Messages.getString("SqlExecJob.11")); //$NON-NLS-1$
		return sb.toString();
	}

	String createCommitMessage(int count) {
		return count + Messages.getString("SqlExecJob.12"); //$NON-NLS-1$
	}

	String createRollbackMessage(int count) {
		return count + Messages.getString("SqlExecJob.13"); //$NON-NLS-1$
	}

	public int getExecuteCount() {
		return executeCount;
	}

	public OracleSourceErrorInfo[] getOracleSourceErrorInfos() {
		return oracleSourceErrorInfos;
	}

	protected class ShowResultAction implements Runnable {
		IDBConfig config = null;

		String query = null;

		TableElement[] elements = null;

		String responseTime = null;

		String message = ""; //$NON-NLS-1$

		public ShowResultAction(IDBConfig config, String query, TableElement[] elements, String responseTime, String message) {
			this.config = config;
			this.query = query;
			this.elements = elements;
			this.responseTime = responseTime;
			this.message = message;
		}

		public ShowResultAction(IDBConfig config, String query, TableElement[] elements, String responseTime) {
			this(config, query, elements, responseTime, ""); //$NON-NLS-1$
		}

		public void run() {
			try {
				IWorkbenchPage page = DbPlugin.getDefault().getPage();
				QueryViewEditorInput input = new QueryViewEditorInput(config, query, secondarlyId);
				IEditorPart editor = IDE.openEditor(page, input, DbPluginConstant.EDITOR_ID_QueryView2, false);

				if (editor instanceof QueryViewEditor2) {
					QueryViewEditor2 tEditor = (QueryViewEditor2) editor;
					tEditor.update(query, elements, responseTime);
					tEditor.setInfomationText(message);

				}

			} catch (Exception e) {
				DbPlugin.log(e);
			}

		}

	}

	// SQLr[̍XV
	protected class UpdateSQLHistoryAction implements Runnable {
		public UpdateSQLHistoryAction() {
		}

		public void run() {
			try {
				if (isSuccess && fSQLHistory != null) {
					IViewPart part = DbPlugin.findView(DbPluginConstant.VIEW_ID_HistoryView);
					if (part instanceof HistoryView) {
						HistoryView hv = (HistoryView) part;
						hv.updateHistoryView(fSQLHistory);
						StatusChangeListener.fireStatusChangeListener(hv, StatusChangeListener.EVT_UpdateHistory);
					}else{
						// SQLr[ꍇ̏
						StatusChangeListener.fireStatusChangeListener(null, StatusChangeListener.EVT_UpdateHistory);
					}

					
				}
			} catch (Exception e) {
				DbPlugin.log(e);
			}

		}

	}

}
