/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.deculture.translate;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import net.morilib.automata.lr.ContextFreeGrammar;
import net.morilib.automata.lr.ContextFreeRule;
import net.morilib.automata.lr.GrammarSymbol;
import net.morilib.automata.lr.LRAction;
import net.morilib.automata.lr.Nonterminal;
import net.morilib.automata.lr.Terminal;
import net.morilib.deculture.misc.CharacterTerminal;
import net.morilib.deculture.misc.ConstantTerminal;
import net.morilib.deculture.misc.DecultureException;
import net.morilib.deculture.sh.DecultureBuiltInCommands;
import net.morilib.sh.ShBuiltInCommands;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShProcess;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2014/01/15
 */
public class DecultureTranslatorJavaScript
extends AbstractDecultureTranslator {

	private abstract class Puts implements ShProcess {

		//
		abstract boolean isput(int st, GrammarSymbol g);

		//
		abstract void put(PrintStream out, String f, int st,
				GrammarSymbol g);

		//
		abstract void start(PrintStream out, int st, GrammarSymbol g);

		//
		abstract String end();

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream stdin, PrintStream stdout,
				PrintStream stderr, String... args)
				throws IOException {
			Set<Map.Entry<GrammarSymbol, LRAction>> s;
			String f;

			for(int k = 0; k < lr.getTable().getSizeOfStates(); k++) {
				stdout.printf("\t\tcase %d:\n", k);
				s = lr.getTable().actions(k).entrySet();
				f = "";

				for(Map.Entry<GrammarSymbol, LRAction> e : s) {
					if(isput(k, e.getKey())) {
						start(stdout, k, e.getKey());
						break;
					}
				}

				for(Map.Entry<GrammarSymbol, LRAction> e : s) {
					if(isput(k, e.getKey())) {
						stdout.printf("\t\t\tif(__c !== null) {\n");
						stdout.printf("\t\t\t\t__t = __c;\n");
						stdout.printf("\t\t\t} else if(this.__lookahead == null) {\n");
						stdout.printf("\t\t\t\t__t = this.__lookahead = this._next();\n");
						stdout.printf("\t\t\t} else {\n");
						stdout.printf("\t\t\t\t__t = this.__lookahead;\n");
						stdout.printf("\t\t\t}\n");
						break;
					}
				}

				for(Map.Entry<GrammarSymbol, LRAction> e : s) {
					if(isput(k, e.getKey())) {
						_put(this, k, e.getKey(), f, stdout);
						f = "} else ";
					}
				}
				if(!f.equals(""))  stdout.printf("\t\t\t}\n");
				stdout.printf("\t\t\t%s\n", end());
			}
			return 0;
		}

	}

	private class PutShift extends Puts {

		@Override
		boolean isput(int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isShift();
		}

		@Override
		void put(PrintStream out, String f, int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			if(a.isShift()) {
				out.printf("\t\t\t\treturn %d;\n",
						a.getNextStateID());
			}
		}

		@Override
		void start(PrintStream out, int st, GrammarSymbol g) {
		}

		@Override
		String end() {
			return "return -1;";
		}

	}

	private class PutReduce extends Puts {

		@Override
		boolean isput(int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isReduce();
		}

		@Override
		void put(PrintStream out, String f, int st, GrammarSymbol g) {
			ContextFreeRule r;
			LRAction a;
			int l;

			a = lr.getTable().action(st, (Terminal)g);
			if(!a.isReduce())  return;
			r = a.getReduceRule();
			l = r.getDerivedSymbolLength();
			out.printf("\t\t\t\tthis._pop(%d);\n", l);
			out.printf("\t\t\t\t__k = this._refsyn();\n");
			out.printf("\t\t\t\t__k = this.getGoto(__k, %d);\n",
					getNonterminalNumber(r.getLeftSymbol()));
			out.printf("\t\t\t\tthis._push(__k, $$);\n");
			out.printf("\t\t\t\treturn __k;\n");
		}

		@Override
		void start(PrintStream out, int st, GrammarSymbol g) {
			ContextFreeRule r;
			LRAction a;
			Object o;
			String s;
			int l;

			a = lr.getTable().action(st, (Terminal)g);
			if(!a.isReduce())  return;
			r = a.getReduceRule();
			l = r.getDerivedSymbolLength();
			o = actionMap.get(a.getReduceRule());

			if(o != null) {
				out.printf("\t\t\t\t$$ = null;\n", l);
				s = ActionConverterJS.translate(r, lr.getDefinition(),
						o.toString());
				out.printf("\t\t\t\t%s\n", s);
			} else if(l > 0) {
				s = ActionConverterJS.translate(r, lr.getDefinition(),
						"$$ = $1;");
				out.printf("\t\t\t\t%s\n", s);
			} else {
				out.printf("\t\t\t\t$$ = null;\n", l);
			}
		}

		@Override
		String end() {
			return "return -1;";
		}

	}

	private class PutAccept extends Puts {

		@Override
		boolean isput(int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			return a.isAccept();
		}

		@Override
		void put(PrintStream out, String f, int st, GrammarSymbol g) {
			LRAction a;

			a = lr.getTable().action(st, (Terminal)g);
			if(a.isAccept()) {
				out.printf("\t\t\t\treturn true;\n");
			}
		}

		@Override
		void start(PrintStream out, int st, GrammarSymbol g) {
		}

		@Override
		String end() {
			return "return false;";
		}

	}

	private class PutGoto implements ShProcess {

		@Override
		public int main(ShEnvironment env, ShFileSystem fs,
				InputStream stdin, PrintStream out,
				PrintStream stderr, String... args)
				throws IOException {
			Set<Map.Entry<Nonterminal, Integer>> s;
			String f;
			int n;

			for(int k = 0; k < lr.getTable().getSizeOfStates(); k++) {
				out.printf("\t\tcase %d:\n", k);
				s = lr.getTable().goTos(k).entrySet();
				f = "";
				for(Map.Entry<Nonterminal, Integer> e : s) {
					n = getNonterminalNumber(e.getKey());
					out.printf("\t\t\t%sif(__n === %d) {\n", f, n);
					out.printf("\t\t\t\treturn %d;\n", e.getValue());
					f = "} else ";
				}
				if(!f.equals(""))  out.printf("\t\t\t}\n");
				out.printf("\t\t\tthrow 'Internal Error';\n");
			}
			return 0;
		}

	}

	//
	private static final Pattern E_CONST = Pattern.compile(
			"([A-Za-z][A-Za-z0-9]*\\.)*[A-Z][A-Z0-9]*");
	private static final String RESOURCE =
			"/net/morilib/deculture/translate/y.tab.js.sh";

	//
	DecultureTranslatorJavaScript(LRObject lr) {
		super(lr);
	}

	private void _put(Puts p, int st, GrammarSymbol g, String f,
			PrintStream out) {
		String x, a;
		int c;

		if(g instanceof CharacterTerminal) {
			c = ((CharacterTerminal)g).getCharacter();
			if(c >= ' ' && c < 0x7f) {
				out.printf("\t\t\t%sif(this.__eq(__t, '%c')) {\n",
						f, (char)c);
			} else if(c == '\n') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\n')) {\n", f);
			} else if(c == '\r') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\r')) {\n", f);
			} else if(c == '\t') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\t')) {\n", f);
			} else if(c == '\f') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\f')) {\n", f);
			} else if(c == '\b') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\b')) {\n", f);
			} else if(c == '\\') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\\\')) {\n", f);
			} else if(c == '\'') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\\'')) {\n", f);
			} else if(c == '\"') {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\\"')) {\n", f);
			} else {
				out.printf("\t\t\t%sif(this.__eq(__t, '\\u%04x')) {\n",
						f, c);
			}
			p.put(out, f, st, g);
		} else if(g instanceof ConstantTerminal) {
			a = ((ConstantTerminal)g).getConstant();
			x = lr.getDefinition().getType(a);
			if(a.equals(__ERROR)) {
				out.printf("\t\t\t\t%sif(this.__eq(__t, \'\')) {\n", f);
			} else if(E_CONST.matcher(x).matches()) {
				out.printf("\t\t\t%sif(this.__eq(__t, %s)) {\n",
						f, x);
			} else {
				out.printf("\t\t\t%sif(__t instanceof %s) {\n",
						f, x);
			}
			p.put(out, f, st, g);
		} else if(g == ContextFreeGrammar.ENDMARKER) {
			out.printf("\t\t\t%sif(this.__end(__t)) {\n", f);
			p.put(out, f, st, g);
		} else {
			throw new DecultureException(g.toString());
		}
	}

	@Override
	protected String getResourceName() {
		return RESOURCE;
	}

	/**
	 * 
	 * @return
	 */
	public ShBuiltInCommands getCommands() {
		DecultureBuiltInCommands c;

		c = new DecultureBuiltInCommands();
		c.putCommand("put_shift", new PutShift());
		c.putCommand("put_reduce", new PutReduce());
		c.putCommand("put_accept", new PutAccept());
		c.putCommand("put_goto", new PutGoto());
		return c;
	}

	/* (non-Javadoc)
	 * @see net.morilib.deculture.translate.DecultureTranslator#getExtension()
	 */
	@Override
	public String getExtension() {
		return ".js";
	}

}
