/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.content.message;

import java.util.*;
import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.content.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.parser.*;
import com.aibonware.viewnaver.task.*;

public abstract class MessageContent implements TabContent {
	public final MessageTitleList messageTitles;
	private final String listUrl;
	private final String title;
	
	public static interface MessageLogReader {
		public TreeMap<Integer, Message> getMessageLog();
	}
	
	public MessageContent(
			MessageTitleList messages, 
			String listUrl, 
			String title,
			MessageLogReader messageLogReader) {
		this.messageTitles = messages;
		this.listUrl = listUrl;
		this.title = title;
	}

	protected static abstract class MessageContentCategory implements TabContentCategory {
		private final String listUrl;
		private final String title;
		private final MessageLogReader messageLogReader;
		
		public MessageContentCategory(String listUrl, String title, MessageLogReader messageLogReader) {
			this.listUrl = listUrl;
			this.title = title;
			this.messageLogReader = messageLogReader;
		}

		@Override public boolean equals(Object o) {
			if(!(o instanceof MessageContentCategory)) return false;
			MessageContentCategory dest = (MessageContentCategory)o;
			return listUrl.equals(dest.listUrl);
		}

		public abstract MessageContent createMessageContent(MessageTitleList messages);
		
		public void createInitialContent(final TabContainer container, final TabContainerView containerView, TabContentView initialView) {
			TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getMessageListFlow();

			taskFlow.addStatusReceiver(((MessageContentView)initialView).createUpdateTaskStatusReceiver(taskFlow));
			
			taskFlow.execute(new WorkerTask() {
				 MessageTitleList messageTitles = null;
				
				 public void exec() {
					ViewNaver.println(title + "O̓ǂݍݒ...");
					TreeMap<Integer, Message> messageLog = messageLogReader.getMessageLog();
					ViewNaver.println(title + "O̓ǂݍ݊B");

					if(messageLog != null) {
						messageTitles = new MessageTitleList(-1, -1, -1);
	
						for(Message message: messageLog.values()) {
							messageTitles.addMessageTitle(message.getTempMessageTitle());
						}
					}
	
					if(ViewNaver.instance.naverSession.nowLogin()) {
						try {
							ViewNaver.println(title + "T[o擾...");
							MessageListParser parser;
							
							parser = new MessageListParser(listUrl);
							MessageTitleList remoteMessages = parser.parse();
	
							if(messageTitles != null) {
								messageTitles.merge(remoteMessages);
							} else {
								messageTitles = remoteMessages;
							}
	
							ViewNaver.println(title + "T[o擾B");
						} catch(Exception e) {
							ViewNaver.err(e);
						}
					}

					messageTitles.sort();

					taskFlow.execute(new GUITask() {
						public void exec() {
							if(messageTitles == null) {
								containerView.revertEmptyView(MessageContentCategory.this);
								return;
							}

							final MessageContent content = createMessageContent(messageTitles);

							container.addContent(content);
							containerView.activate(content.getCategory());
						}
					});
				}
			});
		}
		
		public String getCategoryName() {
			return title;
		}

		@Override public int hashCode() {
			return MessageContentCategory.class.hashCode() + listUrl.hashCode();
		}

		public abstract Class getContentClass();
	};

	public String getContentTitle() {
		return title;
	}

	public abstract TabContentCategory getCategory();

	public void replaceUpdatedContent(final TabContainer container, final TabContainerView containerView, TabContentView contentView) {
		TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getMessageListFlow();

		taskFlow.addStatusReceiver(((MessageContentView)contentView).createUpdateTaskStatusReceiver(taskFlow));

		taskFlow.execute(new WorkerTask() {
			 MessageTitleList messages = null;
	 
			 public void exec() {
				try {
					ViewNaver.println(title + "ǂݍݒ...");
					MessageListParser parser;
					
					parser = new MessageListParser(listUrl);
					messages = parser.parse();
					
					ViewNaver.println(title + "ǂݍݏIB");
				} catch(Exception e) {
					ViewNaver.err(e);
				}

				taskFlow.execute(new GUITask() {
					public void exec() {
						TabContent newContent = ((MessageContentCategory)getCategory()).createMessageContent(messages);

						container.replace(MessageContent.this, newContent);
						containerView.activate(newContent.getCategory());
					}
				});
			}
		});
	}
}
