/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.content.message;

import java.text.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.parser.*;
import com.aibonware.viewnaver.task.*;
import com.aibonware.viewnaver.net.*;
import com.aibonware.viewnaver.browser.*;

@SuppressWarnings("serial")
public class MessageListView extends JTable {
	private MessageTitleList titles = null;
//	private HashMap<Integer, Message> messages = new HashMap<Integer, Message>();
	private final Browser messageArea;
	private final String messageUrlBase;
	private final JLabel messageHeader;
	private final MessageAdapter messageAdapter;

	public static interface MessageAdapter {
		public void saveMessage(Message message);
		public Message getMessage(int messageId);
	};
	
	private Object[] headerNames = new Object[] {"No", "Server state", "Date", "Country", "ID", "Title"};
	private int[] headerWidthList = new int[] {40, 30, 80, 30, 90, 400};

	private DefaultTableModel tableModel = new DefaultTableModel(headerNames, 0) {
		@Override public boolean isCellEditable(int row, int column) {
			return false;
		}
	};

	private TableColumnModel columnModel;
	
	public MessageListView(Browser messageArea, String messageUrlBase, JLabel messageHeader, MessageAdapter messageSaver) {
		this.messageArea = messageArea;
		this.messageUrlBase = messageUrlBase;
		this.messageHeader = messageHeader;
		this.messageAdapter = messageSaver;

		setModel(tableModel);

		columnModel = getColumnModel();

		for(int i=0; i<headerWidthList.length; i++) {
			columnModel.getColumn(i).setPreferredWidth(headerWidthList[i]);
		}

		setShowGrid(false);

		this.getSelectionModel().addListSelectionListener(new ListSelectionListener() {
			public void valueChanged(ListSelectionEvent e) {
				int row = MessageListView.this.getSelectedRow();

				if(row == -1) return;

				MessageTitle title = titles.getMessageTitle(row);
				Message message = messageAdapter.getMessage(title.messageId);

				if(message == null) {
					showEmptyMessage();
				} else {
					showMessage(message, title.title);
				}
			}
		});

		addMouseListener(new MouseAdapter() {
			@Override public void mouseClicked(MouseEvent event) {
				if(event.getClickCount() > 1) {
					int row = rowAtPoint(event.getPoint());
					if(row == -1) return;

					final MessageTitle messageTitle = titles.getMessageTitle(row);

					TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getMessageFlow(messageTitle.messageId);

					taskFlow.execute(new WorkerTask() {
						 public void exec() {
							try {
								ViewNaver.println("bZ[WNo." + messageTitle.messageId + "̓ǂݍ݊Jn..");
								
								MessageParser parser = new MessageParser(messageTitle, MessageListView.this.messageUrlBase + messageTitle.messageId);
								final Message message = parser.parse();
								
								ViewNaver.println("bZ[WNo." + messageTitle.messageId + "̓ǂݍݏIB");
								
								MessageListView.this.messageAdapter.saveMessage(message);
								
								taskFlow.execute(new GUITask() {
									public void exec() {
//										messages.put(messageTitle.messageId, message);
										showMessage(message, messageTitle.title);
									}
								});
							} catch(ParsingException e) {
								ViewNaver.err(e);
							} catch(NetException e) {
								ViewNaver.err(e);
							}
						}
					});
				}
			}
		});
	}
	
	public void showEmptyMessage() {
		messageArea.setBodyHtml("<font color=gray><i>ĩbZ[W̓e͂܂T[o擾Ă܂Bj</i></font>");
		messageHeader.setText(" ");
	}

	public void showMessage(Message message, String title) {
		messageArea.setBodyHtml("<font face=\"MS UI Gothic\">" + message.text + "</font>");
		messageHeader.setText(
				"    "    + message.country
				+ " "     + message.poster
				+ "     " + headerDateFormat.format(message.postDate)
				/*+ "  "    + title */);
	}

	private static SimpleDateFormat titleDateFormat = new SimpleDateFormat("yyyy/MM/dd");
	private static SimpleDateFormat headerDateFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

	public void setMessageList(MessageTitleList newTitles) {
		if(this.titles != null) {
			this.titles.merge(newTitles);
		} else {
			this.titles = newTitles;
		}
		
		this.titles.sort();

		messageArea.setBodyHtml("");
		messageHeader.setText(" ");

		tableModel.setRowCount(0);

		if(titles == null) return;
		
		for(int i=0; i<titles.getMessageCount(); i++) {
			MessageTitle title = titles.getMessageTitle(i);

			String readText = title.received ? " " : "*";
		
			String serverState;
			
			if(title.remoteStored) {
				serverState = "s";
			} else {
				serverState = " ";
			}
			
			tableModel.addRow(new Object[] {
					title.messageId,
					serverState,
					titleDateFormat.format(title.date),
					title.country, 
					readText + title.person, 
					title.title});
		}
	}
	
	public MessageTitle getSelectedMessage() {
		int row = this.getSelectedRow();
		if(row == -1) return null;

		return titles.getMessageTitle(row);
	}
	
	public int[] getSelectedMessageIdList() {
		int[] rows = this.getSelectedRows();
		
		int[] idList = new int[rows.length];
		
		for(int i=0; i<rows.length; i++) {
			idList[i] = titles.getMessageTitle(rows[i]).messageId;
		}
		
		return idList;
	}
	
	public MessageTitle[] getSelectedMessageList() {
		int[] rows = this.getSelectedRows();
		
		MessageTitle[] messageList = new MessageTitle[rows.length];
		
		for(int i=0; i<rows.length; i++) {
			messageList[i] = titles.getMessageTitle(rows[i]);
		}
		
		return messageList;
	}
}
