/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.content.thread;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.content.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.task.*;
import com.aibonware.viewnaver.config.*;

public class ThreadContent extends TabContentProxy<CompositeThread> {
	private final String dispName;
	private final String boardName;
	
	private static String titleToDispName(String poster, String title) {
		ThreadWindowConfig config = (ThreadWindowConfig)ViewNaver.instance.config.getAncestor("ThreadWindow");

		if(title.length() >config.maxThreadTabTitleLen) {
			title = title.substring(0, config.maxThreadTabTitleLen) + "..";
		}

		return poster + ":" + title;
	}
	
	public ThreadContent(CompositeThread target, String boardName) {
		super(target);

		this.boardName = boardName;
		this.dispName = titleToDispName(target.thread.poster, target.thread.title);
	}

	public static ThreadClientContentCategory getCategory(String boardName, int nid, String poster, String title) {
		return new ThreadClientContentCategory(boardName, nid, titleToDispName(poster, title));
	}

	public static ThreadClientContentCategory getCategory(String boardName, int nid) {
		return new ThreadClientContentCategory(boardName, nid, String.valueOf("thread:" + nid));
	}

	private static class ThreadClientContentCategory implements TabContentCategory {
		private final String boardName;
		private final int nid;
		private final String dispName;

		public ThreadClientContentCategory(String boardName, int nid, String dispName) {
			this.boardName = boardName;
			this.nid = nid;
			this.dispName = dispName;
		}

		@Override public boolean equals(Object o) {
			if(!(o instanceof ThreadClientContentCategory)) return false;
			ThreadClientContentCategory dest = (ThreadClientContentCategory)o;

			if(!boardName.equals(dest.boardName)) return false;
			if(nid != dest.nid) return false;
	
			return true;
		}

		public void createInitialContent(final TabContainer container, final TabContainerView containerView, TabContentView initialView) {
			TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getThreadFlow(boardName, nid);
			
			taskFlow.addStatusReceiver(((ThreadView)initialView).createTaskStatusReceiver(taskFlow));
			
			taskFlow.execute(new WorkerTask() {
				public void exec() {
					final CompositeThread newThread = ViewNaver.instance.cache.getCompositeThread(boardName, nid, false);

					if(newThread != null) {
						ViewNaver.instance.cache.putCompositeThread(newThread);
						ViewNaver.instance.threadStorage.storeThread(newThread);
					}

					taskFlow.execute(new GUITask() {
						public void exec() {
							if(newThread == null) {
								containerView.revertEmptyView(ThreadClientContentCategory.this);
								return;
							}

							final ThreadContent content = new ThreadContent(newThread, boardName);

							container.addContent(content);
							containerView.activate(content.getCategory());
						}
					});
				}
			});
		}
		
		public String getCategoryName() {
			return dispName;
		}

		@Override public int hashCode() {
			return boardName.hashCode() + nid;
		}

		public Class getContentClass() {
			return ThreadContent.class;
		}
	};

	public String getContentTitle() {
		return dispName;
	}

	public TabContentCategory getCategory() {
		return new ThreadClientContentCategory(boardName, target.thread.nid, dispName);
	}

	public void replaceUpdatedContent(final TabContainer container, final TabContainerView containerView, TabContentView contentView) {
		TaskFlow taskFlow = ViewNaver.instance.taskFlowList.getThreadFlow(boardName, target.thread.nid);

		taskFlow.addStatusReceiver(((ThreadView)contentView).createTaskStatusReceiver(taskFlow));
		
		taskFlow.execute(new WorkerTask() {
			public void exec() {
				final CompositeThread thread = ViewNaver.instance.cache.getCompositeThread(boardName, target.thread.nid, true);

				taskFlow.execute(new GUITask() {
					public void exec() {
						TabContent newContent = new ThreadContent(thread, boardName);

						container.replace(ThreadContent.this, newContent);
						containerView.activate(newContent.getCategory());
					}
				});
			}
		});
	}
}
