/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.storage;

import java.io.*;
import java.util.*;

import javax.xml.parsers.*;
import javax.xml.transform.*;
import javax.xml.transform.dom.*;
import javax.xml.transform.stream.*;

import org.w3c.dom.*;
import org.xml.sax.*;

import com.aibonware.viewnaver.*;
import com.aibonware.viewnaver.model.*;
import com.aibonware.viewnaver.xml.*;

public class ThreadStorage {
	/** LbVĂt@CoÕXbh̃Xg */
	private HashMap<String, TreeMap<Integer, CompositeThread>> cache = new HashMap<String, TreeMap<Integer, CompositeThread>>();
	
	public ThreadStorage() {}

	public synchronized void storeThread(CompositeThread thread) {
		getThreadsInBoard(thread.boardName).put(thread.thread.nid, thread);
	}

	public synchronized boolean existsThread(String boardName, int nid) {
		if(getThreadsInBoard(boardName).get(nid) != null) return true;
	
		return new File("logs" + File.separator + boardName + File.separator + boardName + "_" + nid + ".xml").exists();
	}
	
	private int getNid(String fileName) {
		int start = fileName.indexOf("_");
		int end = fileName.indexOf(".");
		
		if(start >= end) return -1;
		
		try {
			return Integer.parseInt(fileName.substring(start+1, end));
		} catch(NumberFormatException e) {
			return -1;
		}
	}
	
	private Comparator<File> reverseNidComparator = new Comparator<File>() {
		public int compare(File src, File dest) {
			return -(getNid(src.getName()) - getNid(dest.getName()));
		}
	};

	public void walkFiles(ThreadStorageWalker walker) {
		File[] boardDirs = new File("logs\\").listFiles(new FileFilter() {
			public boolean accept(File file) {
				return file.isDirectory();
			}
		});

		for(File dir: boardDirs) {
			final String dirName = dir.getName();
			
			File[] files = dir.listFiles(new FilenameFilter() {
				public boolean accept(File parent, String fileName) {
					return fileName.startsWith(dirName) && fileName.endsWith(".xml");
				}
			});

			Arrays.sort(files, reverseNidComparator);
			
			for(File file: files) {
				if(!walker.walk(file)) {
					walker.canceled();
					return;
				}
			}
		}

		walker.finished();
	}
	
	public synchronized CompositeThread readThread(String boardName, int nid) throws StorageException {
		CompositeThread thread = getThreadsInBoard(boardName).get(nid);

		if(thread != null) return thread;

		try {
			String fileName = "logs" + File.separator + boardName + File.separator + boardName + "_" + nid + ".xml";
	
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();
			DocumentBuilder builder = dbfactory.newDocumentBuilder();
			Document doc = builder.parse(new BufferedInputStream(new FileInputStream(fileName)));
	
			Element root = doc.getDocumentElement();
			Element threadElem = BaseXMLReader.getAttrElem(root, "CompositeThread");
	
			CompositeThreadReader reader = new CompositeThreadReader();

			return reader.parse(threadElem);
		} catch(IOException e) {
			throw new StorageException(e);
		} catch(XMLReaderException e) {
			throw new StorageException(e);
		} catch(SAXException e) {
			throw new StorageException(e);
		} catch(ParserConfigurationException e) {
			throw new StorageException(e);
		}
	}

	private void writeThread(CompositeThread composite) throws StorageException {
		try {
			String fileName = "logs" + File.separator + composite.boardName + File.separator + composite.boardName + "_" + composite.thread.nid + ".xml";
	
			DocumentBuilderFactory dbfactory = DocumentBuilderFactory.newInstance();	DocumentBuilder builder = dbfactory.newDocumentBuilder();
			Document doc = builder.newDocument();
			Element root = doc.createElement("NDX");
			doc.appendChild(root);
	
			CompositeThreadSerializer serializer = new CompositeThreadSerializer(doc);
			root.appendChild(serializer.serialize(composite));
	
			TransformerFactory factory = TransformerFactory.newInstance();
			Transformer transformer = factory.newTransformer();
	
			transformer.setOutputProperty(OutputKeys.METHOD, "xml");
			transformer.setOutputProperty(OutputKeys.INDENT, "yes");
	
			transformer.transform(new DOMSource(doc), new StreamResult(fileName));
		} catch(TransformerException e) {
			throw new StorageException(e);
		} catch(ParserConfigurationException e) {
			throw new StorageException(e);
		}
	}
	
	/** LbVĂXbht@Cɏo */
	public synchronized int flash() throws StorageException {
		int falshedThreadNum = 0;
		
		for(String boardName: cache.keySet()) {
			final String dirName = "logs" + File.separator + boardName + File.separator;

			if(!ViewNaver.makeSureDir(dirName)) {
				throw new StorageException("fBNg'" + dirName + "'̍쐬Ɏs܂");
			}

			for(CompositeThread thread: cache.get(boardName).values()) {
				writeThread(thread);
				falshedThreadNum++;
			}

			final String finalBoardName = boardName;
			
			File[] files = new File(dirName).listFiles(new FilenameFilter() {
				public boolean accept(File parent, String fileName) {
					return fileName.startsWith(finalBoardName) && fileName.endsWith(".xml");
				}
			});

			if(files.length > ViewNaver.instance.config.capacityConfig.maxStorageThreads * 1.2) {
				System.out.println("trim storage");
				Arrays.sort(files, reverseNidComparator);

				for(int i=ViewNaver.instance.config.capacityConfig.maxStorageThreads; i<files.length; i++) {
					File file = files[i];

					if(!ViewNaver.instance.favoriteThreads.exists(boardName, getNid(file.getName()))) {
						System.out.println("delete " + file.getName());
						file.delete();
					}
				}
			}
		}
		
		cache.clear();

		return falshedThreadNum;
	}

	private TreeMap<Integer, CompositeThread> getThreadsInBoard(String boardName) {
		TreeMap<Integer, CompositeThread> threads = cache.get(boardName);
		
		if(threads == null) {
			threads = new TreeMap<Integer, CompositeThread>();
			cache.put(boardName, threads);
		}

		return threads;
	}
	
	
}
