/*
 * Copyright (c) 2006, team-naver.com
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.aibonware.viewnaver.window;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import com.aibonware.viewnaver.*;

public class ChildWindowActivator {
	private final MainWindow mainWindow;
	private Vector<Window> children = new Vector<Window>();
	private Object nowClosingChild = null;
	
	public ChildWindowActivator(MainWindow mainWindow) {
		this.mainWindow = mainWindow;

		this.mainWindow.addWindowFocusListener(new WindowAdapter() {
			@Override public void windowGainedFocus(WindowEvent e) {
				if(nowClosingChild != null) {
					for(int i=0; i<children.size(); i++) {
						Window childWindow = children.elementAt(i);

						if(childWindow != nowClosingChild && childWindow.isVisible()) {
							childWindow.toFront();
							break;
						}
					}

					nowClosingChild = null;
				} else {
					for(int i=0; i<children.size(); i++) {
						Window childWindow = children.elementAt(i);
	
						if(childWindow.isVisible()) {
							childWindow.setVisible(false);
						}
					}
				}
			}
		});
	}

	private class ChildWindowListener implements ComponentListener, WindowListener, WindowFocusListener {
		public void componentResized(ComponentEvent e) {}
		public void componentMoved(ComponentEvent e) {}
		public void windowOpened(WindowEvent e) {}
		public void windowIconified(WindowEvent e) {}
		public void windowDeiconified(WindowEvent e) {}
		public void windowDeactivated(WindowEvent e) {}
		public void windowActivated(WindowEvent e) {}
		public void windowLostFocus(WindowEvent e) {}

		private void changeShowChildWindowButton() {
			for(int i=0; i<children.size(); i++) {
				if(!children.elementAt(i).isVisible()) {
					ViewNaver.instance.mainWindow.showChildWindowButton.setSelected(false);
					return;
				}
			}
			
			ViewNaver.instance.mainWindow.showChildWindowButton.setSelected(true);
		}
		
		public void componentHidden(ComponentEvent e) {
//			ViewNaver.instance.mainWindow.showChildWindowButton.setSelected(false);
			changeShowChildWindowButton();
		}

		public void componentShown(ComponentEvent e) {
//			ViewNaver.instance.mainWindow.showChildWindowButton.setSelected(true);
			changeShowChildWindowButton();
		}

		public void windowClosed(WindowEvent e) {
			Window childWindow = (Window)e.getWindow();

			children.removeElement(childWindow);
			childWindow.removeComponentListener(this);
			childWindow.removeWindowListener(this);
			childWindow.removeWindowFocusListener(this);
		}

		public void windowGainedFocus(WindowEvent e) {
			Window childWindow = (Window)e.getWindow();

			children.removeElement(childWindow);
			children.insertElementAt(childWindow, 0);
		}

		public void windowClosing(WindowEvent e) {
			nowClosingChild = (Window)e.getWindow();
		}
	}

	ChildWindowListener childWindowListener = new ChildWindowListener();

	public void addChildWindow(Window childWindow) {
		children.addElement(childWindow);
		childWindow.addComponentListener(childWindowListener);
		childWindow.addWindowListener(childWindowListener);
		childWindow.addWindowFocusListener(childWindowListener);
	}
	
	public void activateChildren() {
		for(int i=children.size()-1; i>=0; i--) {
			Window childWindow = children.elementAt(i);

			if(!childWindow.isVisible()) {
				childWindow.setVisible(true);
			}
		}
		
		for(int i=0; i<children.size(); i++) {
			Window childWindow = children.elementAt(i);

			if(childWindow.isVisible()) {
				childWindow.toFront();
				break;
			}			
		}
	}
	
	public void setClosingChild(Object child) {
		nowClosingChild = child;
	}
}
