#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <strings.h>
#include <unistd.h>
#include <dirent.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "info.h"
#include "buflist.h"
#include "cmd.h"
#include "mem.h"
#include "dfxfunc.h"
#include "xutil.h"
#include "misc.h"
#include "debug.h"
#include "dfval.h"

struct DfxBufList{
  DfBufCommon b;
  int alloc;
  DfBuf **array;
};

static int Bl_Free(void *list);
static int Bl_Active(Widget w, void *ptr);
static int Bl_Inactive(Widget w, void *ptr);
static int Bl_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv);

static int doOpen(Widget w, struct DfxBufList *list, const char **param);

static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur);
static int fillArray(struct DfxBufList *buf);


static const DfVerb2 BL_Verb = {
  {
    Bl_Free,
    Bl_Active,
    Bl_Inactive,
    BF2_KeyPress,
    BF2_ResizeDraw,
    BF2_Draw,
  },
  BF2_DrawCaption,
  drawSingleItem,
  Bl_Process,
  SetCursor,
};

DfBuf *MakeBufList(DfBuf *parent)
{
  struct DfxBufList *buf;
  DfBuf *b;

  if(!parent){
    return NULL;
  }

  buf = MakeBuffer(sizeof(struct DfxBufList), NULL, DT_BUFLIST, &BL_Verb);
  if(buf == NULL){
    return NULL;
  }
  DfBufCommon_Init(&buf->b);
  fillArray(buf);
  Str_InitStr(&buf->b.caption, vOptions.msg[CAP_BUFFERS]);
  b = GetBuffer(buf);
  b->stat = parent->stat & ~(BS_VISIBLE | BS_ENABLE);
  b->rc = parent->rc;

  SetFocusLink(parent, b);

  AddBuffer(parent, b);

  return GetBuffer(buf);
}

static int Bl_Free(void *ptr)
{
  struct DfxBufList *list;

  list = ptr;

  bFree(list->array);
  bFree(list);

  return 0;
}


static int countup(DfBuf *cur, void *p)
{
  int *n = p;

  if(IS_ENABLE(cur) && (cur->type != DT_BUFLIST)){
    dprintf("count %d.\n", *n);
    (*n)++;
  }
  return 0;
}

static int addItemToArray(DfBuf *cur, void *p)
{
  DfBuf ***buf = p;

  if (IS_ENABLE(cur) && (cur->type != DT_BUFLIST)) {
    dprintf("store: %p %p\n", *buf, cur);
    **buf = cur;
    (*buf)++;
  }
  return 0;
}

static int fillArray(struct DfxBufList *buf)
{
  DfBuf **ptr;
#ifdef DEBUG
  int n = 0;
  int i;
#endif

  EnumBuffers(dfx->lists, countup, &buf->b.nItems);
  
  buf->array = bMalloc(sizeof(DfBuf*) * buf->b.nItems);
  if(buf->array == NULL){
    return 1;
  }

#ifdef DEBUG
  dprintf("array %p\n", buf->array);
  
  ptr = buf->array;
  for(i = 0; i < n && dprintf(" "); i++){
    dprintf("%p", ptr[i]); 
  }
  dprintf("\n");
#endif

  ptr = buf->array;
  EnumBuffers(dfx->lists, addItemToArray, &ptr);

#ifdef DEBUG
  ptr = buf->array;
  for(i = 0; i < n && dprintf(" "); i++){
    dprintf("%p", ptr[i]); 
  }
  dprintf("\n");
#endif
  BF2_CalcCursors(&buf->b);

  return 0;
}

static int Bl_Active(Widget w, void *ptr)
{
  XRectangle rc;
  struct DfxBufList *list;
  struct DfDrawInfo di;

  list = ptr;

  CmdSetTable(CMD_BUFLIST);

  calcCursorRect(&list->b, list->b.nCur, &rc);
  if(w){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
  }
  return 0;
}

static int Bl_Inactive(Widget w, void *ptr)
{
  struct DfxBufList *list;
  struct DfDrawInfo di;

  list = ptr;

  if(w && IS_VISIBLE(list)){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
  }
  return 0;
}


static int Bl_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv)
{
  struct DfxBufList *list;
  int redraw;
  int pre_cur;
  int cmd;

  cmd = c->builtin;
  redraw = 0;

  list = ptr;
  pre_cur = list->b.nCur;

  switch(cmd){
  case QUIT:
    KillBuffer(w, &list->b.b);
    break;
  case CLOSE:
    KillBuffer(w, &list->b.b);
    break;
  case CURUP:
    upCursor(w, &list->b);
    break;
  case CURDOWN:
    downCursor(w, &list->b);
    break;
  case OPEN:
    redraw = doOpen(w, list, argv);
    break;
  case PAGEUP:
    upPage(w, &list->b);
    break;
  case PAGEDOWN:
    downPage(w, &list->b);
    break;
  case TOP:
    cursorToTop(w, &list->b);
    break;
  case BOTTOM:
    cursorToBottom(w, &list->b);
    break;
  case INFO:
    ToggleInfo(w, GetFilerBuffer(GetBuffer(list)));
    SetCursor(w, &list->b, list->b.nCur);
    break;
  default:
    return CREQ_TOPARENT;
  }

  if(redraw){
    RedrawInfo(w);
    RedrawFrame(w, GetBuffer(list), &list->b.b.rc);
  }

  return CREQ_DONE;
}


static int doOpen(Widget w, struct DfxBufList *list, const char **param)
{
  int cur;
  DfBuf *buf;

  cur = list->b.nCur;

  buf = list->array[cur];

  do{
    if (!IS_ENABLE(buf)){
      break;
    }
    if (IS_VISIBLE(buf)){
      SetActiveFrame(w, buf);
    } else {
      SwitchBuffer(w, buf, GetBuffer(list));
    }
  }while(0);
  KillBuffer(w, GetBuffer(list));

  return 0;
}


static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur)
{
  long int fg;
  long int bg;
  XRectangle rc;
  int draw_line;
  struct DfxBufList *buf;
  DfBufCommon *cb;

  buf = p;;

  cb = (void*)buf->array[cur];
  dprintf("draw: %d, %p\n", cur, cb);

  draw_line = 0;
  fg = vOptions.colors[COLOR_FILE];
  bg = vOptions.colors[COLOR_BK];
  if(cur == buf->b.nCur){
    if(dfx->active == GetBuffer(buf)){
      fg = vOptions.colors[COLOR_CURFG];
      bg = vOptions.colors[COLOR_CURBG];
    }else{
      draw_line = 1;
    }
  }
  calcCursorRect(&buf->b, cur, &rc);
  XSetForeground(di->d, di->gc, bg);
  XFillRectangle(di->d, XtWindow(di->w), di->gc, rc.x, rc.y, rc.width, rc.height);

  XSetForeground(di->d, di->gc, fg);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, rc.x + 4, rc.y + vnFontBase, Str_Get(&cb->caption), Str_Length(&cb->caption));
  if(draw_line){
    XSetForeground(di->d, di->gc, vOptions.colors[COLOR_CURBG]);
    XDrawLine(di->d, XtWindow(di->w), di->gc, rc.x, rc.y + rc.height - 1, rc.x + rc.width - 1, rc.y + rc.height - 1);
  }
}

void BL_Update(DfBuf *buf, int add)
{
  DfBuf *top;
  DfBuf *cur;
  struct DfxBufList *bl;
  DfBuf **tmp;

  if (buf->type == DT_BUFLIST) {
    return;
  }

  top = dfx->lists;
  cur = top;
  do{
    cur = CONTENT_OF(DfBuf, list, ListNext(&cur->list));
    if (cur->type == DT_BUFLIST) {
      bl = (struct DfxBufList *)cur;
      if(add){
	tmp = bReAlloc(bl->array, sizeof(DfBuf*) * (bl->b.nItems + 1));
	if(tmp != NULL){
	  tmp[bl->b.nItems] = buf;
	  bl->b.nItems++;
	  bl->array = tmp;
	}
      }else{
	int i;
	for(i = 0; i < bl->b.nItems; i++){
	  if(bl->array[i] == buf){
	    bl->b.nItems--;
	    for(; i < bl->b.nItems; i++){
	      bl->array[i] = bl->array[i + 1];
	    }
	    break;
	  }
	}
      }
      if(IS_VISIBLE(cur)){
	RedrawFrame(dfx->w, cur, &cur->rc);
      }
    }
  }while(top != cur);
}


