#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

#include "config.h"
#include "dftype.h"
#include "str.h"
#include "list.h"
#include "buffer.h"
#include "filer.h"
#include "status.h"
#include "cmdlist.h"
#include "cmdinput.h"
#include "dialog.h"
#include "task.h"
#include "cmd.h"
#include "misc.h"
#include "mem.h"
#include "extmacro.h"
#include "dfxfunc.h"
#include "dfval.h"
#include "debug.h"

static int chownInvokeFile(DfTask *t);
static int chownSubDirectory(DfTask *t);

static uid_t get_input_uid(DfTask *t);
static gid_t get_input_gid(DfTask *t);

task_vtable vtable_chown = {
  commonInvoke,
  commonNotify,
  commonDone,
  commonInit,
  chownInvokeFile,
  chownSubDirectory
};

task_vtable vtable_chgrp = {
  commonInvoke,
  commonNotify,
  commonDone,
  commonInit,
  chownInvokeFile,
  chownSubDirectory
};


static int chownInvokeFile(DfTask *t)
{
  int len;
  char *name;
  int s_idx;
  DfStr src;
  int ret;
  struct stat st;
  uid_t owner;
  gid_t gid;

  name = t->now_proc;

  switch(t->state){
  case CMD_SKIP:
    len = strlen(name) + 1;
    name += len;
    t->now_proc = name;
    break;

  default:
    break;
  }

  if(!*name){
    return 0;
  }

  owner = -1;
  gid = -1;
  if(t->op == DO_CHOWN){
    owner = get_input_uid(t);
  }else{
    gid = get_input_gid(t);
  }

  setupPathes(t, &src, NULL, 1);
  s_idx = Str_Length(&src);

  while(*name){
    Str_Overwrite(&src, s_idx, name);
    /* check already exist */

    lstat(Str_Get(&src), &st);
    if(S_ISDIR(st.st_mode)){
      if(IsDots(name)){
	goto NEXT;
      }
      if(initSubDir(t, Str_Get(&src), NULL)){
	t->err = 0;
	t->state = STATE_NONE;
	ret = 0;
      }else{
	t->state = STATE_FAIL;
	ret = 1;
      }

      Str_Free(&src, 0);
      return ret;
    }
    ret = chown(Str_Get(&src), owner, gid);

    if(ret != 0){
      t->err = ret;
      t->state = STATE_FAIL;

      Str_Free(&src, 0);
      return 1;
    }
    t->update = 1;

  NEXT:
    len = strlen(name) + 1;
    name += len;
    t->now_proc = name;
  }

  Str_Free(&src, 0);

  return 0;
}

static int chownSubDirectory(DfTask *t)
{
  DfSubDir *sd;
  struct stat st;
  int ret;
  int len;
  uid_t owner;
  gid_t gid;

  sd = t->sub;

  switch(t->state){
  case CMD_RETRY:
    if(sd->de != NULL){
      break;
    }
    /*THRU*/
  default:
    sd->de = readdir(sd->dir);
  }

  owner = -1;
  gid = -1;
  if(t->op == DO_CHOWN){
    owner = get_input_uid(t);
  }else{
    gid = get_input_gid(t);
  }

  for(;sd->de; sd->de = readdir(sd->dir)){
    Str_Overwrite(sd->src, sd->s_idx, sd->de->d_name);

    lstat(Str_Get(sd->src), &st);
    if(S_ISDIR(st.st_mode)){
      if(IsDots(sd->de->d_name)){
	continue;
      }
      if(initSubDir(t, NULL, NULL)){
	t->err = 0;
	t->state = STATE_NONE;
      }else{
	t->state = STATE_FAIL;
      }
      return 1;
    }

    ret = chown(Str_Get(sd->src), owner, gid);

    /* next */
    if(ret != 0){
      t->err = ret;
      t->state = STATE_FAIL;
      return 1;
    }
    t->update = 1;
  }
  closedir(sd->dir);
  Str_SetLength(sd->src, sd->s_idx);

  /* detach */
  t->sub = sd->next;

  if(t->sub == NULL){
    Str_Free(sd->src, 0);
    len = strlen(t->now_proc) + 1;
    t->now_proc += len;
  }

  bFree(sd);
  return 0;
}

static uid_t get_input_uid(DfTask *t)
{
  struct passwd *pw;
  uid_t owner;

  pw = getpwnam(Str_Get(&t->input));
  if(pw != NULL){
    owner = pw->pw_uid;
  }else{
    if(!IsAllNum(Str_Get(&t->input))){
      return 0;
    }
    owner = atoi(Str_Get(&t->input));
  }

  return owner;
}


static gid_t get_input_gid(DfTask *t)
{
  struct group *gr;
  gid_t gid;

  gr = getgrnam(Str_Get(&t->input));
  if(gr != NULL){
    gid = gr->gr_gid;
  }else{
    if(!IsAllNum(Str_Get(&t->input))){
      return 0;
    }
    gid = atoi(Str_Get(&t->input));
  }

  return gid;
}

