#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "dir.h"
#include "view.h"
#include "info.h"
#include "status.h"
#include "cmdlist.h"
#include "cmdinput.h"
#include "dialog.h"
#include "task.h"
#include "cmd.h"
#include "xutil.h"
#include "misc.h"
#include "mem.h"
#include "extmacro.h"
#include "dfxfunc.h"
#include "debug.h"
#include "dfval.h"

extern DfxAppList *vAppsList;

int MakeArgList(DfStr *s, const char **argv);

static int FL_Free(void *list);
static int FL_Active(Widget w, void *ptr);
static int FL_Inactive(Widget w, void *ptr);
static int FL_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv);
static int FL_ResizeDraw(Widget w, void *ptr);

static int FL_ActiveScan(Widget w, void *ptr);
static int FL_InactiveScan(Widget w, void *ptr);
static int FL_DrawScan(void *ptr, struct DfDrawInfo *di);
static int FL_ResizeScan(Widget w, void *ptr);

static int FL_DrawEmpty(void *ptr, struct DfDrawInfo *di);

static int doOpen(Widget w, DfFileList *list, const char **param);
static void *NewBuffer(Widget w, DfBuf *b);
static int openDir(Widget w, DfFileList *list, const char **param, int *redraw);
static int openViewer(Widget w, DfFileList *list, const char **param);

int chDir(Widget w, DfFileList *list, const char *new_dir);
static void markFile(Widget w, DfFileList *list, int n, int redraw);
static int gotoParent(Widget w, DfFileList *list);

static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur);
static void drawSingleItemScan(void *p, struct DfDrawInfo *di, int cur);

static int findFileName(DfFileList *list, const char *target, int def);

static void renumberBuffer(int del);
static int setSort(Widget w, DfFileList  *list, sort_type sort_type);

static void calcDialogRect(DfFileList *l, XRectangle *rc);

static int doCopy(Widget w, DfFileList  *list, const char **param);
static int doMove(Widget w, DfFileList  *list, const char **param);
static int doLink(Widget w, DfFileList  *list, const char **param);
static int doSymLink(Widget w, DfFileList  *list, const char **param);
static int doDelete(Widget w, DfFileList  *list, const char **param);
static int doChdir(Widget w, DfFileList  *list, const char **param);
static int doIncSearch(Widget w, DfFileList  *list, const char **param);
static int doMultimark(Widget w, DfFileList *list, const char **param);
static int doClearmark(Widget w, DfFileList *list);
static int doReversemark(Widget w, DfFileList  *list);
static int doMakeFile(Widget w, DfFileList  *list, const char **param);
static int doMakeDir(Widget w, DfFileList  *list, const char **param);
static int doRename(Widget w, DfFileList  *list, const char **param);
static int doFilter(Widget w, DfFileList  *list, const char **param);
static int doIFilter(Widget w, DfFileList  *list, const char **param);
static int doChmod(Widget w, DfFileList  *list, const char **param);
static int doChown(Widget w, DfFileList  *list, const char **param);
static int doChgrp(Widget w, DfFileList  *list, const char **param);
static int doQuit(Widget w, DfFileList  *list, const char **param);
static int doExecute(Widget w, DfFileList  *list, const char **param);
static void toggleDotfiles(Widget w, DfFileList *l);

static int mkfileInit(DfTask *t);
static int mkfileFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st);
static int mkdirFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st);

static task_result renameNotify(DfTask *t);
static int renameFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st);
static task_result chdirInvoke(DfTask *t);
static task_result isearchNotify(DfTask *t);
static task_result isearchDone(DfTask *t);
static task_result multimarkInvoke(DfTask *t);
static task_result execInvoke(DfTask *t);


static const DfVerb2 FL_Verb = {
  {
    FL_Free,
    FL_Active,
    FL_Inactive,
    BF2_KeyPress,
    FL_ResizeDraw,
    BF2_Draw
  },
  BF2_DrawCaption,
  drawSingleItem,
  FL_Process,
  SetCursor,
};

static const DfVerb2 FL_ScanVerb = {
  {
    FL_Free,
    FL_ActiveScan,
    FL_InactiveScan,
    BF2_KeyPress,
    FL_ResizeScan,
    FL_DrawScan
  },
  BF2_DrawCaption,
  drawSingleItemScan,
  BF2_ProcessBusy,
  SetCursor,
};

static const DfVerb2 FL_EmptyVerb = {
  {
    FL_Free,
    FL_Active,
    FL_Inactive,
    BF2_KeyPress,
    FL_ResizeDraw,
    FL_DrawEmpty
  },
  BF2_DrawCaption,
  drawSingleItemScan,
  FL_Process,
  SetCursor,
};

static const task_vtable vtable_chdir = {
  chdirInvoke,
  NULL,
  commonDone,
};

static const task_vtable vtable_search = {
  NULL,
  isearchNotify,
  isearchDone,
};

static const task_vtable vtable_mark = {
  multimarkInvoke,
  NULL,
  commonDone,
};

static const task_vtable vtable_mkfile = {
  commonInvoke,
  NULL,
  commonDone,
  mkfileInit,
  commonProcSelectedFile,
  NULL,
  mkfileFileOp,
};

static const task_vtable vtable_mkdir = {
  commonInvoke,
  NULL,
  commonDone,
  mkfileInit,
  commonProcSelectedFile,
  NULL,
  mkdirFileOp
};

static const task_vtable vtable_exec = {
  execInvoke,
  NULL,
  commonDone,
};

static const task_vtable vtable_exec2 = {
  execInvoke,
  NULL,
  commonDone,
};

static const task_vtable vtable_rename = {
  commonInvoke,
  renameNotify,
  commonDone,
  commonInit,
  commonProcSelectedFile,
  commonProcSubDirectories,
  renameFileOp
};


/* filer */
df_stat SetFilerState(Widget w, DfFileList *list, df_stat s)
{
  df_stat r = list->stat;

  if(s == r){
    return s;
  }

  switch(s){
  case DS_WAIT_DIR:
    dprintf("filer is waiting on readdir.\n");
    list->b.b.v = (DfVerb*)&FL_ScanVerb;
    list->b.nStart = 0;
    list->b.nOffset = 0;
    break;
  default:
    dprintf("set state to filer %d.\n", s);
    if(list->b.nItems == 0){
      list->b.b.v = (DfVerb*)&FL_EmptyVerb;
    }else{
      list->b.b.v = (DfVerb*)&FL_Verb;
    }
    CalcBufferOffset(w, (DfBufCommon *)list, list->b.nCur);
  }
  list->stat = s;


  if(IS_VISIBLE(list)){
    RedrawFrame(w, GetBuffer(list), &GetBuffer(list)->rc);
  }

  return r;
}

void FL_Init(DfFileList *list)
{
  memset(list, 0, sizeof(*list));
  ListClear(&list->b.b.list);
  ListClear(&list->b.b.focus);
  list->b.b.type = DT_FILER;
  list->b.b.v = (DfVerb*)&FL_ScanVerb;
}

static int FL_Free(void *ptr)
{
  DfFileList *list;

  list = ptr;

  renumberBuffer(list->n);

  BufComon_Free(&list->b);

  bFree(list->files);
  bFree(list->raw);
  bFree(list->pszFilenames);
  bFree(list);
  dfx->nFiler--;

  return 0;
}

static int FL_Active(Widget w, void *ptr)
{
  XRectangle rc;
  DfFileList *list;
  struct DfDrawInfo di;

  list = ptr;

  CmdSetTable(CMD_FILER);
  if(list->b.nItems == 0){
    return 0;
  }
  calcCursorRect(&list->b, list->b.nCur, &rc);
  if(w){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
    SetInfo(w, list);
  }
  return 0;
}

static int FL_Inactive(Widget w, void *ptr)
{
  DfFileList *list;
  struct DfDrawInfo di;

  list = ptr;
  if(list->b.nItems == 0){
    return 0;
  }

  if(w && IS_VISIBLE(list)){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(list));
    drawSingleItem(list, &di, list->b.nCur);
  }
  return 0;
}

static int FL_ResizeDraw(Widget w, void *ptr)
{
  CalcColumnWidth(ptr);
  BF2_ResizeDraw(w, ptr);

  return 0;
}

static int FL_ActiveScan(Widget w, void *ptr)
{
  CmdSetTable(CMD_FILER);
  return 0;
}

static int FL_InactiveScan(Widget w, void *ptr)
{
  CmdSetTable(CMD_FILER);
  return 0;
}

static int FL_ResizeScan(Widget w, void *ptr)
{
  BF2_ResizeDraw(w, ptr);

  return 0;
}

static void drawSingleItemDI(Widget w, DfFileList *l, int n)
{
  struct DfDrawInfo di;
  XRectangle rc;

  calcCursorRect(&l->b, n, &rc);
  di.w = w;
  di.d = XtDisplay(w);
  di.gc = GetBufferGC(w, GetBuffer(l));
  di.clip = rc;
  SetClipRect(di.d, di.gc, &rc);
  XClearArea(XtDisplay(w), XtWindow(w),
	     rc.x, rc.y, rc.width, rc.height, False);

  drawSingleItem(l, &di, l->b.nCur);
}

static int FL_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv)
{
  DfFileList *list;
  int pre_cur;
  DfBuf *b;
  int cmd;

  cmd = c->builtin;

  list = ptr;

  pre_cur = list->b.nCur;
  switch(cmd){
  case OPEN:
  case COPY:
  case MOVE:
  case DELETE:
  case RENAME:
  case HORIZ_LEFT:
  case HORIZ_RIGHT:
  case CHMOD:
  case CHOWN:
  case CHGRP:
  case CURUP:
  case CURDOWN:
  case MARK:
  case MULTIMARK:
  case CLEARMARK:
  case REVMARK:
  case PAGEUP:
  case PAGEDOWN:
  case TOP:
  case BOTTOM:
  case ISEARCH:
  case VIEW:
    if(list->b.nItems == 0){
      St_SetMsg(w, vOptions.msg[STM_NOFILES]);
      /*      return 0; */
      return CREQ_DONE;
    }
  }
  
  switch(cmd){
  case COPY:
    doCopy(w, list, argv);
    break;
  case MOVE:
    doMove(w, list, argv);
    break;
  case LINK:
    doLink(w, list, argv);
    break;
  case SYMLINK:
    doSymLink(w, list, argv);
    break;
  case DELETE:
    doDelete(w, list, argv);
    break;
  case MKFILE:
    doMakeFile(w, list, argv);
    break;
  case MKDIR:
    doMakeDir(w, list, argv);
    break;
  case RENAME:
    doRename(w, list, argv);
    break;
  case HORIZ_LEFT:
    if(vnHoriz){
      vnHoriz--;
      drawSingleItemDI(w, list, list->b.nCur);
    }
    break;
  case HORIZ_RIGHT:
    if(vnHoriz + 1 < strlen(list->files[list->b.nCur]->name)){
      vnHoriz++;
      drawSingleItemDI(w, list, list->b.nCur);
    }
    break;
  case FILTER:
    doFilter(w, list, argv);
    break;
  case IFILTER:
    doIFilter(w, list, argv);
    break;
  case CHMOD:
    doChmod(w, list, argv);
    break;
  case CHOWN:
    doChown(w, list, argv);
    break;
  case CHGRP:
    doChgrp(w, list, argv);
    break;
  case QUIT:
    doQuit(w, list, argv);
    break;

  case SPLIT_H:
    b = SplitWindow(w, &list->b.b, SPLIT_HRIZ, NewBuffer);
    break;
  case SPLIT_V:
    b = SplitWindow(w, &list->b.b, SPLIT_VERT, NewBuffer);
    break;
  case CLOSE:
    KillFrame(w, &list->b.b);
    UpdateFrames(w, GetBuffer(list));
    break;
  case KILLOTHER:
    KillOtherFrames(w, GetBuffer(list));
    break;
  case NEWBUFFER:
    b = NewBuffer(w, &list->b.b);
    SwitchBuffer(w, b, GetBuffer(list));
    break;
  case CURUP:
    upCursor(w, &list->b);
    break;
  case CURDOWN:
    downCursor(w, &list->b);
    break;
  case GOTO:
    doChdir(w, list, argv);
    break;
  case OPEN:
    doOpen(w, list, argv);
    break;
  case MARK:
    markFile(w, list, list->b.nCur, 1);
    downCursor(w, &list->b);
    break;
  case MULTIMARK:
    doMultimark(w, list, argv);
    break;
  case CLEARMARK:
    doClearmark(w, list);
    break;
  case REVMARK:
    doReversemark(w, list);
    break;
  case PAGEUP:
    upPage(w, &list->b);
    break;
  case PAGEDOWN:
    downPage(w, &list->b);
    break;
  case TOP:
    cursorToTop(w, &list->b);
    break;
  case BOTTOM:
    cursorToBottom(w, &list->b);
    break;
  case GOTOPARENT:
    gotoParent(w, list);
    break;
  case ISEARCH:
    doIncSearch(w, list, argv);
    break;
  case RELOAD:
    reload(w, list, NULL);
    break;
  case INFO:
    ToggleInfo(w, list);
    SetCursor(w, &list->b, list->b.nCur);
    break;
  case SORTNAME:
    setSort(w, list, SORT_NAME);
    break;
  case SORTSIZE:
    setSort(w, list, SORT_SIZE);
    break;
  case SORTTIME:
    setSort(w, list, SORT_TIME);
    break;
  case SORTEXT:
    setSort(w, list, SORT_EXT);
    break;
  case SORTATTR:
    setSort(w, list, SORT_ATTR);
    break;
  case EXECUTE:
    doExecute(w, list, argv);
    break;
  case SHOWDOTS:
    toggleDotfiles(w, list);
    RedrawInfo(w);
    RedrawFrame(w, GetBuffer(list), &list->b.b.rc);
    break;

  case VIEW:
    openViewer(w, list, argv);
    break;

  default:
    return CREQ_IGNORE;
  }

  return CREQ_DONE;
}


static void callbackOpenexec(DfTask *t)
{
  struct parse_extention *pe = t->extend;

  DfxExec(t->w, t->owner, Str_Get(&t->cwd), Str_Get(&pe->cmd), NULL);
  Str_Free(&pe->cmd, 0);
}


static int doOpen(Widget w, DfFileList *list, const char **param)
{
  int redraw;
  int cont;
  int cur;
  int n;
  DfFile *f;
  DfStr c;
  const char *p;

  redraw = 0;

  cont = openDir(w, list, param, &redraw);
  if(!cont){
    return redraw;
  }

  if(vAppsList && vAppsList->num){
    cur = list->b.nCur;
    f = list->files[cur];

    dprintf("file: %s\n", f->name);
    for(n = 0; n < vAppsList->num; n++){
      dprintf("pat-%d: %s\n", n, vAppsList->list.idx[n]);
      if(IsMatch(vAppsList->list.idx[n], f->name) == MATCH){
	p = skipPsz(Str_Get(&vAppsList->list.cmds), n);
	dprintf("match cmd : %s\n", p);
	Str_InitStr(&c, p);
	ParseMacro2(w, &c, GetBuffer(list), callbackOpenexec, NULL);
	return redraw;
      }
    }
  }
  cont = openViewer(w, list, param);

  return redraw;
}

static int openDir(Widget w, DfFileList *list, const char **param, int *redraw)
{
  DfStr s;
  int cur;
  int attr;
  DfFile *f;
  struct stat st;

  cur = list->b.nCur;
  f = list->files[cur];
  attr = f->attr;

  if(S_ISLNK(attr)){
    Str_InitStr(&s, list->cwd);
    Str_Add(&s, f->name);
    stat(Str_Get(&s), &st);
    Str_Free(&s, 0);
    attr = st.st_mode;
  }

  if(!S_ISDIR(attr)){
    return 1;
  }

  if(strcmp(f->name, "..") == 0){
    *redraw = gotoParent(w, list);
    return 0;
  }

  if(!chDir(w, list, f->name)){
    *redraw = 1;
    return 0;
  }

  return 0;
}

static int openViewer(Widget w, DfFileList *list, const char **param)
{
  DfStr s;
  int cur;
  DfFile *f;
  DfViewer *v;
  int r;

  v = Vw_Create(GetBuffer(list));
  if(!v){
    return 0;
  }

  cur = list->b.nCur;
  f = list->files[cur];

  Str_InitStr(&s, list->cwd);
  Str_Add(&s, f->name);

  r = Vw_Open(w, v, Str_Get(&s));
  switch(r){
  case 0:
    break;
  default:
    St_SetMsg(w, strerror(r));
  }
  Str_Free(&s, 0);

  if(r != 0){
    St_SetMsg(w, vOptions.msg[STM_CANTVIEW]);
    /* instead of KillBuffer(). not added viewer buffer to buffer-lists yet.*/
    GetBuffer(v)->v->free(GetBuffer(v));
    return 0;
  }

  /* add to lists */
  AddBuffer(GetBuffer(list), GetBuffer(v));

  SetModalBuffer(w, GetBuffer(list), GetBuffer(v));

  return 0;
}

struct chdir_cb_data{
  Widget w;
  DfFileList *l;
  DfStr path;
};
static void chdir_callback(void *ptr)
{
  struct chdir_cb_data *cb = ptr;
  DfFileList *list = cb->l;
  int ret;

  St_SetMsg(cb->w, vOptions.msg[STM_CHDIR]);

  ret = 0;
  if(list->b.nItems != 0){
    SortFileList(list);
    if(list->mode & SHOW_DOTS){
      ret = findFileName(list, "..", 0);
    }
  }
  CalcBufferOffset(cb->w, &list->b, ret);
  Str_Free(&cb->path, 0);
  bFree(ptr);
}

int chDir(Widget w, DfFileList *list, const char *new_dir)
{
  struct chdir_cb_data *cb;

  cb = bMalloc(sizeof(*cb));
  cb->w = w;
  cb->l = list;
  Str_InitStr(&cb->path, new_dir);
  Str_RegPath(&cb->path, 0, 0);

  GetFiles(list, w, Str_Get(&cb->path), chdir_callback, cb, 0);

  return 0;
}


static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur)
{
  DfFileList *list;
  DfFile *files;
  long int fg;
  long int bg;
  XRectangle rc;
  int draw_line;
  char *pszName;
  int len;
#if DRAW_SIZE
  char buf[32];
  int cx;
  int x;
#endif

  list = p;
  files = list->files[cur];

  draw_line = 0;
  fg = files->color;
  bg = vOptions.colors[COLOR_BK];
  if(files->nSelect){
    fg = vOptions.colors[COLOR_SELFG];
    bg = vOptions.colors[COLOR_SELBG];
  }
  if(cur == list->b.nCur){
    if(dfx->active == GetBuffer(list)){
      fg = vOptions.colors[COLOR_CURFG];
      bg = vOptions.colors[COLOR_CURBG];
    }else{
      draw_line = 1;
    }
  }
  calcCursorRect(&list->b, cur, &rc);
  XSetForeground(di->d, di->gc, bg);
  XFillRectangle(di->d, XtWindow(di->w), di->gc, rc.x, rc.y, rc.width, rc.height);

  if(files->nSelect){
    XSetForeground(di->d, di->gc, vOptions.colors[COLOR_MARK]);
    XFillRectangle(di->d, XtWindow(di->w), di->gc, rc.x, rc.y, 4, rc.height);
  }
  XSetForeground(di->d, di->gc, fg);

  pszName = files->name;
  len = strlen(pszName);
  if(cur == list->b.nCur && vnHoriz){
    if(vnHoriz < len){
      pszName += vnHoriz;
      len -= vnHoriz;
    }else{
      pszName = pszName + len - 1;
      len = 1;
    }
  }
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, rc.x + 4, rc.y + vnFontBase, pszName, len);

#if DRAW_SIZE
  /* file size */
  cx = list->cx[FILECX_SIZE] + list->cx[FILECX_DATE] + vnFontHeight;
  x = rc.x + rc.width - cx;
  if(x < rc.x + 4 + list->cx[FILECX_NAME] + vnFontHeight){
    x = rc.x + 4 + list->cx[FILECX_NAME] + vnFontHeight;
  }
  if(!S_ISDIR(files->attr)){
    len = StrSize(buf, sizeof(buf), files->size);
    XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, x + list->cx[FILECX_SIZE] - files->cx[FILECX_SIZE], rc.y + vnFontBase, buf, len);
  }
 
  /* m time */
  x += (list->cx[FILECX_SIZE] + vnFontHeight);
  len = StrDate(buf, sizeof(buf), &files->mtime);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, x, rc.y + vnFontBase, buf, len);
#endif

  if(draw_line){
    XSetForeground(di->d, di->gc, vOptions.colors[COLOR_CURBG]);
    XDrawLine(di->d, XtWindow(di->w), di->gc, rc.x, rc.y + rc.height - 1, rc.x + rc.width - 1, rc.y + rc.height - 1);
  }
}

static int FL_DrawScan(void *ptr, struct DfDrawInfo *di)
{
  DfBufCommon *b = ptr;
  DfVerb2 *v = (DfVerb2*)b->b.v;;
  int x;
  int y;
  const char *msg;

  (*v->draw_caption)(di->w, b, di->d, di->gc);

  x = b->b.rc.x;
  y = b->b.rc.y + vnFontBase + vnFontHeight;
  msg = vOptions.msg[STM_SCANNING];
  XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, x, y, msg, strlen(msg));

  return 0;
}

static int FL_DrawEmpty(void *ptr, struct DfDrawInfo *di)
{
  DfBufCommon *b;
  DfVerb2 *v;
  int x;
  int y;
  const char *msg;

  b = ptr;
  v = (DfVerb2*)b->b.v;

  (*v->draw_caption)(di->w, b, di->d, di->gc);

  x = b->b.rc.x;
  y = b->b.rc.y + vnFontBase + vnFontHeight;
  msg = vOptions.msg[STM_EMPTY];
  XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, x, y, msg, strlen(msg));

  return 0;
}


static void drawSingleItemScan(void *p, struct DfDrawInfo *di, int cur)
{
  return;
}


static void makeCaption(DfFileList *l, const char *cwd)
{
  char *p;
  int len;
  int idx;

  len = strlen(cwd) + 32;

  p = bMalloc(len);
  idx = sprintf(p, "%d:", l->n);
  strcpy(p + idx, cwd);
  Str_InitPsz(&l->b.caption, p);
  l->cwd = p + idx;
}

struct newbuffer_cb_data{
  Widget w;
  DfFileList *cur;
  DfFileList *new;
};
static void newbuffer_callback(void *ptr)
{
  struct newbuffer_cb_data *cb = ptr;
  DfFileList *b = cb->new;

  SortFileList(b);

  bFree(ptr);
}

static void *NewBuffer(Widget w, DfBuf *b)
{
  DfFileList *list;
  DfFileList *n;
  struct newbuffer_cb_data *nb;

  list = (DfFileList *)b;

  n = bMalloc(sizeof(*n));
  FL_Init(n);
  n->n = dfx->nFiler + 1;
  n->b.nCur = list->b.nCur;
  n->b.nStart = list->b.nStart;
  makeCaption(n, list->cwd);
  n->b.b.rc = list->b.b.rc;
  n->b.b.stat |= BS_ENABLE;
  n->sort = list->sort;

  nb = bMalloc(sizeof(*b));
  nb->w = w;
  nb->cur = list;
  nb->new = n;

  AddBuffer(b, &n->b.b);
  dfx->nFiler++;
  GetFiles(n, w, NULL, newbuffer_callback, nb, 0);

  return n;
}

static void markFile(Widget w, DfFileList *list, int n, int redraw)
{
  DfFile *f;
  int i;
  int sel;

  if(list->b.nItems <= n){
    return;
  }

  f = list->files[n];
  if(IsDots(f->name)){
    return;
  }

  if(f->nSelect){
    sel = f->nSelect;
    f->nSelect = 0;
    f = list->raw;
    for(i = 0; i < list->nHasFiles; i++){
      if(sel < f->nSelect){
	f->nSelect--;
      }
      f++;
    }
    list->nSelects--;
  }else{
    list->nSelects++;
    f->nSelect = list->nSelects;
  }
  if(redraw){
    drawSingleItemDI(w, list, n);
  }
}

struct gotoparent_cb_data{
  Widget w;
  char *org;
  DfFileList *l;
};

static void gotoparent_callback(void *ptr)
{
  struct gotoparent_cb_data *cb;
  int cur = 0;;

  cb = ptr;
  if(cb->l->b.nItems != 0){
    SortFileList(cb->l);
    cur = findFileName(cb->l, cb->org, 0);
  }
  CalcBufferOffset(cb->w, &cb->l->b, cur);
  bFree(cb->org);
  bFree(cb);
}

static int gotoParent(Widget w, DfFileList *list)
{
  char *p;
  struct gotoparent_cb_data *cb;
  char *org;
  const char *cwd;
  int len;

  cwd = list->cwd;
  if(cwd[0] == '/' && cwd[1] == '\0'){
    /*  /  */
    return 0;
  }

  p = FindLastChar(cwd, '/');
  if(!p[0]){
    return 0;
  }

  if(p[1] == '\0'){
    p[0] = '\0';
    org = FindLastChar(cwd, '/');
    if(org[0]){
      p = org;
    }
  }
  cb = bMalloc(sizeof(*cb));
  cb->w = w;
  len = strlen(p);
  cb->org = NULL;
  if(1 < len){
    cb->org = bMalloc(len);
    memcpy(cb->org, p + 1, len - 1);
    cb->org[len - 1] = '\0';
  }
  cb->l = list;

  p[1] = '\0';

  Str_SetLength(&list->b.caption, strlen(Str_Get(&list->b.caption)));

  St_SetMsg(w, vOptions.msg[STM_GOTO_PARENT]);

  if(!GetFiles(list, w, NULL, gotoparent_callback, cb, 0)){
    return 0;
  }

  return 1;
}

struct reload_cb_data{
  Widget w;
  DfFileList *l;
  DfStr o;
  int cur;
  int set_cursor;
};

static void reload_callback(void *ptr)
{
  struct reload_cb_data *cb = ptr;
  DfFileList *l;
  int cur = 0;

  l = cb->l;

  dprintf("reload: new caption [%s].\n", Str_Get(&l->b.caption));
  dprintf("reload: cwd is %s.\n", l->cwd);

  if(l->b.nItems != 0){
    SortFileList(l);
    if(cb->set_cursor){
      cur = findFileName(l, Str_Get(&cb->o), cb->cur);
      Str_Free(&cb->o, 0);
    }
  }
  CalcBufferOffset(cb->w, &l->b, cur);
  bFree(cb);
}

int reload(Widget w, DfFileList *l, const char *cursor_pos)
{
  struct reload_cb_data *cb;
  int cur;

  cb = bMalloc(sizeof(*cb));
  cb->w = w;
  cb->l = l;
  cb->cur = 0;
  cb->set_cursor = 0;
  if (cursor_pos){
    Str_InitStr(&cb->o, cursor_pos);
    cb->cur = l->b.nCur;
    cb->set_cursor = 1;
  } else {
    if(l->b.nItems){
      cur = l->b.nCur;
      cb->cur = l->b.nCur;
      Str_InitStr(&cb->o, l->files[cur]->name);
      cb->set_cursor = 1;
    }
  }
  GetFiles(l, w, NULL, reload_callback, cb, 0);

  return 0;
}


static int findFileName(DfFileList *list, const char *target, int def)
{
  DfFile **f;
  int n;

  if(!target){
    return def;
  }

  f = list->files;
  n = list->b.nItems;
  while(n){
    if(strcmp(f[0]->name, target) == 0){
      return list->b.nItems - n;
    }
    f++;
    n--;
  }
  return def;
}

DfFileList *GetNextFileBuf(DfFileList *list)
{
  DfBuf *b;

  b = GetBuffer(list);
  do{
    b = NextBuffer(b);
  }while(b->type != DT_FILER);

  return (DfFileList*)b;
}

DfFileList *GetPrevFileBuf(DfFileList *list)
{
  DfBuf *b;

  b = GetBuffer(list);
  do{
    b = PrevBuffer(b);
  }while(b->type != DT_FILER);

  return (DfFileList*)b;
}

static void renumberBuffer(int del)
{
  DfFileList *list;
  DfFileList *p;
  char *cap;
  int i;

  list = (DfFileList*)dfx->lists;
  p = list;
  do{
    if(p->b.b.type == DT_FILER){
      if(del < p->n){
	p->n--;
	cap = Str_Get(&p->b.caption);
	i = sprintf(cap, "%d", p->n);
	if(cap + i + 1 != p->cwd){
	  strcpy(cap + i + 1, p->cwd);
	}
	cap[i] = ':';
      }
    }
    p = NextBuffer(p);
  }while(list != p);
}

static int setSort(Widget w, DfFileList  *list, sort_type sort_type)
{
  int nCur;
  char *pszCur;
  XRectangle rc;

  if(list->b.nItems == 0){
    return 0;
  }

  nCur = list->b.nCur;
  pszCur = list->files[nCur]->name;
  list->sort = sort_type;
  SortFileList(list);

  nCur = findFileName(list, pszCur, nCur);
  SetCursor(w, (DfBufCommon*)list, nCur);
  GetClientRect(GetBuffer(list), &rc);
  RedrawFrame(w, GetBuffer(list), &rc);

  return 1;
}

int MakeSelectList(DfStr *s, DfFileList *list)
{
  DfFile **f;
  DfFile **selects;
  int n;
  int sel;

  if(list->b.nItems == 0){
    Str_Init(s);
    return 0;
  }

  f = list->files;
  sel = list->nSelects;
  if(sel == 0){
    Str_InitStr(s, f[list->b.nCur]->name);
    Str_AddChar(s, '\0');
    Str_Shrink(s);
    return 1;
  }

  selects = bMalloc(sizeof(*selects) * sel);
  n = list->b.nItems;
  while(n){
    if(f[0]->nSelect){
      selects[f[0]->nSelect - 1] = *f;
      if(--sel == 0){
	/* all listed */
	break;
      }
    }
    f++;
    n--;
  }

  sel = list->nSelects;
  Str_Init(s);

  for(n = 0; n < sel; n++){
    Str_AddNUL(s, selects[n]->name);
  }
  Str_AddChar(s, '\0');
  Str_Shrink(s);

  bFree(selects);

  return list->nSelects;
}

int MakeArgList(DfStr *s, const char **argv)
{
  Str_Init(s);
  while(*argv){
    Str_AddNUL(s, *argv);
    argv++;
  }
  Str_AddChar(s, '\0');
  Str_Shrink(s);

  return 0;
}

const char *MakeArgList2(DfStr *s, const char **argv)
{
  const char *p = NULL;;

  Str_Init(s);
  p = *argv;
  argv++;
  Str_AddNUL(s, p);

  p = *argv;
  argv++;
  while(*argv){
    Str_AddNUL(s, p);
    p = *argv;
    argv++;
  }

  Str_AddChar(s, '\0');
  Str_Shrink(s);

  return p;
}

static int doCopy(Widget w, DfFileList *list, const char **param)
{
  DfStr s;
  DfTask *t;
  DfInputDialog *d;
  DfFileList *b;
  const char *dst;

  t = MakeTask(w, GetBuffer(list), DO_COPY, DLG_COPY_FAIL, 0);
  if(!t){
    return 0;
  }

  dst = NULL;
  if(param && *param){
    dst = MakeArgList2(&s, param);
    if(dst && *dst){
      Str_InitStr(&t->dst, dst);
    }else{
      dst = NULL;
    }
  }else{
    MakeSelectList(&s, list);
  }

  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, b->cwd);
  t->flag = DLGF_UPDATEDST | DLGF_USESRC | DLGF_USEDST | DLGF_SUBDIR;
  t->files = (Str_Get(&s));
  t->now_proc = t->files;

  t->v = &vtable_copy;
  if(dst){
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, b->cwd, copyKeyProc);
  ShowInputDialog(d, CAP_COPY);

  return 0;
}

static int doLink(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  DfTask *t;
  DfInputDialog *d;
  DfFileList *b;
  const char *dst;

  t = MakeTask(w, GetBuffer(list), DO_LINK, DLG_COPY_FAIL, 0);
  if(!t){
    return 0;
  }
  dst = NULL;
  if(param && *param){
    dst = MakeArgList2(&s, param);
    if(dst && *dst){
      Str_InitStr(&t->dst, dst);
    }else{
      dst = NULL;
    }
  }else{
    MakeSelectList(&s, list);
  }

  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, b->cwd);
  t->flag = DLGF_UPDATECWD | DLGF_UPDATEDST | DLGF_USESRC | DLGF_LINK;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_link;
  if(dst){
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, b->cwd, inputKeyProc);
  ShowInputDialog(d, CAP_LINK);

  return 0;
}

static int doSymLink(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  DfTask *t;
  DfInputDialog *d;
  DfFileList *b;
  const char *dst;

  t = MakeTask(w, GetBuffer(list), DO_SYMLINK, DLG_COPY_FAIL, 0);
  if(!t){
    return 0;
  }

  dst = NULL;
  if(param && *param){
    dst = MakeArgList2(&s, param);
    if(dst && *dst){
      Str_InitStr(&t->dst, dst);
    }else{
      dst = NULL;
    }
  }else{
    MakeSelectList(&s, list);
  }

  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, b->cwd);
  t->flag = DLGF_UPDATECWD | DLGF_UPDATEDST | DLGF_USESRC | DLGF_LINK;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_symlink;
  if(dst){
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, b->cwd, inputKeyProc);
  ShowInputDialog(d, CAP_SYMLINK);

  return 0;
}


static int doDelete(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  char *p;
  DfTask *t;
  DfMessageBox *d;

  if(param && *param){
    MakeArgList(&s, param);
  }else{
    MakeSelectList(&s, list);
  }
  p = Str_Get(&s);

  t = MakeTask(w, GetBuffer(list), DO_DELETE, DLG_DELETEP, 0);
  if(!t){
    return 0;
  }
  Str_InitStr(&t->cwd, list->cwd);

  t->flag = DLGF_USESRC | DLGF_UPDATECWD | DLGF_SUBDIR;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_delete;

  d = MakeMessageBox(t);
  d->b.task = t;
  calcDialogRect(list, &d->b.rc);
  ShowDialog(GetBuffer(list), d, vOptions.msg[DLG_DELETEP], DLG_OK | DLG_CANCEL);

  return 0;
}

static int doMove(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  char *p;
  DfTask *t;
  DfInputDialog *d;
  DfFileList *b;
  const char *dst;

  t = MakeTask(w, GetBuffer(list), DO_MOVE, DLG_MOVE_FAIL, 0);
  if(!t){
    return 0;
  }
  dst = NULL;
  if(param && *param){
    dst = MakeArgList2(&s, param);
    if(dst && *dst){
      Str_InitStr(&t->dst, dst);
    }else{
      dst = NULL;
    }
  }else{
    MakeSelectList(&s, list);
  }
  p = Str_Get(&s);

  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, b->cwd);

  t->flag = DLGF_UPDATECWD | DLGF_UPDATEDST | DLGF_USESRC | DLGF_USEDST | DLGF_SUBDIR;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_move;
  if(dst){
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, b->cwd, copyKeyProc);
  ShowInputDialog(d, CAP_MOVE);

  return 0;
}

static int doChdir(Widget w, DfFileList  *list, const char **param)
{
  DfFileList *b;
  DfInputDialog *d;
  DfTask *t;

  if(param && *param){
    return !chDir(w, list, *param);
  }

  t = MakeTask(w, GetBuffer(list), DO_CHDIR, 0, 0);
  if(!t){
    return 0;
  }
  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, b->cwd);

  t->now_proc = t->files;
  t->v = &vtable_chdir;

  d = MakeInputDialog(t, b->cwd, inputKeyProc);
  ShowInputDialog(d, CAP_CHDIR);

  return 0;
}

static int doIncSearch(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;
  struct dftask_isearch_info *info;


  t = MakeTask(w, GetBuffer(list), DO_ISEARCH, 0, sizeof(struct dftask_isearch_info));
  if(!t){
    return 0;
  }

  t->now_proc = t->files;
  t->v = &vtable_search;
  info = (struct dftask_isearch_info *)t->extend;

  d = MakeInputDialog(t, NULL, isearchKeyProc);
  ShowInputDialog(d, CAP_ISEARCH);
  info->text = d->text;

  return 0;
}

static int doMultimark(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;
  const char *dst;
  DfStr s;

  t = MakeTask(w, GetBuffer(list), DO_MARK, 0, 0);
  if(!t){
    return 0;
  }
  t->files = NULL;
  t->now_proc = NULL;
  t->list = NULL;

  t->v = &vtable_mark;

  dst = NULL;
  if(param && *param){
    MakeArgList(&s, param);
    t->files = Str_Get(&s);
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_MULTIMARK);

  return 0;
}

static int doClearmark(Widget w, DfFileList  *list)
{
  DfFile **f;
  int num;
  int n;

  f = list->files;
  num = list->b.nItems;

  n = 0;
  while(num){
    if(f[0]->nSelect){
      f[0]->nSelect = 0;
      n++;
    }
    f++;
    num--;
  }
  list->nSelects -= n;
  RedrawFrame(w, GetBuffer(list), &list->b.b.rc);

  return 0;
}

static int doReversemark(Widget w, DfFileList  *list)
{
  DfFile **f;
  int num;
  int n;
  int m;

  f = list->files;
  num = list->b.nItems;

  n = 0;
  m = 0;
  while(num){
    if(f[0]->nSelect){
      f[0]->nSelect = 0;
      m++;
    }else{
      if(!IsDots(f[0]->name)){
	n++;
	f[0]->nSelect = n;
      }
    }
    f++;
    num--;
  }
  list->nSelects = list->nSelects + n - m;
  RedrawFrame(w, GetBuffer(list), &list->b.b.rc);

  return 0;
}


static int doMakeFile(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;
  DfStr s;

  t = MakeTask(w, GetBuffer(list), DO_MKFILE, 0, 0);
  if(!t){
    return 0;
  }
  Str_InitStr(&t->cwd, list->cwd);

  t->flag = DLGF_UPDATECWD;
  t->v = &vtable_mkfile;
  if(param && *param){
    MakeArgList(&s, param);
    t->files = Str_Get(&s);
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_MKFILE);

  return 0;
}

static int mkfileInit(DfTask *t)
{
  switch(t->state){
  case STATE_INIT:
    if(t->files == NULL){
      Str_AddChar(&t->input, '\0');
      t->files = Str_Get(&t->input);
      Str_InitNull(&t->input);
    }
    t->state = STATE_NONE;
  default:
    ;
  }
  t->now_proc = t->files;
  commonInit(t);

  return 0;
}

static int mkfileFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st)
{
  int fd;
  fd = open(Str_Get(src), O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
  if(fd == -1){
    return errno;
  }

  return 0;
}

static int mkdirFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st)
{
  int ret;

  ret = mkdir(Str_Get(src), 0777);
  if(ret == -1){
    return errno;
  }

  return 0;
}


static int doMakeDir(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;
  DfStr s;

  t = MakeTask(w, GetBuffer(list), DO_MKDIR, 0, 0);
  if(!t){
    return 0;
  }
  Str_InitStr(&t->cwd, list->cwd);

  t->update = 0;
  t->flag = DLGF_UPDATECWD;
  t->v = &vtable_mkdir;

  if(param && *param){
    MakeArgList(&s, param);
    t->files = Str_Get(&s);
    DlgSendEvent(t, taskCbInvoke);
    return 0;
  }

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_MKDIR);

  return 0;
}

static int doExecute(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;

  if(param && param[0]){
    DfxExecArgv(w, GetBuffer(list), list->cwd, param);
    return 0;
  }

  t = MakeTask(w, GetBuffer(list), DO_EXEC, 0, 0);
  if(!t){
    return 0;
  }

  Str_Init(&t->input);

  t->list = list;
  Str_InitStr(&t->cwd, list->cwd);
  t->flag = DLGF_MULTIPARAM;
  t->files = NULL;
  t->now_proc = t->files;

  t->v = &vtable_exec;

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_EXEC);

  return 0;
}


int DoExecStr(Widget w, DfBuf *b, DfStr *cmd)
{
  DfStr s;
  int n;
  char *p;
  DfFileList *list;
  DfTask *t;
  DfInputDialog *d;

  list = GetFilerBuffer(b);
  n = MakeSelectList(&s, list);
  p = Str_Get(&s);

  t = MakeTask(w, b, DO_EXEC, 0, 0);
  if(!t){
    Str_Free(cmd, 0);
    return 0;
  }
  t->list = list;
  Str_InitStr(&t->cwd, list->cwd);
  t->input = *cmd;
  t->files = p;
  t->flag = DLGF_MULTIPARAM;
  t->now_proc = t->files;

  t->v = &vtable_exec2;

  d = MakeInputDialog(t, Str_Get(cmd), inputKeyProc);
  ShowInputDialog(d, CAP_EXEC);

  return 0;
}


static int doRename(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  int n;
  DfTask *t;
  DfInputDialog *d;
  DfFileList *b;
  DfFile *f;

  t = MakeTask(w, GetBuffer(list), DO_RENAME, 0, 0);
  if(!t){
    return 0;
  }
  n = list->b.nCur;

  f = list->files[n];
  MakeSelectList(&s, list);

  b = GetNextFileBuf(list);

  t->list = b;
  Str_InitStr(&t->cwd, list->cwd);
  Str_InitStr(&t->input, f->name);
  t->flag = DLGF_UPDATECWD | DLGF_USESRC | DLGF_BOTHCWD;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_rename;

  d = MakeInputDialog(t, f->name, inputKeyProc);
  ShowInputDialog(d, CAP_RENAME);

  return 0;
}

static int doFilter(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;

  t = MakeTask(w, GetBuffer(list), DO_FILTER, 0, 0);
  if(!t){
    return 0;
  }
  Str_InitStr(&t->cwd, list->cwd);

  t->update = 0;
  t->list = list;
  t->v = &vtable_filter;

  d = MakeInputDialog(t, NULL, inputKeyProc);
  if(list->filter){
    Str_InitPsz(&d->input, list->filter);
  }
  ShowInputDialog(d, CAP_FILTER);

  return 0;
}

static void filterApply(DfTask *t);
static int filterKeyProc(DfDialog *d, Widget w, KeySym k, char ch, Boolean *cont);

static int doIFilter(Widget w, DfFileList  *list, const char **param)
{
  DfTask *t;
  DfInputDialog *d;
  struct dftask_isearch_info *info;

  t = MakeTask(w, GetBuffer(list), DO_FILTER, 0, sizeof(struct dftask_isearch_info));
  if(!t){
    return 0;
  }

  Str_InitStr(&t->cwd, list->cwd);
  if(list->filter){
    Str_InitPsz(&t->input, list->filter);
  }

  t->update = 0;
  t->list = list;
  t->v = &vtable_ifilter;
  info = t->extend;

  d = MakeInputDialog(t, NULL, filterKeyProc);
  ShowInputDialog(d, CAP_FILTER);
  info->text = d->text;

  return 0;
}

static int filterKeyProc(DfDialog *d, Widget w, KeySym k, char ch, Boolean *cont)
{
  DlgSendEvent(d->task, filterApply);

  return TASK_LEAVE;
}

static void filterApply(DfTask *t)
{
  DfFileList *list = t->list;
  struct dftask_isearch_info *info;
  char *p;

  info = t->extend;
  XtVaGetValues(info->text, XtNstring, &p, NULL);

  if(p && list->filter && strcmp(p, list->filter) == 0){
    return;
  }

  if(Str_IsNull(&t->input)){
    Str_InitStr(&t->input, p);
  }else{
    Str_Overwrite(&t->input, 0, p);
  }

  list->filter = Str_Get(&t->input);
  ApplyFilter(t->w, list, 1);
  RedrawFrame(t->w, GetBuffer(list), &list->b.b.rc);
}

static int doChmod(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  int n;
  char *p;
  DfTask *t;
  DfChmodDialog *d;
  DfFile *f;

  t = MakeTask(w, GetBuffer(list), DO_CHMOD, DLG_CHMODP, sizeof(struct dftask_chmod_info));
  if(!t){
    return 0;
  }
  d = MakeChmodDialog(t);
  n = list->b.nCur;
  f = list->files[n];
  MakeSelectList(&s, list);
  p = Str_Get(&s);

  Str_InitStr(&t->cwd, list->cwd);
  t->flag = DLGF_UPDATECWD | DLGF_USESRC ;
  t->files = Str_Get(&s);
  t->now_proc = t->files;

  t->v = &vtable_chmod;
  d->mode = f->attr;
  d->colum = 0;

  calcDialogRect(list, &d->b.rc);
  ShowChmodDialog(GetBuffer(list), d);

  return 0;
}

static int doChown(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  char *p;
  DfTask *t;
  DfInputDialog *d;

  t = MakeTask(w, GetBuffer(list), DO_CHOWN, DLG_CHOWNP, 0);
  if(!t){
    return 0;
  }

  MakeSelectList(&s, list);

  Str_InitStr(&t->cwd, list->cwd);
  p = Str_Get(&s);
  t->flag = DLGF_UPDATECWD | DLGF_USESRC ;
  t->files = p;
  t->now_proc = p;

  t->v = &vtable_chown;

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_CHOWN);

  return 0;
}

static int doChgrp(Widget w, DfFileList  *list, const char **param)
{
  DfStr s;
  char *p;
  DfTask *t;
  DfInputDialog *d;

  t = MakeTask(w, GetBuffer(list), DO_CHGRP, DLG_CHOWNP, 0);
  if(!t){
    return 0;
  }

  MakeSelectList(&s, list);

  Str_InitStr(&t->cwd, list->cwd);
  p = Str_Get(&s);
  t->flag = DLGF_UPDATECWD | DLGF_USESRC ;
  t->files = p;
  t->now_proc = p;

  t->v = &vtable_chown;

  d = MakeInputDialog(t, NULL, inputKeyProc);
  ShowInputDialog(d, CAP_CHGRP);

  return 0;
}
static int doQuit(Widget w, DfFileList  *list, const char **param)
{
  if(1 < dfx->nFiler){
    KillBuffer(w, &list->b.b);

    /*  ToDo:set visible next buffer */
    return 0;
  }
  if(ListIsEmpty(&vDlgList)){
    exit(0);
  }
 
  /* cann't close yet. */
  St_SetMsg(w, vOptions.msg[STM_CANTCLOSE]);
  return 0;
}

/**/

static task_result chdirInvoke(DfTask *t)
{
  chDir(t->w, (DfFileList*)t->owner, Str_Get(&t->input));

  return TASK_FREE;
}

static task_result isearchNotify(DfTask *t)
{
  DfFileList *l;
  int i;
  int end;
  char *p;
  int dir;
  Arg a[1];
  int redraw;
  struct dftask_isearch_info *info;

  l = (DfFileList*)t->owner;
  info = t->extend;
  
  XtSetArg(a[0], XtNstring, &p);
  XtGetValues(info->text, a, 1);

  i = l->b.nCur;
  end = i;
  dir = 0;
  switch(t->state){
  case CMD_ISEARCHNEXT:
    if(l->b.nItems == 0){
      return TASK_LEAVE;
    }
    goto NEXT;
    break;
  case CMD_ISEARCHPREV:
    if(l->b.nItems == 0){
      return TASK_LEAVE;
    }
    dir = 1;
    goto NEXT;
  case CMD_ISEARCHOPEN:
    if(l->b.nItems == 0){
      return TASK_LEAVE;
    }
    if(!openDir(dfx->w, l, NULL, &redraw)){
      p = "";
      InvalidateRect(dfx->w, &l->b.b.rc);
      XtSetArg(a[0], XtNstring, p);
      XtSetValues(info->text, a, 1);
    }
    return TASK_LEAVE;

  case CMD_ISEARCHPARENT:
    if(gotoParent(dfx->w, l)){
      InvalidateRect(dfx->w, &l->b.b.rc);
      return TASK_LEAVE;
    }
    break;
  default:
    break;
  }
  do{
    if(IsMatch(p, l->files[i]->name) != 2){
      SetCursor(dfx->w, (DfBufCommon*)l, i);
      break;
    }
NEXT:
    if(dir){
      if(i == 0){
	i = l->b.nItems - 1;
      }else{
	i--;
      }
    }else{
      i++;
      if(l->b.nItems <= i){
	i = 0;
      }
    }
  }while(end != i);

  return TASK_LEAVE;
}

static task_result isearchDone(DfTask *t)
{
  St_SetMsg(t->w, NULL);
  return TASK_FREE;
}


static int multiMarking(Widget w, DfFileList *l, const char *pattern);
static task_result multimarkInvoke(DfTask *t)
{
  DfFileList *l;
  const char *pat;
  int cnt;

  l = GetFilerBuffer(t->owner);


  if(t->flag & DLGF_INPUT){
    pat = Str_Get(&t->input);
    cnt = multiMarking(t->w, l, pat);
  }else{
    cnt = 0;
    pat = t->files;
    while(pat && *pat){
      cnt += multiMarking(t->w, l, pat);
      pat += (strlen(pat) + 1);
    }
  }

  if(cnt){
    St_SetMsg(t->w, vOptions.msg[STM_MATCH]);
    RedrawFrame(t->w, t->owner, &t->owner->rc);
  }else{
    St_SetMsg(t->w, vOptions.msg[STM_NOMATCH]);
  }

  return TASK_FREE;
}

static int multiMarking(Widget w, DfFileList *l, const char *pattern)
{
  DfFile **f;
  int i;
  int n;
  int cnt;

  f = l->files;
  n = l->b.nItems;
  cnt = 0;
  for(i = 0; i < n; i++){
    if(f[i]->nSelect == 0 ){
      if(IsMatch(pattern, f[i]->name) == 0){
	markFile(w, l, i, 0);
	cnt++;
      }
    }
  }

  return cnt;
}


static task_result renameNotify(DfTask *t)
{
  DfStr msg;
  DfInputDialog *d;

  Str_InitStr(&msg, t->now_proc);
  Str_AddChar(&msg, ':');
  Str_Add(&msg, strerror(t->err));
  St_SetMsg(t->w, Str_Get(&msg));
  Str_Free(&msg, 0);

  d = MakeInputDialog(t, Str_Get(&t->input), inputKeyProc);
  ShowInputDialog(d, MSG_MAX);

  return TASK_LEAVE;
}

static int renameFileOp(DfTask *t, DfStr *src, DfStr *dst, struct stat *st)
{
  int ret;

  ret = rename(Str_Get(src), Str_Get(dst));
  if(ret){
    t->state = STATE_FAIL;
    t->err  = errno;
    return ret;
  }

  return 0;
}

static task_result execInvoke(DfTask *t)
{
  const char *cmd;
  int builtin;
  int fail;

  cmd = Str_Get(&t->input);

  St_SetMsg(t->w, NULL);
  if(cmd && cmd[0]){
    fail = DfxExec(t->w, t->owner, Str_Get(&t->cwd), cmd, &builtin);
    if(fail == 0 && builtin == 0){
      KillChildren(t->w, t->owner);
    }
  }

  return TASK_FREE;
}


static void toggleDotfiles(Widget w, DfFileList *l)
{
  int m;
  char *p;

  l->mode ^= SHOW_DOTS;

  m = l->b.nCur;
  p = NULL;
  if(l->b.nItems){
    p = l->files[m]->name;
  }

  ApplyFilter(w, l, l->mode & MODE_IFILTER);
  SortFileList(l);
  if(p){
    m = findFileName(l, p, m);
  }

  SetCursor(w, (DfBufCommon*)l, m);

}

static void calcDialogRect(DfFileList *l, XRectangle *rc)
{
  DfBufCommon *b;
  int n;
  int min;
  int max;
  XRectangle rcBottom;

  calcCursorRect(&l->b, l->b.nCur, rc);
  if(l->nSelects == 0){
    return;
  }

  b = &l->b;
  min = b->nStart;
  max = min + l->b.b.rc.height / vnFontHeight - 1;
  if(b->nItems <= max){
    max = b->nItems - 1;
  }

  for(n = min; n <= max; n++){
    if(l->files[n]->nSelect){
      min = n;
      goto FOUND;
    }
  }
  return;
FOUND:

  while(min < max){
    if(l->files[max]->nSelect){
      break;
    }
    max--;
  }
  calcCursorRect(&l->b, min, rc);
  if(min != max){
    calcCursorRect(&l->b, max, &rcBottom);
    RectAdd(rc, &rcBottom);
  }
}


