#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "config.h"


#ifdef MEMDEBUG
#define MEM_GUARD (4096)
#define GUARD_PTR(p) ((void*)(((char*)(p)) - MEM_GUARD))
#define APP_PTR(p)   ((void*)(((char*)(p)) + MEM_GUARD))

#define  ALLOW_FREE  1
#define  DELETE_INFO 0

#ifdef X86_BREAK
#define BREAK __asm("int $3")
#endif

#ifndef BREAK
#define BREAK do{}while(0)
#endif

typedef struct _meminfo{
  void *p;
  int size;
  const char *file;
  const char *func;
  int line;
  int freed;
}MEMINFO;


MEMINFO *vMemInfo;
int vnMems;

void *trap_ptr;

static void AddMemInfo(void *p, int size, const char *file, const char *func, int line);
static void DeleteMemInfo(MEMINFO *p);
MEMINFO *FindMemInfo(void *p);
void DumpMemInfo(void);
void trap_free(void *p);
void dbgFree(void *ptr, const char *file, const char *func, int line);
void CheckAllMem(void);
int ChkMem_info(void *ptr, MEMINFO *info);
void CheckPtr(void *ptr);


void trap_free(void *p)
{
  trap_ptr = p;
}

int is_trap(void *ptr)
{
  return trap_ptr == ptr;
}


void *dbgMalloc(int size, const char *file, const char *func, int line)
{
  void *p;
  p = malloc(size + (MEM_GUARD * 2));
  if(p == NULL){
    return NULL;
  }
  memset(p, 0xAA, size + (MEM_GUARD * 2));
  AddMemInfo(APP_PTR(p), size, file, func, line);

  CheckAllMem();
  return APP_PTR(p);
}

void *dbgReAlloc(void *ptr, int size, const char *file, const char *func, int line)
{
  void *p;
  MEMINFO *info;
  int n;

  if(!ptr){
    return dbgMalloc(size, file, func, line);
  }
    
  info = FindMemInfo(ptr);
  if(info == NULL){
    fprintf(stderr, "BUG: realloc() called with invalid pointer %p. %s %s %d\n",
	    ptr, file, func, line);
    return NULL;
  }
#if 0
  p = realloc(ptr, size);
  info->size = size;
  info->p = p;
  info->file = file;
  info->func = func;
  info->line = line;
#else
  p = dbgMalloc(size, file, func, line);
  n = size < info->size ? size : info->size;
  memcpy(p, ptr, n);
  if(memcmp(p, ptr, n)){
    BREAK;
  }
  
  dbgFree(ptr, file, func, line);
#endif

  return p;
}

void dbgFree(void *ptr, const char *file, const char *func, int line)
{
  MEMINFO *info;

  CheckAllMem();
  if(!ptr){
    return;
  }

  if(trap_ptr == ptr){
    fprintf(stderr, "TRAP: free() called with watch pointer %p. %s %s %d\n",
	    ptr, file, func, line);
  }
  info = FindMemInfo(ptr);
  if(info == NULL){
    fprintf(stderr, "BUG: free() called with invalid pointer %p. %s %s %d\n",
	    ptr, file, func, line);
    return;
  }
  if(info->freed){
    DumpMemInfo();
    fprintf(stderr, "BUG: free() double called with pointer %p. %s %s %d\n",
	    ptr, file, func, line);
    fprintf(stderr, "    already released at %s %s %d\n",
	    info->file, info->func, info->line);
    return;
  }

  memset(GUARD_PTR(ptr), 0xAA, info->size + (MEM_GUARD * 2));
#if ALLOW_FREE
  free(GUARD_PTR(ptr));
#endif
  info->freed = 1;

#if DELETE_INFO
  DeleteMemInfo(info);
#endif
}

static int check_clean(void *ptr)
{
  unsigned int *p = ptr;
  int cnt = MEM_GUARD / sizeof(unsigned int);

  while(cnt){
    if(*p != 0xAAAAAAAA){
      return 1;
    }
    p++;
    cnt--;
  }
  return 0;
}

int ChkMem(void *ptr)
{
  MEMINFO *info;
 
  info = FindMemInfo(ptr);
  if(info == NULL){
    fprintf(stderr, "BUG: checker called with invalid pointer %p.\n",
	    ptr);
    return 1;
  }
  if(ptr != info->p){
    BREAK;
  }

  return ChkMem_info(ptr, info);
}

int ChkMem_info(void *ptr, MEMINFO *info)
{
  if(ptr != info->p){
    BREAK;
  }

  if(check_clean(GUARD_PTR(ptr))){
    fprintf(stderr, "BUG: detected buffer underrun. %p. %s %s %d.\n",
	    ptr, info->file, info->func, info->line);
    CheckPtr(ptr);
    BREAK;
    return 1;
  }
    
  if(check_clean(((char *)ptr) + info->size)){
    fprintf(stderr, "BUG: detected buffer overrun. %p. %s %s %d.\n",
	    ptr, info->file, info->func, info->line);
    CheckPtr(ptr);
    BREAK;
    return 1;
  }

  return 0;  
}

void CheckAllMem(void)
{
  MEMINFO *p;
  int n;
  int total;

  total = 0;
  p = vMemInfo;
  for(n = 0; n < vnMems; n++){
    if(p->freed == 0 && ChkMem_info(p->p, p)){
      fprintf(stderr, "%d: %p:%8dBytes %s:%s() %d is broken.\n",
	      n + 1, p->p ,p->size, p->file, p->func, p->line);
      BREAK;
    }
    p++;
  }
}

void CheckPtr(void *ptr)
{
  MEMINFO *p;
  char *range;
  int n;

  p = vMemInfo;
  for(n = 0; n < vnMems; n++){
    range = p->p;
    range += p->size;
    if(p->p <= ptr && ptr <= (void*)range){
      fprintf(stderr, "%d: %p is into %p:%8dBytes %s:%s() %d\n",
	      n + 1, ptr, p->p ,p->size, p->file, p->func, p->line);
    }
    p++;
  }
}


void DumpMemInfo(void)
{
  MEMINFO *p;
  int n;
  int total;

  total = 0;
  p = vMemInfo;
  for(n = 0; n < vnMems; n++){
    if(!p->freed){
      fprintf(stderr, "%d: %p:%8dBytes %s:%s() %d\n",
	      n + 1, p->p ,p->size, p->file, p->func, p->line);
      total += p->size;
    }
    p++;
  }
  fprintf(stderr, "Total %d.\n", total);
}

void MemModInfo(void *ptr, const char *file, const char *func, int line)
{
  MEMINFO *info;

  info = FindMemInfo(ptr);
  if(info == NULL){
    fprintf(stderr, "BUG: modify info() called with invalid pointer %p. %s %s %d\n",
	    ptr, file, func, line);
    return;
  }

  info->file = file;
  info->func = func;
  info->line = line;
}

static void AddMemInfo(void *p, int size, const char *file, const char *func, int line)
{
  MEMINFO *newInfo;

  newInfo = FindMemInfo(p);
  if(newInfo){
    newInfo->p = p;
    newInfo->size = size;
    newInfo->file = file;
    newInfo->func = func;
    newInfo->line = line;
    newInfo->freed = 0;
    return;
  }

  if(vnMems){
    newInfo = realloc(vMemInfo, sizeof(MEMINFO) * (vnMems + 1));
  }else{
    newInfo = malloc(sizeof(MEMINFO));
  }
    
  if(!newInfo){
    fprintf(stderr, "not enough memory at AddMemInfo.\n");
    return;
  }
  vMemInfo = newInfo;
  newInfo = vMemInfo + vnMems;

  newInfo->p = p;
  newInfo->size = size;
  newInfo->file = file;
  newInfo->func = func;
  newInfo->line = line;
  newInfo->freed = 0;
  vnMems++;
}

static void DeleteMemInfo(MEMINFO *p)
{
  int n;

  n = p -vMemInfo;
  n = (vnMems - n - 1) * sizeof(MEMINFO);
  memmove(p, p + 1, n);

  vnMems--;
}

MEMINFO *FindMemInfo(void *ptr)
{
  MEMINFO *p;
  int n;

  p = vMemInfo;
  for(n = 0; n < vnMems; n++){
    if(p->p == ptr){
      return p;
    }
    p++;
  }
  return NULL;
}

#endif

