#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <dirent.h>
#include <errno.h>
#include <sys/stat.h>
#include <pthread.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/keysym.h>
#include <X11/Shell.h>

#include <X11/Xaw/Form.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/AsciiText.h>

#include "config.h"
#include "dftype.h"
#include "list.h"
#include "str.h"
#include "buffer.h"
#include "filer.h"
#include "status.h"
#include "dialog.h"
#include "task.h"
#include "mem.h"
#include "misc.h"
#include "fop.h"
#include "dfxfunc.h"
#include "dfval.h"
#include "debug.h"

#define CACHE_TEXT 0

DfList vDlgList = {NULL, NULL};
DfList vNotifyTaskList = {NULL, NULL};
pthread_mutex_t vMtxNotify;

extern Widget toplevel;


#define ALIGNED(N) ((((N)+7)/8)*8)

DfTask *MakeTask(Widget w, DfBuf *owner, df_opcode op, int msg_id, int ext_size)
{
  DfTask *t;
  int size;

  size = ALIGNED(sizeof(DfTask)) + ext_size;
  dprintf("MakeTask: requiered %u + %d = %d bytes\n", sizeof(DfTask), ext_size, size);

  t = bMalloc(size);
  if(t == NULL){
    return NULL;
  }
  memset(t, 0, size);
  ListAdd(&vDlgList, &t->link);
  Str_InitNull(&t->cwd);
  Str_InitNull(&t->dst);
  Str_InitNull(&t->input);
  Str_InitNull(&t->cand);
  t->owner = owner;
  t->op = op;
  t->state = STATE_INIT;
  t->init_msg_id = msg_id;
  t->msg_id = msg_id;
  t->err = 0;
  t->w = w;
  t->list = NULL;
  t->sub = NULL;
  t->files = NULL;
  t->now_proc = NULL;
  if(ext_size){
    t->extend = (char*)t + ALIGNED(sizeof(DfTask));
    dprintf("MakeTask: %p/%p(%p)\n", t, t->extend, (char*)t + sizeof(DfTask));
  }

  return t;
}

int OpCancelOperate(void)
{
  DfTask *t;
  DfList *l;

  l = &vDlgList;
  l = ListNext(l);
  while(l != &vDlgList){
    t = CONTENT_OF(DfTask, link, l);
    dprintf("break.\n");
    t->break_flg = 1;
    l = ListNext(l);
  }
  return 0;
}



void DlgSendEvent(DfTask *t, dlg_request rq)
{
  int n;
  XClientMessageEvent ev;

  ev.type = ClientMessage;
  ev.window = XtWindow(t->w);
  ev.message_type = DRQ_CALLBACK;
  ev.format = 32;

  t->callback = rq;
  dprintf("send_event: attempt to lock mutex.\n");
  pthread_mutex_lock(&vMtxNotify);
  dprintf("send_event: locked and add to list %p.\n", t);
  ListAddTail(&vNotifyTaskList, &t->link_notify);
  pthread_mutex_unlock(&vMtxNotify);
  dprintf("send_event: mutex_released.\n");

  n = XSendEvent(XtDisplay(t->w), XtWindow(t->w), False, NoEventMask, (XEvent*)&ev);
  if(t->flag & DLGF_THREAD){
    XFlush(XtDisplay(t->w));
  }
}

void DfxSendMsg(int msg, int err)
{
  Display *d;
  XClientMessageEvent ev;

  ev.type = ClientMessage;
  ev.window = XtWindow(dfx->w);
  ev.message_type = DRQ_STMSG;
  ev.format = 32;
  ev.data.l[0] = msg;
  ev.data.l[1] = err;

  d = XOpenDisplay(NULL);
  XSendEvent(d, XtWindow(dfx->w), False, NoEventMask, (XEvent*)&ev);
  XCloseDisplay(d);
}



int setupPathes(DfTask *t, DfStr *src, DfStr *dst, int bSlash)
{
  /* setup source path */
  if(src){
    Str_InitNull(src);
    Str_Copy(src, &t->cwd);
    if(bSlash){
      SetLastSlash(src);
    }
  }

  /* construct mdestination path */
  if(dst){
    Str_InitNull(dst);
    if(IsAbsPath(Str_Get(&t->dst))){
      Str_Copy(dst, &t->dst);
    }else{
      Str_Copy(dst, &t->cwd);
      SetLastSlash(dst);
      Str_Add(dst, Str_Get(&t->dst));
    }
    Str_RegPath(dst, bSlash, 0);
  }

  return 0;
}
DfSubDir *initSubDir(DfTask *t, const char *src, const char *dst)
{
  DfSubDir *sd;
  int n;

  n = 0;
  if(src){
    n++;
  }
  if(dst){
    n++;
  }

  if(src || dst){
    /* initial */
    sd = bMalloc(sizeof(DfSubDir) + (sizeof(DfStr) * n));
    if(sd == NULL){
      t->err = errno;
      return NULL;
    }
    if(src){
      sd->src = (DfStr*)(sd + 1);
      Str_InitStr(sd->src, src);
      if(dst){
	sd->dst = sd->src + 1;
	Str_InitStr(sd->dst, dst);
      }else{
	sd->dst = NULL;
      }
    }else{
      if(dst){
	sd->dst = (DfStr*)(sd + 1);
	Str_InitStr(sd->dst, dst);
      }
    }
  }else{
    sd = bMalloc(sizeof(DfSubDir));
    if(sd == NULL){
      t->err = errno;
      return NULL;
    }
    sd->src = t->sub->src;
    sd->dst = t->sub->dst;
  }
  if(sd->src){
    SetLastSlash(sd->src);
    sd->s_idx = Str_Length(sd->src);
  }
  if(sd->dst){
    SetLastSlash(sd->dst);
    sd->d_idx = Str_Length(sd->dst);
  }

  dprintf("initSubDir: %s\n", Str_Get(sd->src));
  sd->dir = opendir(Str_Get(sd->src));
  sd->de = NULL;
  sd->mode = 0;

  sd->parent = t;
  sd->next = t->sub;
  t->sub = sd;

  return sd;
}



int commonInit(DfTask *t)
{
  char *name;
  struct stat st;
  int exist;
  int dir;
  char *slash;
  char *ptr;
  DfInputDialog *d;

  t->msg_id = t->init_msg_id;
  switch(t->state){
  case STATE_INIT:
    t->state = STATE_NONE;
    if(t->flag & DLGF_LINK){
      exist = stat(Str_Get(&t->input), &st);
      dir = 0;
      if(exist & S_ISDIR(st.st_mode)){
	dir = 1;
      }else{
	if(IsEndSlash(&t->input)){
	  dir = 1;
	}
      }
      if(dir){
	Str_Move(&t->dst, &t->input);
      }else{
	Str_Move(&t->dst, &t->input);
	ptr = Str_Get(&t->dst);
	slash = FindLastChar(ptr, '/');
	if(*slash){
	  Str_Overwrite(&t->input, 0, slash + 1);
	}
	Str_SetLength(&t->dst, slash - ptr);
      }
    }else if(t->flag & DLGF_INPUT){
      if(t->flag & DLGF_USEDST){
	Str_Move(&t->dst, &t->input);
      }
      t->flag &= ~DLGF_INPUT;
    }
    break;
  case STATE_NONE:
  case STATE_COPYAS:
    break;

  case CMD_YES:
  case CMD_NO:
  case CMD_OK:
  case CMD_RETRY:
  case CMD_OVERWRITE:
    break;
  case CMD_SKIP:
    break;

  case CMD_RENAME:
    t->state = STATE_COPYAS;
    if(t->sub){
      name = t->sub->de->d_name;
    }else{
      name = t->now_proc;
    }

    d = MakeInputDialog(t, name, inputKeyProc);
    ShowInputDialog(d, CAP_RENAME);
    return 1;

  case CMD_CANCEL:
    DlgSendEvent(t, taskCbDone);
    return 1;
  default:
    ;
  }


  return 0;
}

static void *operation_thread(void *ptr)
{
  DfTask *t;

  t = ptr;
  t->flag |= DLGF_THREAD;

  commonInvoke(t);

  return NULL;
}

task_result commonThreadedInvoke(DfTask *t)
{

  pthread_create(&t->t_info, NULL, operation_thread, t);
  St_SetMsg(t->w, NULL);

  return TASK_LEAVE;
}


task_result commonInvoke(DfTask *t)
{
  int ret;

  if(t->v->init(t)){
    return TASK_LEAVE;
  }

  while(t->break_flg == 0 && *t->now_proc){
    if(t->sub){
      dprintf("commonInvoke: sub directory\n");
      ret = t->v->procSubDirectories(t);
    }else{
      dprintf("commonInvoke: current file %s\n", t->now_proc);
      ret = t->v->procSelectedFiles(t);
    }
    if(t->break_flg){
      t->state = STATE_CANCEL;
      break;
    }
    if(ret){
      /* error occurs */
      DlgSendEvent(t, taskCbNotify);
      return TASK_LEAVE;
    }
  }

  DlgSendEvent(t, taskCbDone);
  dprintf("done.\n");
  return TASK_LEAVE;
}


task_result commonDone(DfTask *t)
{
  const char *msg;
  DfFileList *l;
  DfFileList *cur;
  DfBuf *b;

  msg = NULL;
  switch(t->state){
  case STATE_ALREADYEXIST:
    msg = vOptions.msg[STM_ALREADY_EXIST];
    break;

  case STATE_CANCEL:
    break;

  default:
    if(t->err){
      msg = strerror(t->err);
    }
  }
  St_SetMsg(t->w, msg);

  if(!t->update){
    return TASK_FREE;
  }

  cur = GetFilerBuffer(t->owner);
  if(t->flag & DLGF_UPDATECWD){
    reload(t->w, cur, Str_Get(&t->dst));
  }

  b = dfx->lists;
  do{
    b = NextBuffer(b);
    if(b->type != DT_FILER){
      continue;
    }
    if(b == (DfBuf *)cur){
      continue;
    }

    l = (DfFileList*)b;
    if(t->flag & DLGF_UPDATECWD){
      if(strcmp(Str_Get(&t->cwd), l->cwd) == 0){
	reload(t->w, l, NULL);
	continue;
      }
    }
    if(t->flag & DLGF_UPDATEDST){
      if(strcmp(Str_Get(&t->dst), l->cwd) == 0){
	reload(t->w, l, NULL);
      }
    }
  }while(b != dfx->lists);

  return TASK_FREE;
}

int commonProcSelectedFile(DfTask *t)
{
  int len;
  char *name;
  char *tmp;
  int d_idx;
  int s_idx;
  DfStr dst;
  DfStr src;
  int use_dst;
  int ret;
  struct stat st;

  name = t->now_proc;

  switch(t->state){
  case CMD_SKIP:
    len = strlen(name) + 1;
    name += len;
    t->now_proc = name;
    break;

  default:
    break;
  }

  if(!*name){
    return 0;
  }

  use_dst = 0;
  if(t->flag & DLGF_USEDST){
    use_dst = 1;
    setupPathes(t, &src, &dst, 1);
    s_idx = Str_Length(&src);
    d_idx = Str_Length(&dst);
    dprintf("selected file: make directory %s\n", Str_Get(&dst));
    ret = CreateNestDirectory(Str_Get(&dst));
    if(ret){
      Str_Free(&src, 0);
      Str_Free(&dst, 0);
      t->msg_id = DLG_CANNOT_MKDIR_DEST;
      t->err = ret;
      t->state = STATE_FAIL;
      return 1;
    }
  }else if(t->flag & DLGF_BOTHCWD){
    use_dst = 2;
    Str_InitNull(&src);
    Str_Copy(&src, &t->cwd);
    SetLastSlash(&src);

    Str_InitNull(&dst);
    Str_Copy(&dst, &t->cwd);
    SetLastSlash(&dst);
    s_idx = Str_Length(&src);
    d_idx = Str_Length(&dst);
  }else if(t->flag & DLGF_LINK){
    use_dst = 3;
    setupPathes(t, &src, &dst, 0);
    SetLastSlash(&dst);
    s_idx = Str_Length(&src);
    d_idx = Str_Length(&dst);
  }else{
    setupPathes(t, &src, NULL, 1);
    s_idx = Str_Length(&src);
    d_idx = 0;/* against warnning */
    Str_InitNull(&dst);
  }

  while(*name && (t->break_flg == 0)){
    Str_Overwrite(&src, s_idx, name);
    switch(use_dst){
    case 0:
      if(t->state == STATE_COPYAS){
	t->state = STATE_NONE;
	Str_Overwrite(&src, s_idx, Str_Get(&t->input));
      }
      break;
    case 1:
      if(t->state == STATE_COPYAS){
	t->state = STATE_NONE;
	Str_Overwrite(&dst, d_idx, Str_Get(&t->input));
      }else{
	Str_Overwrite(&dst, d_idx, name);
      }
      break;
    case 2:
      Str_Overwrite(&src, s_idx, name);
      Str_Overwrite(&dst, d_idx, Str_Get(&t->input));
      break;
    case 3:
      Str_Overwrite(&src, s_idx, name);
      tmp = name;
      if(Str_Length(&t->input)){
	tmp = Str_Get(&t->input);
      }
      Str_Overwrite(&dst, d_idx, tmp);
      break;
    }

    if(use_dst){
      /* check already exist */
      if(t->state != CMD_OVERWRITE){
	if(lstat(Str_Get(&dst), &st) == 0){
	  if(t->state == CMD_SKIP){
	    goto NEXT;
	  }
	  St_SetMsg(t->w, vOptions.msg[STM_ALREADY_EXIST]);
	  t->err = EEXIST;
	  dprintf("selected files: already exist.\n");
	  t->state = STATE_ALREADYEXIST;

	  Str_Free(&src, 0);
	  Str_Free(&dst, 0);
	  return 1;
	}
      }
    }

    /*d->op = DO_COPY;*/
    if(t->flag & DLGF_USESRC){
      if(lstat(Str_Get(&src), &st)){
	t->err = errno;
	Str_Free(&src, 0);
	Str_Free(&dst, 0);
	return 1;
      }
      if(S_ISDIR(st.st_mode)){
	if(IsDots(name)){
	  goto NEXT;
	}
	if(t->flag & DLGF_SUBDIR){
          do{
	    if(t->v->predirwalk){
	      dprintf("selected file: found directory. %s, %s\n", Str_Get(&src), Str_Get(&dst));
	      ret = t->v->predirwalk(t, &src, &dst);
	      if(ret != 0){
		dprintf("selected file: pre directry walker stated to break. errno is %d.\n", t->err);
		if(t->err){
		  ret = 1;
		  break;
		}
		t->update = 1;
		goto NEXT;
	      }
	      t->update = 1;
	    }

	    if(initSubDir(t, Str_Get(&src), Str_Get(&dst))){
	      t->err = 0;
	      ret = 0;
	    }else{
	      t->state = STATE_FAIL;
	      ret = 1;
	    }
	  }while(0);

	  Str_Free(&src, 0);
	  Str_Free(&dst, 0);
	  return ret;
	}
      }
    }

    ret = t->v->fileop(t, &src, &dst, &st);
    if(ret != 0){
      t->err = ret;

      Str_Free(&src, 0);
      Str_Free(&dst, 0);
      return 1;
    }
    t->update = 1;
  NEXT:
    len = strlen(name) + 1;
    name += len;
    t->now_proc = name;
  }

  Str_Free(&src, 0);
  Str_Free(&dst, 0);

  return 0;
}

int commonProcSubDirectories(DfTask *t)
{
  DfSubDir *sd;
  struct stat st;
  struct stat d_st;
  int ret = 0;
  int len;

  sd = t->sub;

  if(t->flag & DLGF_USEDST){
    ret = CreateNestDirectory(Str_Get(sd->dst));
    if(ret){
      t->msg_id = DLG_CANNOT_MKDIR_DEST;
      t->err = ret;
      t->state = STATE_FAIL;
      return 1;
    }
  }

  switch(t->state){
  case CMD_RETRY:
    if(sd->de == NULL && sd->dir != NULL){
      sd->de = readdir(sd->dir);
    }
    break;
  case CMD_SKIP:
    if(sd->dir != NULL){
      sd->de = readdir(sd->dir);
    } else {
      goto detach;
    }
    break;
  default:
    if(sd->dir != NULL){
      sd->de = readdir(sd->dir);
     }
  }

  if(sd->dir){
    for(;sd->de; sd->de = readdir(sd->dir)){
      if(t->break_flg){
	break;
      }
      Str_Overwrite(sd->src, sd->s_idx, sd->de->d_name);
      if(t->flag & DLGF_USEDST){
	if(t->state == STATE_COPYAS){
	  t->state = STATE_NONE;
	  Str_Overwrite(sd->dst, sd->d_idx, Str_Get(&t->input));
	}else{
	  Str_Overwrite(sd->dst, sd->d_idx, sd->de->d_name);
	}
      }

      lstat(Str_Get(sd->src), &st);
      if(S_ISDIR(st.st_mode)){
	if(!(t->flag & DLGF_ALLOWDOTFILE) && IsDots(sd->de->d_name)){
	  continue;
	}
	if(t->v->predirwalk){
	  ret = t->v->predirwalk(t, sd->src, sd->dst);
	  if(ret != 0){
	    dprintf("sub directories: pre directry walker stated to break. errno is %d.\n", t->err);
	    if(t->err){
	      return 0;
	    }
	    t->update = 1;
	    continue;
	  }
	}
	if(initSubDir(t, NULL, NULL)){
	  t->err = 0;
	  t->state = STATE_NONE;
	}else{
	  t->state = STATE_FAIL;
	}
	return 0;
      }

      if(t->flag & (DLGF_USEDST | DLGF_BOTHCWD)){
	/* check already exist */
	switch(t->state){
	case CMD_OVERWRITE:
	  break;
	case CMD_SKIP:
	  continue;
	default:
	  if(lstat(Str_Get(sd->dst), &d_st) == 0){
	    dprintf("sub directories: already exist.\n");
	    St_SetMsg(t->w, vOptions.msg[STM_ALREADY_EXIST]);
	    t->state = STATE_ALREADYEXIST;
	    return 1;
	  }
	}
      }
      ret = t->v->fileop(t, sd->src, sd->dst, &st);
      if(ret != 0){
	t->err = ret;
	return 1;
      }

      /* next */
      t->update = 1;
    }
    closedir(sd->dir);
    sd->dir = NULL;
  }
  Str_SetLength(sd->src, sd->s_idx);
  if(t->flag & DLGF_USEDST){
    Str_SetLength(sd->dst, sd->d_idx);
  }

  ret = 0;

  if(t->break_flg == 0 && t->v->postdirwalk){
    ret = t->v->postdirwalk(t, sd->src, sd->dst);
    if(ret != 0){
      t->err = ret;
      ret = 1;
    }
  }

  /* detach */
detach:
  t->sub = sd->next;

  if(t->sub == NULL){
    Str_Free(sd->src, 0);
    Str_Free(sd->dst, 0);
    len = strlen(t->now_proc) + 1;
    t->now_proc += len;
  }

  bFree(sd);
  return ret;
}

task_result commonNotify(DfTask *t)
{
  DfStr msg;
  DfMessageBox *d;
  int msg_id;
  int buttons;
  const char *p;

  dprintf("common_notify: state %d\n", t->state);
  buttons =  DLG_RETRY | DLG_SKIP | DLG_CANCEL;
  switch(t->state){
  case STATE_NONE:
    dprintf("common_notify: retry to invoke. state %d\n", t->state);
    return t->v->invoke(t);

  case STATE_ALREADYEXIST:
    /* overwrite? */
    msg_id = DLG_OVERWRITEP;
    buttons = DLG_OVERWRITE | DLG_SKIP | DLG_RENAME | DLG_CANCEL;
    break;
  case STATE_DELETEONMOVEFAIL:
    /* fail */
    msg_id = DLG_MOVE_FAIL_DELETE;
    break;
  default:
    dprintf("common_notify: to recover. state %d\n", t->state);
    msg_id = t->msg_id;
    break;
  }

  p = t->now_proc;
  if(t->sub){
    p = Str_Get(t->sub->dst ? t->sub->dst : t->sub->src);
  }
  Str_InitStr(&msg, p);

  Str_AddChar(&msg, '\n');
  if(t->err){
    Str_Add(&msg, strerror(t->err));
    Str_AddChar(&msg, '\n');
    St_SetMsg(t->w, strerror(t->err));
  }
  Str_Add(&msg, vOptions.msg[msg_id]);

  d = MakeMessageBox(t);
  ShowDialog(t->owner, d, Str_Get(&msg), buttons);
  Str_Free(&msg, 0);

  return TASK_LEAVE;
}

int PreDirDelete(DfTask *t, DfStr *src, DfStr *dst)
{
  int ret;

  if(t->err != 0){
    t->state = STATE_FAIL;
    return 0;
  }

  dprintf("pre remove directory: enter.\n");
  ret = DeleteDirectory(Str_Get(src));
  switch(ret){
  case 0:
    dprintf("pre remove directory: allow to remove %s.\n", Str_Get(src));
    t->err = 0;
    t->state = STATE_NONE;
    return 1;/* nohing to do  */
  case ENOTEMPTY:
    dprintf("pre remove directory: directory %s is not empty. but allow to remove.\n", Str_Get(src));
    return 0;/* allow to continue */
  default:
    return ret;
  }
}

int PostDirDelete(DfTask *t, DfStr *src, DfStr *dst)
{
  int ret;

  if(t->err != 0){
    t->state = STATE_FAIL;
    return 0;
  }

  ret = DeleteDirectory(Str_Get(src));
  if(ret != 0){
    t->err = ret;
    t->state = STATE_FAIL;
    return ret;
  }
  t->update = 1;

  return ret;
}

void taskCbInvoke(DfTask *t)
{
  dprintf("request invoke\n");
  if(t->v->invoke){
    if((*t->v->invoke)(t) == 0){
      FreeTask(t);
    }
  }else{
    if(t->v->done){
      (*t->v->done)(t);
    }
    St_SetMsg(t->w, NULL);
    FreeTask(t);
  }
}

void taskCbNotify(DfTask *t)
{
  task_result result = TASK_FREE;

  dprintf("request notify\n");
  if(t->flag & DLGF_THREAD){
    dprintf("wait for thread complete.\n");
    pthread_join(t->t_info, NULL);
    t->flag &= ~ DLGF_THREAD;
  }
  if(t->err){
    St_SetMsg(t->w, strerror(t->err));
  }

  if(t->v->notify){
    result = (*t->v->notify)(t);
  }else if(t->v->done){
    result = (*t->v->done)(t);
  }

  if(result == TASK_FREE){
    FreeTask(t);
  }
}

void taskCbDone(DfTask *t)
{
  dprintf("request done\n");
  if(t->flag & DLGF_THREAD){
    dprintf("wait for thread complete.\n");
    pthread_join(t->t_info, NULL);
    t->flag &= ~ DLGF_THREAD;
  }
  if(t->v->done){
    (*t->v->done)(t);
  }
  FreeTask(t);
}

void taskCbNone(DfTask *t)
{
  return;
}

void FreeTask(DfTask *t)
{

  dprintf("Free task.\n");  
  ListDel(&t->link);
  Str_Free(&t->cwd, 0);
  Str_Free(&t->input, 0);
  Str_Free(&t->dst, 0);
  bFree(t->files);
  bFree(t);
}
