#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <dirent.h>

#include <fcntl.h>
#include <unistd.h>
#include <errno.h> 
#include <sys/types.h> 
#include <sys/stat.h> 
#include <pthread.h>
#include <iconv.h>

#include <X11/Intrinsic.h>
#include <X11/keysym.h>
#include <X11/StringDefs.h>

#include "config.h"
#include "str.h"
#include "dftype.h"
#include "list.h"
#include "buffer.h"
#include "filer.h"
#include "task.h"
#include "dialog.h"
#include "view.h"
#include "status.h"
#include "mem.h"
#include "cmd.h"
#include "cmdinput.h"
#include "dfxfunc.h"
#include "xutil.h"
#include "misc.h"
#include "dfval.h"
#include "cdetect.h"
#include "conv.h"

#define VIEWER_READSIZE 4096

typedef enum{
  NOT_OPEN,
  OPENED,
  CLOSED
}view_stat;

struct _dfViewer{
  DfBufCommon b;
  int fd;
  view_stat vs;
  DfStr raw;
  DfStr view;
  int charset;
  struct lines l;
  CharConv ctx;
  DfStr find;
  int find_line;
  int find_col;
};

static int Vw_Free(void *v);
static int Vw_Active(Widget w, void *ptr);
static int Vw_Inactive(Widget w, void *ptr);
static int Vw_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv);

static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur);
static int Vw_AdjustNewLine(DfViewer *v);

static void Vw_Close(DfViewer *v);

static int Vw_read(DfViewer *v, int *err);
static int Vw_read2conv(DfViewer *v);
static int Vw_read4line(DfViewer *v, int line);

static int vw_upCursor(Widget w, DfViewer *v);
static int vw_downCursor(Widget w, DfViewer *v);
static int vw_upPage(Widget w, DfViewer *v);
static int vw_downPage(Widget w, DfViewer *v);
static void setPosition(Widget w, DfViewer *v, int n);
static int doSearch(Widget w, DfViewer  *v);
static int doNextFind(Widget w, DfViewer *v);
static int doPrevFind(Widget w, DfViewer *v);
static int doFindForward(Widget w, DfViewer *v, int col, int line);
static int doFindBackword(Widget w, DfViewer *v, int col, int line);

static int viewIsValid(DfViewer *v, int col, int line, int *pcol, int *pline);
static task_result viewer_invoke(DfTask *t);
static task_result viewer_done(DfTask *t);
static int Vw_DrawWait(void *ptr, struct DfDrawInfo *di);

static const DfVerb2 Vw_VerbWait = {
  {
    Vw_Free,
    Vw_Active,
    Vw_Inactive,
    BF2_KeyPress,
    BF_ResizeNop,
    Vw_DrawWait,
  },
  BF2_DrawCaption,
  drawSingleItem,
  Vw_Process
};

static const DfVerb2 Vw_Verb = {
  {
    Vw_Free,
    Vw_Active,
    Vw_Inactive,
    BF2_KeyPress,
    BF_ResizeNop,
    BF2_Draw,
  },
  BF2_DrawCaption,
  drawSingleItem,
  Vw_Process
};


struct viewer_ext{
  const char *filename;
};

static const task_vtable view_task =
{
  viewer_invoke,
  NULL,
  viewer_done,
  NULL
};


int Vw_Open(Widget w, DfViewer *v, const char *filename)
{
  DfTask *t;
  struct viewer_ext *ext;
  int r;

  Str_InitStr(&v->b.caption, filename);

  t = MakeTask(w, GetBuffer(v), 0, 0, sizeof(struct viewer_ext));
  if(t == NULL){
    return errno;
  }

  ext = t->extend;
  ext->filename = bMalloc(strlen(filename) + 1);
  if(!ext->filename){
    r = errno;
    FreeTask(t);
    return r;
  }
  strcpy(ext->filename, filename);

  t->v = &view_task;

  (*t->v->invoke)(t);

  return 0;
}

static void *operation_thread(void *ptr)
{
  DfTask *t = ptr;
  DfViewer *v = (DfViewer *)t->owner;
  struct viewer_ext *e = t->extend; 
  enum CHARSET code;
  int fd;
  int nl;
  int len;
  int err;

  fd = open(e->filename,  O_RDONLY);
  if(fd == -1){
    t->err = errno;

    bFree(e->filename);
    DlgSendEvent(t, taskCbDone);
    return NULL;
  }
  bFree(e->filename);

  v->fd = fd;
  v->vs = OPENED;
  v->l.num = 0;
#if HAVE_ICONV
  v->ctx.ictx = NULL;
#endif

  len = Vw_read(v, &err);
  if(len < 0){
    t->err = err;

    DlgSendEvent(t, taskCbDone);
    return NULL;
  }

  if(len == 0){
    Vw_Close(v);
    t->err = 0;

    DlgSendEvent(t, taskCbDone);
    return NULL;
  }

  code = CodeDetect(Str_Get(&v->raw), Str_Length(&v->raw), &nl);
  v->ctx.charset = code;
  SetupEncoding(&v->ctx, &v->raw, &v->view);
  St_SetMsg(dfx->w, code_sets[code]);

  v->ctx.newline = nl;

  CharsetConvert(&v->ctx, &v->l);
  Vw_AdjustNewLine(v);

  t->err = 0;

  DlgSendEvent(t, taskCbDone);

  return NULL;
}

static task_result viewer_invoke(DfTask *t)
{
  t->flag |= DLGF_THREAD;
  pthread_create(&t->t_info, NULL, operation_thread, t);

  return TASK_LEAVE;
}

static task_result viewer_done(DfTask *t)
{
  DfBuf *b = t->owner;

  b->v = (DfVerb*)&Vw_Verb;
  if(!t->err){
    RedrawFrame(t->w, b, &b->rc);
  }else{
    St_SetMsg(t->w, strerror(t->err));
    KillBuffer(t->w, b);
  }

  return TASK_FREE;
}

static void Vw_Close(DfViewer *v)
{
  ConvertClose(&v->ctx);
  close(v->fd);

  v->vs = CLOSED;
}


DfViewer *Vw_Create(DfBuf *parent)
{
  DfViewer *v;
  DfBufCommon *cb;
  DfBuf *b;

  v = MakeBuffer(sizeof(DfViewer), parent, DT_VIEWER, &Vw_VerbWait);
  if(!v){
    return NULL;
  }

  cb = &v->b;
  b = &cb->b;

  DfBufCommon_Init(cb);

  v->vs = NOT_OPEN;
  Str_InitNull(&v->raw);
  Str_InitNull(&v->view);
  Str_InitNull(&v->find);
  v->l.index = bMalloc(4096);
  v->l.index[0].pos = 0;
  v->l.index[0].len = 0;
  v->l.num = 0;
  v->l.alloced = ALLOC_LINES;
  v->charset = 0;
  v->find_line = 0;;
  v->find_col = 0;

  return v;
}

static int Vw_Free(void *p)
{
  DfViewer *v = p;

  if(v->vs == OPENED){
    close(v->fd);
  }
  BufComon_Free(&v->b);
  Str_Free(&v->raw, 0);
  Str_Free(&v->view, 0);
  bFree(v->l.index);
  bFree(v);

  return 0;
}

static int Vw_Active(Widget w, void *ptr)
{
  CmdSetTable(CMD_VIEWER);

  return 0;
}

static int Vw_Inactive(Widget w, void *ptr)
{
  return 0;
}

static int Vw_Process(Widget w, void *ptr, const DfCmdStr *c, const char **argv)
{
  DfViewer *v;
  int cmd;

  v = ptr;
  cmd = c->builtin;

  switch(cmd){
  case QUIT:
  case CLOSE:
    KillBuffer(w, GetBuffer(v));
    break;

  case CHCHARSET:
    v->ctx.charset = NextCharset(v->ctx.charset);
    Str_SetLength(&v->view, 0);
    v->l.num = 0;
    SetupEncoding(&v->ctx, &v->raw, &v->view);
    St_SetMsg(dfx->w, code_sets[v->ctx.charset]);
    CharsetConvert(&v->ctx, &v->l);
    Vw_AdjustNewLine(v);
    RedrawFrame(w, GetBuffer(v), &v->b.b.rc);
    break;
  case CHNEWLINE:
    switch(v->ctx.newline){
    case NEWLINE_LF:
    case NEWLINE_CR:
      v->ctx.newline++;
      break;
    case NEWLINE_CRLF:
      v->ctx.newline = NEWLINE_LF;
    }
    SetupNewline(&v->ctx);
    Vw_AdjustNewLine(v);
    RedrawFrame(w, GetBuffer(v), &v->b.b.rc);
    break;
  case VIEWFIND:
    doSearch(w, v);
    break;
  case VIEWFINDNEXT:
    doNextFind(w, v);
    break;
  case VIEWFINDPREV:
    doPrevFind(w, v);
    break;
  case CURUP:
    vw_upCursor(w, v);
    break;
  case CURDOWN:
    vw_downCursor(w, v);
    break;
  case PAGEDOWN:
    vw_downPage(w, v);
    break;
  case PAGEUP:
    vw_upPage(w, v);
    break;
  default:
    return CREQ_IGNORE;
  }

  return CREQ_DONE;
}

static int drawLinePart(Widget w, Display *display, GC gc, int *dx, int dy, char *str, int str_len, int width)
{
  char *find = str;
  int len = str_len;
  int fw;
  int x = *dx;

  while(str_len){
    switch(*find){
    case '\t':
      len = find - str;
      XmbDrawImageString(display, XtWindow(w), vFontSet, gc, x, dy + vnFontBase, str, len);
      fw =  XmbTextEscapement(vFontSet, str, len) + vnTabWidth;
      x += fw;
      x = (x / vnTabWidth) * vnTabWidth;
      if(width < x){
	return 1; /* done */
      }

      len = str_len - 1;
      if(len == 0){
	*dx = x;
	return 0;
      }

      str = find + 1;
    }
    str_len--;
    find++;
  }

  if(len){
    fw =  XmbTextEscapement(vFontSet, str, len);
    XmbDrawImageString(display, XtWindow(w), vFontSet, gc, x, dy + vnFontBase, str, len);
    x += fw;
  }
  *dx = x;

  return 0;
}

static int comp_position(int line0, int col0, int line1, int col1)
{
  if(line0 < line1){
    return -1;
  }
  if(line0 > line1){
    return 1;
  }

  return col0 - col1;
}

static inline void swap_byte(char *a, char *b)
{
  char x = *a;
  *a = *b;
  *b = x;
}

static void drawSingleItem(void *p, struct DfDrawInfo *di, int cur)
{
  DfViewer *v;
  char *str;
  XRectangle rc;
  int left;
  int len;
  int x;
  int start_col;
  int start_line;
  int end_col;
  int end_line;
  char *find_str;
  int now_col;

  v = p;

  /* check */
  Vw_read4line(v, cur);

  str = Str_Get(&v->view) + v->l.index[cur].pos;
  calcCursorRect(&v->b, cur, &rc);
  x = rc.x;

  left = v->l.index[cur].len;

  find_str = Str_Get(&v->find);
  if(find_str == NULL){
    XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
    XSetBackground(di->d, di->gc, vOptions.colors[COLOR_BK]);
    drawLinePart(di->w, di->d, di->gc, &x, rc.y, str, left, rc.x + rc.width);
    return;
  }

  start_col = v->find_col;
  start_line = v->find_line;
  now_col = 0;

  viewIsValid(v, start_col + strlen(find_str), start_line, &end_col, &end_line);

  /* chech range. does text overwrap find strings. */
  if(comp_position(cur, 0, end_line, end_col) <= 0 && comp_position(cur, left, start_line, start_col) >= 0){
    /* draw til find_str */
    if(comp_position(cur, 0, start_line, start_col) < 0){
      XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
      XSetBackground(di->d, di->gc, vOptions.colors[COLOR_BK]);
      drawLinePart(di->w, di->d, di->gc, &x, rc.y, str, start_col, rc.x + rc.width);
      left -= start_col;
      str += start_col;
      now_col += start_col;
    }

    /* draw find_str */
    if(comp_position(start_line, start_col, cur, now_col) <= 0 && comp_position(cur, now_col, end_line, end_col) <= 0){
      if(cur == end_line){
	len = end_col - now_col;
      }else{
	len = left;
      }
      XSetForeground(di->d, di->gc, vOptions.colors[COLOR_BK]);
      XSetBackground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
      drawLinePart(di->w, di->d, di->gc, &x, rc.y, str, len, rc.x + rc.width);
      left -= len;
      str += len;
    }
  }

  /* draw from find_str to end of line */
  XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
  XSetBackground(di->d, di->gc, vOptions.colors[COLOR_BK]);
  drawLinePart(di->w, di->d, di->gc, &x, rc.y, str, left, rc.x + rc.width);

}

static int Vw_read(DfViewer *v, int *err)
{
  char buf[VIEWER_READSIZE];
  int len;

  if(err){
    err = 0;
  }
  len = read(v->fd, buf, VIEWER_READSIZE);
  if(len == 0){
    Vw_Close(v);
    return 0;
  }

  if(len == -1){
    if(err){
      *err = errno;
    }
    Vw_Close(v);
    return -1;
  }

  if(!Str_AddLen(&v->raw, buf, len)){
    if(err){
      *err = ENOMEM;
    }
    return -1;
  }

  return len;
}

static int Vw_read2conv(DfViewer *v)
{
  int len;
  int err;

  len = Vw_read(v, &err);

  if(len < 0){
    return 1;
  }

  if(len){
    CharsetConvert(&v->ctx, &v->l);
    Vw_AdjustNewLine(v);
  }

  return 0;
}

static int Vw_read4line(DfViewer *v, int line)
{
  /* check */
  while(v->b.nItems <= line + 1 && v->vs == OPENED){
    /* add */
    if(Vw_read2conv(v)){
      break;
    }
  }
  return 0;
}

int Vw_AdjustNewLine(DfViewer *v)
{
  int n;

  n = v->l.num;

  v->l.index[n].len = Str_Length(&v->view) - v->l.index[n].pos;
  if(v->l.index[n].len == 0){
    v->b.nItems = v->l.num;
  }else{
    v->b.nItems = v->l.num + 1;
  }
  BF2_CalcCursors(&v->b);

  return 0;
}

static int vw_upCursor(Widget w, DfViewer *v)
{
  int n;

  n = v->b.nStart;
  if(n){
    setPosition(w, v, n - 1);
  }

  return 0;
}

static int vw_downCursor(Widget w, DfViewer *v)
{
  int n;

  n = v->b.nStart;
  setPosition(w, v, n + 1);

  return 0;
}

static int vw_upPage(Widget w, DfViewer *v)
{
  int n;

  n = v->b.b.rc.height / vnFontHeight - 1;
  switch(n){/* :D  */
  case 0:
    break;
  default:
    n--;
  }

  if(v->b.nStart < n){
    n = 0;
  }else{
    n = v->b.nStart - n;
  }
  setPosition(w, v, n);

  return 0;
}

static int vw_downPage(Widget w, DfViewer *v)
{
  int n;

  n = v->b.b.rc.height / vnFontHeight - 1;
  switch(n){/* :D  */
  case 0:
    break;
  default:
    n--;
  }

  setPosition(w, v, v->b.nStart + n);

  return 0;
}


static void setPosition(Widget w, DfViewer *v, int n)
{
  int pre;
  int num;
  int scroll;
  GC gc;
  int max;
  XRectangle rc;
  XRectangle rc_gap;
  struct DfDrawInfo di;

  if(n == v->b.nStart){
    return;
  }
  if(n < 0){
    n = 0;
  }

  GetClientRect(GetBuffer(v), &rc);
  rc.y += vnFontHeight;
  rc.height -= vnFontHeight;

  num = rc.height / vnFontHeight;

  pre = v->b.nStart;
  Vw_read4line(v, n + num);
  max = v->b.nItems;

  /* adjust bottom */
  if(max < n + num){
    if(max - num < 0){
      n = 0;
    }else{
      n = max - num;
    }
  }

  if(n == v->b.nStart){
    return;
  }

  if(v->b.nStart < n){
    /* down (roll up) */
    scroll = n - v->b.nStart;
  }else{
    /* up (roll down) */
    scroll = v->b.nStart - n;
  }
  v->b.nStart = n;
  if(num <= scroll){
    RedrawFrame(w, GetBuffer(v), &rc);
    return;
  }

  gc = GetBufferGC(w, GetBuffer(v));
  rc_gap = rc;

  di.d = XtDisplay(w);
  di.w = w;
  di.gc = gc;

  ScrollRect(&di, &rc, &rc_gap, (pre - n) * vnFontHeight);
  DrawGap(&di, &v->b, &rc, &rc_gap);
  ClearClip(XtDisplay(w), gc);
}

static char viewGetChar(DfViewer *v, int line, int col)
{
  char *str;

  if(v->l.num < line){
    return 0;
  }
  while(v->l.index[line].len <= col){
    col -= v->l.index[line].len;
    line++;
    if(v->l.num < line){
      return 0;
    }
  }

  str = Str_Get(&v->view) + v->l.index[line].pos;
  return str[col];
}

static int findLastChar(const char *p, char ch, int len, int idx)
{
  int n = len - 1;

  if(len < 3){
    return 1;
  }

  idx = len - idx - 1;
  while(n){
    n--;
    if(p[n] == ch){
      return len - n - 1;
    }
  }
  return len;
}

static int findFirstChar(const char *p, char ch, int len, int idx)
{
  int n = 1;

  if(len < 3){
    return 1;
  }

  /*
   *  01234
   *  type
   *  4123
   */
  while(n < len){
    if(p[n] == ch){
      return n;
    }
    n++;
  }
  return len;
}

static int viewIsValid(DfViewer *v, int col, int line, int *pcol, int *pline)
{
  while(v->l.index[line].len <= col){
    col -= v->l.index[line].len;
    line++;

    while(v->l.num <= line + 1){
      if(v->vs != OPENED){
	return 0;
      }
      /* add */
      if(Vw_read2conv(v)){
	return 0;
      }
    }
  }

  if(pcol){
    *pcol = col;
  }
  if(pline){
    *pline = line;
  }

  return 1;
}

static int doNextFind(Widget w, DfViewer *v)
{
  int col = v->find_col;
  int line = v->find_line;

  if(Str_Get(&v->find) == NULL){
    return 0;
  }

  if(!viewIsValid(v, col + Str_Length(&v->find) + 1, line, &col, &line)){
    return 0;
  }

  return doFindForward(w, v, col + 1, line);
}

static int doPrevFind(Widget w, DfViewer *v)
{
  int col = v->find_col;
  int line = v->find_line;

  if(Str_Get(&v->find) == NULL){
    return 0;
  }

  while(col == 0){
    if(line == 0){
      return 0;
    }
    line--;
    col += v->l.index[line].len;
  }
  if(col){
    col--;
  }

  return doFindBackword(w, v, col, line);
}

static void findRedraw(Widget w, DfViewer *v, int from, int to)
{
  struct DfDrawInfo di;
  int top = v->b.nStart;
  int bottom = top + v->b.b.rc.height / vnFontHeight - 1;

  if(to < top && bottom < from){
    return;
  }

  if(from < top){
    from = top;
  }
  if(bottom < to){
    to = bottom;
  }

  while(from <= to){
    di.w = w;
    di.d = XtDisplay(w);
    di.gc = GetBufferGC(w, GetBuffer(v));
    drawSingleItem(v, &di, from);
    from++;
  }
}

static void findDone(Widget w, DfViewer *v, int line, int col)
{
  int eol;
  int erase_line1;
  int erase_line2;
  int n;

  n = v->b.b.rc.height / vnFontHeight - 1;

  erase_line1 = v->find_line;
  viewIsValid(v, v->find_col + Str_Length(&v->find), erase_line1, NULL, &erase_line2);
  viewIsValid(v, col + Str_Length(&v->find), line, NULL, &eol);

  v->find_line = line;
  v->find_col = col;
  setPosition(w, v, line);

  findRedraw(w, v, line, eol);
  findRedraw(w, v, erase_line1, erase_line2);
}


static int doFindForward(Widget w, DfViewer *v, int col, int line)
{
  int idx;
  int len;
  char *p;
  char ch;

  p = Str_Get(&v->find);
  len = Str_Length(&v->find);
  idx = len - 1;

  while(line < v->l.num){
    if(!viewIsValid(v, col + idx, line, NULL, NULL)){
      break;
    }
    ch = viewGetChar(v, line, col + idx);
    if(p[idx] == ch){
      if(idx == 0){
	findDone(w, v, line, col);
	St_SetMsg(w, vOptions.msg[STM_VWFOUND]);
	return 1;
      }
      idx--;
    }else{
      col += findLastChar(p, ch, len, idx);
      idx = len - 1;
      if(!viewIsValid(v, col, line, &col, &line)){
	break;
      }
    }
  }

  St_SetMsg(w, vOptions.msg[STM_VWNOTFOUND]);
  return 0;
}

static int doFindBackword(Widget w, DfViewer *v, int col, int line)
{
  int idx;
  int len;
  char *p;
  char ch;
  int back;

  p = Str_Get(&v->find);
  len = Str_Length(&v->find);
  idx = 0;

  for(;;){
    ch = viewGetChar(v, line, col + idx);
    if(p[idx] == ch){
      idx++;
      if(idx == len){
	findDone(w, v, line, col);
	St_SetMsg(w, vOptions.msg[STM_VWFOUND]);
	return 1;
      }
    }else{
      back = findFirstChar(p, ch, len, idx);
      while(col < back){
	if(line){
	  line--;
	  col += v->l.index[line].len;
	}else{
	  goto NOTFOUND;
	}
      }
      col -= back;
      idx = 0;
    }
  }
NOTFOUND:
  St_SetMsg(w, vOptions.msg[STM_VWNOTFOUND]);

  return 0;
}


static task_result findInvoke(DfTask *t)
{
  DfViewer *v = (DfViewer*)t->owner;
  int nFindLen = Str_Length(&v->find);
  int from;
  int to;

  Str_Move(&v->find, &t->input);
  if(!doFindForward(t->w, v, v->find_col, v->find_col)){
    /* clear */
    Str_Free(&v->find, 0);
    /* redraw */
    if(nFindLen){
      from = v->find_line;
      viewIsValid(v, v->find_col + nFindLen, from, NULL, &to);
      findRedraw(dfx->w, v, from, to); 
    }
  }

  return TASK_FREE;
}

static const task_vtable vtable_search = {
  findInvoke,
  NULL,
  commonDone,
  NULL,
};

static int doSearch(Widget w, DfViewer  *v)
{
  DfTask *t;
  DfInputDialog *d;

  t = MakeTask(w, GetBuffer(v), DO_ISEARCH, 0, 0);
  if(!t){
    return 0;
  }

  t->v = &vtable_search;

  d = MakeInputDialog(t, "", inputKeyProc);
  ShowInputDialog(d, CAP_ISEARCH);

  return 0;
}

static int Vw_DrawWait(void *ptr, struct DfDrawInfo *di)
{
  DfBufCommon *b = ptr;
  DfVerb2 *v = (DfVerb2*)b->b.v;;
  int x;
  int y;
  const char *msg;

  (*v->draw_caption)(di->w, b, di->d, di->gc);

  x = b->b.rc.x;
  y = b->b.rc.y + vnFontBase + vnFontHeight;
  msg = vOptions.msg[STM_SCANNING];
  XSetForeground(di->d, di->gc, vOptions.colors[COLOR_FILE]);
  XmbDrawString(di->d, XtWindow(di->w), vFontSet, di->gc, x, y, msg, strlen(msg));

  return 0;
}
