
#include "defs.h"

#include "dictbar.h"
#include "dicts.h"
#include "ebook.h"
#include "jcode.h"
#include "libxml/tree.h"
#include "limits.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "preferences.h"
#include "selection.h"
#include "toolbar.h"

static void dicts_cell_edited(GtkCellRendererText *cell, const gchar *path_string, const gchar *new_text, gpointer data)
{
    gchar *old_text;
    gint column = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(cell), "column"));
    GtkTreeIter iter;
    GtkTreePath *path = gtk_tree_path_new_from_string(path_string);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &iter, path);
    gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, column, &old_text, -1);
    g_free(old_text);
    gtk_tree_store_set(GTK_TREE_STORE(dicts.store), &iter, column, new_text, -1);
    gtk_tree_path_free(path);
}

static gboolean dicts_row_drop_possible(GtkTreeDragDest *drag_dest, GtkTreePath *dest, GtkSelectionData *selection_data)
{
    GtkTreeSelection *selection;
    GtkTreeIter iter;
    GtkTreePath *src;

    if(!dest) return FALSE;

    if(GTK_TREE_STORE(drag_dest) == GTK_TREE_STORE(dicts.store))
        selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view));
    else
        return FALSE;

    if(!gtk_tree_selection_get_selected(selection, NULL, &iter))
        return FALSE;

    src = gtk_tree_model_get_path(GTK_TREE_MODEL(drag_dest), &iter);
    return (gtk_tree_path_get_depth(src) == gtk_tree_path_get_depth(dest));
}

static void dicts_item_move(GtkWidget *widget, gpointer data, gboolean up)
{
    GtkTreeIter iter, iter2;
    GtkTreeSelection *selection;
    GtkTreePath *path, *path_orig;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view));
    if(gtk_tree_selection_get_selected(selection, NULL, &iter) == FALSE)
    {
        LOG(LOG_WARNING, _("No dictionary selected"));
        return;
    }
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(dicts.store), &iter);
    path_orig = gtk_tree_path_copy(path);

    up ? gtk_tree_path_prev(path) : gtk_tree_path_next(path);
    if((gtk_tree_path_compare(path, path_orig) != 0) && gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &iter2, path))
    {
        my_gtk_tree_store_swap(dicts.store, &iter, &iter2);
        gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view)), &iter);
    }
    else if(gtk_tree_path_get_depth(path) > 1)
    {
        gtk_tree_path_up(path);
        if(gtk_tree_path_compare(path, path_orig))
        {
            gtk_tree_path_free(path_orig);
            path_orig = gtk_tree_path_copy(path);
            up ? gtk_tree_path_prev(path) : gtk_tree_path_next(path);
            if(gtk_tree_path_compare(path, path_orig))
            {
                GtkTreeIter parent;
                if(gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &parent, path))
                {
                    GtkTreeIter new_iter;
                    gchar *title;
                    gboolean editable;
                    BOOK_INFO *binfo;

                    gtk_tree_store_append(GTK_TREE_STORE(dicts.store), &new_iter, &parent);

                    gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), 
                                       &iter, 
                                       DICT_ALIAS, &title,
                                       DICT_BINFO, &binfo,
                                       DICT_EDITABLE, &editable,
                                       -1);

                    gtk_tree_store_set(dicts.store, &new_iter,
                                       DICT_ALIAS, title,
                                       DICT_BINFO, binfo,
                                       DICT_EDITABLE, editable,
                                       -1);
                    g_free(title);

                    gtk_tree_store_remove(GTK_TREE_STORE(dicts.store), &iter);

                    gtk_tree_view_expand_row(GTK_TREE_VIEW(dicts.view), path, TRUE);
                    gtk_tree_selection_select_iter(gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view)), &new_iter);

                }
            }
        }
    }
    gtk_tree_path_free(path);
    gtk_tree_path_free(path_orig);
}

static void dicts_item_up(GtkWidget *widget, gpointer data)
{
    dicts_item_move(widget, data, TRUE);
}

static void dicts_item_down(GtkWidget *widget,gpointer data)
{
    dicts_item_move(widget, data, FALSE);
}

static void dicts_item_remove(GtkWidget *widget, gpointer data)
{
    gboolean b;
    gchar *title;
    GtkTreeIter iter, iter2;
    GtkTreePath *path;
    GtkTreeSelection *selection;
    BOOK_INFO *binfo;
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view));
    if(!gtk_tree_selection_get_selected(selection, NULL, &iter))
        return;
    path = gtk_tree_model_get_path(GTK_TREE_MODEL(dicts.store), &iter);
    b = (gtk_tree_path_get_depth(path) == 1);
    while(b ? gtk_tree_model_iter_children(GTK_TREE_MODEL(dicts.store), &iter2, &iter) : TRUE)
    {
        GtkTreeIter *iter1 = b ? &iter2 : &iter;
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), iter1, DICT_ALIAS, &title, DICT_BINFO, &binfo, -1);
        gtk_tree_store_remove(dicts.store, iter1);
        ebook_free(binfo);
        g_free(title);
        if(!b) break;
    }
    if(b)
        gtk_tree_store_remove(GTK_TREE_STORE(dicts.store), &iter);
    gtk_tree_path_free(path);
}

static void dicts_add_group(GtkWidget *widget, gpointer data)
{
    gchar name[32];
    gint n = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(dicts.store), NULL);
    GtkTreeIter iter;

    sprintf(name, "Group %d", n + 1);
    gtk_tree_store_append(GTK_TREE_STORE(dicts.store), &iter, NULL);
    gtk_tree_store_set(GTK_TREE_STORE(dicts.store), &iter, DICT_ALIAS, name, DICT_EDITABLE, TRUE, -1);
}

static void dicts_search_recursive(gchar *path, gint depth)
{
    static gboolean group_exists = FALSE;
    static gchar *dirname = NULL;
    static GtkTreeIter iter;
    const gchar *name;
    gchar fullpath[PATH_MAX];
    GDir *dir;
    GtkTreeIter iter1;
    g_strstrip(path);

    if(!(dir = g_dir_open(path, 0, NULL)))
    {
        LOG(LOG_CRITICAL, "Failed to open directory %s", path);
        return;
    }
    if(!depth)
    {
        dirname = g_strrstr(path, G_DIR_SEPARATOR_S);
        dirname = &(dirname[1]);
        dirname = g_strdup(dirname);
        group_exists = FALSE;
    }
    while((name = g_dir_read_name(dir)))
    {
        sprintf(fullpath, "%s%s%s", path, G_DIR_SEPARATOR_S, name);
        if(g_file_test(fullpath, G_FILE_TEST_IS_DIR))
        {
            if(depth < gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dicts.search_depth)))
                dicts_search_recursive(fullpath, depth + 1);
        }
        if(g_file_test(fullpath, G_FILE_TEST_IS_REGULAR))
        {
            if(!strcasecmp(name, "catalog") || !strcasecmp(name, "catalogs"))
            {
                gint i;
                BOOK_INFO *binfo;
                GtkTreePath *tree_path;
                for(i = 0; (binfo = ebook_load(path, i)); i++)
                {
                    if(!group_exists)
                    {
                        gtk_tree_store_append(GTK_TREE_STORE(dicts.store), &iter, NULL);
                        gtk_tree_store_set(GTK_TREE_STORE(dicts.store), &iter, DICT_ALIAS, dirname, DICT_EDITABLE, TRUE, -1);
                        group_exists = TRUE;
                    }
                    gtk_tree_store_append(GTK_TREE_STORE(dicts.store), &iter1, &iter);
                    gtk_tree_store_set(GTK_TREE_STORE(dicts.store), &iter1, DICT_ALIAS, binfo->title, DICT_BINFO, binfo, DICT_EDITABLE, TRUE, -1);
                    tree_path = gtk_tree_model_get_path(GTK_TREE_MODEL(dicts.store), &iter);
                    gtk_tree_view_expand_row(GTK_TREE_VIEW(dicts.view), tree_path, TRUE);
                    gtk_tree_path_free(tree_path);
                }
            }
        }
    }
    if(!depth)
    {
        g_free(dirname);
        dirname = NULL;
        group_exists = FALSE;
    }
    g_dir_close(dir);
}

static void dicts_browse_disk_cb(gchar *path)
{
    gtk_entry_set_text(GTK_ENTRY(dicts.path), path);
    gtk_widget_grab_focus(dicts.path);
}

void dicts_browse_disk(GtkWidget *widget, gpointer data)
{
    static gchar *path = NULL;
    GtkWidget *dlg = gtk_file_chooser_dialog_new("Select folder", GTK_WINDOW(dicts.dlg), GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER, 
                                                    GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
    browse_callback cb = (browse_callback)data;
    if(path)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(dlg), path);
    if(gtk_dialog_run(GTK_DIALOG(dlg)) == GTK_RESPONSE_ACCEPT)
    {
        g_free(path);
        path = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dlg));
        cb(path);
    }
    gtk_widget_destroy(dlg);
}

static void dicts_search_disk(GtkWidget *widget, gpointer data)
{
    if(!gtk_entry_get_text_length(GTK_ENTRY(dicts.path)))
        return;
    gchar *dirname = (gchar*)gtk_entry_get_text(GTK_ENTRY(dicts.path));
    dirname = g_strdup(dirname);
    dicts_search_recursive(dirname, 0);
    g_free(dirname);
}

static gboolean dicts_drag_data_received(GtkTreeDragDest *drag_dest, GtkTreePath *dest, GtkSelectionData  *selection_data)
{
    return FALSE;
}

#define G_NODE(node) ((GNode *)node)
#define VALID_ITER(iter, tree_store) (iter!= NULL && iter->user_data != NULL && tree_store->stamp == iter->stamp)

// gtk_tree_store_swap: copied from GTK+-2.2
void my_gtk_tree_store_swap(GtkTreeStore *tree_store, GtkTreeIter *a, GtkTreeIter *b)
{
    GNode *tmp, *node_a, *node_b, *parent_node, *a_prev, *a_next, *b_prev, *b_next;
    gint i, a_count, b_count, length, *order;
    GtkTreePath *path_a, *path_b;
    GtkTreeIter parent;

    g_return_if_fail(GTK_IS_TREE_STORE (tree_store));
    g_return_if_fail(VALID_ITER (a, tree_store));
    g_return_if_fail(VALID_ITER (b, tree_store));

    node_a = G_NODE (a->user_data);
    node_b = G_NODE (b->user_data);

    // basic sanity checking
    g_return_if_fail(node_a != node_b);

    path_a = gtk_tree_model_get_path(GTK_TREE_MODEL (tree_store), a);
    path_b = gtk_tree_model_get_path(GTK_TREE_MODEL (tree_store), b);

    g_return_if_fail(path_a && path_b);

    gtk_tree_path_up(path_a);
    gtk_tree_path_up(path_b);

    if((gtk_tree_path_get_depth(path_a) != 0) || (gtk_tree_path_get_depth(path_b) != 0))
    {
        if (gtk_tree_path_compare (path_a, path_b))
        {
            gtk_tree_path_free (path_a);
            gtk_tree_path_free (path_b);
            g_warning ("Given childs are not in the same level\n");
            return;
        }
    }

    if(gtk_tree_path_get_depth(path_a) == 0)
    {
        parent_node = G_NODE (tree_store->root);
    }
    else
    {
        gtk_tree_model_get_iter (GTK_TREE_MODEL (tree_store), &parent, path_a);
        parent_node = G_NODE (parent.user_data);
    }

    gtk_tree_path_free (path_b);

    // old links which we have to keep around
    a_prev = node_a->prev;
    a_next = node_a->next;

    b_prev = node_b->prev;
    b_next = node_b->next;

    // fix up links if the nodes are next to eachother
    if(a_prev == node_b)    a_prev = node_a;
    if(a_next == node_b)    a_next = node_a;

    if(b_prev == node_a)    b_prev = node_b;
    if(b_next == node_a)	b_next = node_b;

    // counting nodes
    tmp = parent_node->children;
    i = a_count = b_count = 0;
    while (tmp)
    {
        if(tmp == node_a)   a_count = i;
        if(tmp == node_b)   b_count = i;
        tmp = tmp->next;
        i++;
    }
    length = i;

    // hacking the tree
    a_prev ? (a_prev->next = node_b) : (parent_node->children = node_b);
    if(a_next) a_next->prev = node_b;
    b_prev ? (b_prev->next = node_a) : (parent_node->children = node_a);
    if(b_next)  b_next->prev = node_a;

    node_a->prev = b_prev;
    node_a->next = b_next;
    node_b->prev = a_prev;
    node_b->next = a_next;

    // emit signal
    order = g_new (gint, length);
    for (i = 0; i < length; i++)
    {
        if(i == a_count)
            order[i] = b_count;
        else if(i == b_count)
            order[i] = a_count;
        else
            order[i] = i;
    }

    gtk_tree_model_rows_reordered(GTK_TREE_MODEL (tree_store), path_a, (gtk_tree_path_get_depth(path_a) == 0) ? NULL : &parent, order);
    gtk_tree_path_free(path_a);
    g_free(order);
}

void my_gtk_combo_box_set_active_text(GtkComboBox *combo, const gchar *text)
{
    if(!text) return;
    gchar *str;
    gint i = 0;
    GtkTreeIter iter;
    GtkTreeModel *model = gtk_combo_box_get_model(combo);

    if(!gtk_tree_model_get_iter_first(model, &iter)) return;
    do {
        gtk_tree_model_get(model, &iter, 0, &str, -1);
        if(!g_ascii_strcasecmp(str, text))
        {
            g_free(str);
            gtk_combo_box_set_active(combo, i);
            break;
        }
        g_free(str);
        i++;
    } while(gtk_tree_model_iter_next(model, &iter));
    if(gtk_combo_box_get_active(combo) < 0)
        gtk_combo_box_set_active(combo, 0);
}

void dicts_close_dlg(GtkWidget *w, gpointer data)
{
    dictbar.group = gtk_combo_box_get_active_text(GTK_COMBO_BOX(dictbar.combo));
    gtk_grab_remove(dicts.dlg);
    gtk_widget_destroy(dicts.dlg);
    popupwnd_close(NULL, NULL);
    dictbar_update();
    dicts_save();
    dicts_info(NULL);
    dicts.show_info = FALSE;
}

static void dicts_info_close(GtkWidget *w, gpointer data)
{
    dicts_info(NULL);
    dicts.show_info = FALSE;
}

void dicts_info(BOOK_INFO *binfo)
{
    gchar buff[128];
    GtkWidget *btn;
    static GtkWidget *dlg = NULL, *vbox, *book_title, *book_path, *subbook_no, *app_path, *app_subbook_no, *searches;
    if(!binfo && dlg)
    {
        gtk_widget_destroy(dlg);
        dlg = NULL;
        return;
    }
    if(!dicts.show_info)
        return;
    if(binfo && !dlg)
    {
        dlg = gtk_dialog_new();
        gtk_window_set_title(GTK_WINDOW(dlg), "Book info");
        //gtk_widget_set_size_request(dlg, 300, -1);

        vbox = gtk_vbox_new(FALSE, 5);
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), vbox, TRUE, TRUE, 5);
        gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);

        gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("Name")), FALSE, FALSE, 0);
        gtk_editable_set_editable(GTK_EDITABLE(book_title = gtk_entry_new()), FALSE);
        gtk_box_pack_start(GTK_BOX(vbox), book_title, FALSE, FALSE, 0);

        gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("Path")), FALSE, FALSE, 0);
        gtk_editable_set_editable(GTK_EDITABLE(book_path = gtk_entry_new()), FALSE);
        gtk_widget_set_tooltip_text(GTK_WIDGET(book_path), binfo->path);
        gtk_box_pack_start(GTK_BOX(vbox), book_path, FALSE, FALSE, 0);

        gtk_box_pack_start (GTK_BOX(vbox), gtk_label_new(_("Subbook number")), FALSE, FALSE, 0);
        gtk_editable_set_editable(GTK_EDITABLE(subbook_no = gtk_entry_new()), FALSE);
        gtk_box_pack_start (GTK_BOX(vbox), subbook_no, FALSE, FALSE, 0);

        gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("Appendix path")), FALSE, FALSE, 0);
        gtk_editable_set_editable(GTK_EDITABLE(app_path = gtk_entry_new()), FALSE);
        gtk_box_pack_start(GTK_BOX(vbox), app_path, FALSE, FALSE, 0);

        gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("Appendix subbook number")), FALSE, FALSE, 0);
        gtk_editable_set_editable(GTK_EDITABLE(app_subbook_no = gtk_entry_new()), FALSE);
        gtk_box_pack_start(GTK_BOX(vbox), app_subbook_no, FALSE, FALSE, 0);

        gtk_box_pack_start(GTK_BOX(vbox), gtk_label_new(_("Search methods")), FALSE, FALSE, 0);
        searches = gtk_label_new("\n\n\n\n\n\n");
        gtk_misc_set_alignment(GTK_MISC(searches), 0, 0.5);
        gtk_box_pack_start(GTK_BOX(vbox), searches, FALSE, FALSE, 0);

        btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
        gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->action_area), btn, TRUE, TRUE, 0);
        g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dicts_info_close), NULL);

        gtk_window_set_transient_for(GTK_WINDOW(dlg), GTK_WINDOW(dicts.dlg));
        gtk_widget_show_all(dlg);
    }
    gtk_entry_set_text(GTK_ENTRY(book_title), binfo->title);
    gtk_entry_set_text(GTK_ENTRY(book_path), binfo->path);
    gtk_entry_set_text(GTK_ENTRY(app_path), binfo->appendix_path ? binfo->appendix_path : "");

    g_sprintf(buff, "%d", binfo->subbook_no);
    gtk_entry_set_text(GTK_ENTRY(subbook_no), buff);

    g_sprintf(buff, "%d", binfo->appendix_subbook_no);
    gtk_entry_set_text(GTK_ENTRY(app_subbook_no), buff);

    memset(buff, 0, sizeof(buff));
    if(eb_have_word_search(binfo->book))
        g_strlcat(buff, SEARCH_FORWARD, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    if(eb_have_endword_search(binfo->book))
        g_strlcat(buff, SEARCH_BACKWARD, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    if(eb_have_exactword_search(binfo->book))
        g_strlcat(buff, SEARCH_EXACTWORD, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    if(eb_have_keyword_search(binfo->book))
        g_strlcat(buff, SEARCH_KEYWORD, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    if(eb_have_menu(binfo->book))
        g_strlcat(buff, SEARCH_MENU, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    if(eb_have_copyright(binfo->book))
        g_strlcat(buff, SEARCH_COPYRIGHT, sizeof(buff));
    g_strlcat(buff, "\n", sizeof(buff));

    gtk_label_set_text(GTK_LABEL(searches), buff);
}

static void dicts_info_update()
{
    gint n;
    GtkTreeIter iter;
    GtkTreePath *path;
    GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view));
    BOOK_INFO *binfo;

    if(!gtk_tree_selection_get_selected(sel, NULL, &iter))
        return;

    path= gtk_tree_model_get_path(GTK_TREE_MODEL(dicts.store), &iter);
    n = gtk_tree_path_get_depth(path);
    gtk_tree_path_free(path);
    if(n != 2)
        return;

    gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_BINFO, &binfo, -1);
    dicts.show_info = TRUE;
    dicts_info(binfo);
}

static void dicts_item_info(GtkWidget *w, gpointer data)
{
    dicts_info_update();
}

static void dicts_selection_changed(GtkTreeSelection *selection, gpointer data)
{
    if(!dicts.show_info)
        return;
    dicts_info_update();
}

void dicts_show(GtkWidget *w, gpointer data)
{
    GtkWidget *vbox, *hbox, *vbox1, *vbox2, *btn, *frame, *scroll;

    GtkCellRenderer *renderer;
    GtkTreeSelection *select;
    GtkTreeDragDestIface *iface;

    dicts.dlg = gtk_dialog_new();
    gtk_window_set_title(GTK_WINDOW(dicts.dlg), "Books");
    gtk_window_set_position(GTK_WINDOW(dicts.dlg), GTK_WIN_POS_CENTER);
    g_signal_connect(G_OBJECT(dicts.dlg), "delete_event", G_CALLBACK(dicts_close_dlg), NULL);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dicts.dlg)->vbox), vbox, TRUE, TRUE, 0);

    hbox = gtk_hbox_new(FALSE, 10);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 0);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);

    scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_container_add(GTK_CONTAINER(frame), scroll);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    dicts.view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(dicts.store));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(dicts.view), FALSE);
    gtk_tree_view_expand_all(GTK_TREE_VIEW(dicts.view));
    gtk_container_add(GTK_CONTAINER(scroll), dicts.view);
    gtk_widget_set_size_request(dicts.view, -1, 300);

    select = gtk_tree_view_get_selection(GTK_TREE_VIEW(dicts.view));
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(dicts_selection_changed), NULL);

    iface = GTK_TREE_DRAG_DEST_GET_IFACE(dicts.store);
    iface->row_drop_possible = dicts_row_drop_possible;

    g_signal_connect(G_OBJECT(dicts.view), "drag_data_received", G_CALLBACK(dicts_drag_data_received), NULL);

    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(dicts.view), TRUE);

    renderer = gtk_cell_renderer_text_new();
    g_signal_connect(G_OBJECT(renderer), "edited", G_CALLBACK(dicts_cell_edited), NULL);
    g_object_set_data(G_OBJECT(renderer), "column", (gint *)DICT_ALIAS);

    gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(dicts.view), -1, _("Name"), renderer, "text", DICT_ALIAS, "editable", DICT_EDITABLE, NULL);

    vbox1 = gtk_vbox_new(TRUE, 5);
    gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
    vbox2 = gtk_vbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(vbox1), vbox2, TRUE, TRUE, 0);
    btn = toolbar_button(dicts_add_group, GTK_STOCK_ADD, _("Add group"), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox2), btn, FALSE, FALSE, 2);
    btn = toolbar_button(dicts_item_remove, GTK_STOCK_REMOVE, _("Remove item"), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox2), btn, FALSE, FALSE, 2);

    vbox2 = gtk_vbox_new(FALSE, 5);
    gtk_box_pack_start(GTK_BOX(vbox1), vbox2, TRUE, TRUE, 0);
    btn = toolbar_button(dicts_item_up, GTK_STOCK_GO_UP, _("Move item up"), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox2), btn, FALSE, FALSE, 2);
    btn = toolbar_button(dicts_item_down, GTK_STOCK_GO_DOWN, _("Move item down"), TRUE);
    gtk_box_pack_start(GTK_BOX(vbox2), btn, FALSE, FALSE, 2);
    btn = toolbar_button(dicts_item_info, GTK_STOCK_INFO, _("Show dictionary info"), TRUE);
    gtk_box_pack_end(GTK_BOX(vbox2), btn, FALSE, FALSE, 2);

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Path")), FALSE, FALSE, 2);

    gtk_box_pack_start(GTK_BOX(hbox), dicts.path = gtk_entry_new(), TRUE, TRUE, 2);
    gtk_widget_set_size_request(GTK_WIDGET(dicts.path), 200, -1);

    btn = gtk_button_new_with_label("...");
    g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dicts_browse_disk), (gpointer)dicts_browse_disk_cb);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new(_("Depth")), FALSE, FALSE, 2);

    dicts.search_depth = gtk_spin_button_new_with_range(1, 20, 1);
    gtk_box_pack_start(GTK_BOX(hbox), dicts.search_depth, FALSE, FALSE, 2);
    gtk_widget_set_tooltip_text(dicts.search_depth, "Specify search depth. 0 means to search only specified directory.");

    gtk_box_pack_start(GTK_BOX(hbox), btn = gtk_button_new_with_label(_("Search")), FALSE, FALSE, 2);
    g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dicts_search_disk), (gpointer)btn);

    btn = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dicts.dlg)->action_area), btn, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked", G_CALLBACK(dicts_close_dlg), NULL);

    selection_lookup_stop();
    gtk_window_set_transient_for(GTK_WINDOW(dicts.dlg), GTK_WINDOW(mainwnd.wnd));
    gtk_widget_show_all(dicts.dlg);

    dicts.show_info = FALSE;
    dicts_info(NULL);
}

gboolean dicts_check_binfo(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    gint n = gtk_tree_path_get_depth(path);
    gpointer *q = (gpointer*)data;
    if(n == 2)
    {
        BOOK_INFO *binfo = NULL;
        gtk_tree_model_get(GTK_TREE_MODEL(model), iter, DICT_BINFO, &binfo, -1);
        if(*q == (gpointer)binfo->book)
        {
            *q = (gpointer*)gtk_tree_model_get_path(GTK_TREE_MODEL(model), iter);
            return TRUE;
        }
    }
    return FALSE;
}

BOOK_INFO* dicts_find_binfo(EB_Book *book)
{
    gpointer q = (gpointer)book;
    GtkTreeIter iter;
    GtkTreePath *path;
    BOOK_INFO *binfo;
    gtk_tree_model_foreach(GTK_TREE_MODEL(dicts.store), dicts_check_binfo, &q);
    if(q == (gpointer)book)
        return NULL;
    path = (GtkTreePath*)q;
    gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &iter, path);
    gtk_tree_path_free(path);
    gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_BINFO, &binfo, -1);
    return binfo;
}

GtkTreePath* dicts_find_path(EB_Book *book)
{
    gpointer q = (gpointer)book;
    gtk_tree_model_foreach(GTK_TREE_MODEL(dicts.store), dicts_check_binfo, &q);
    if(q == (gpointer)book)
        return NULL;
    return (GtkTreePath*)q;
}

gboolean dicts_save_item(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    guchar *title;
    gint n = gtk_tree_path_get_depth(path);
    static xmlNodePtr group;
    if(n == 1)
    {
        xmlNodePtr node = (xmlNodePtr)data;
        gtk_tree_model_get(model, iter, DICT_ALIAS, &title, -1);
        group = xmlAddChild(node, xmlNewNode(NULL, (xmlChar*)"group"));
        xmlNewProp(group, (xmlChar*)"name", title);
        g_free(title);
    }
    else if(n == 2)
    {
        gchar buf[16];
        xmlNodePtr node = xmlAddChild(group, xmlNewNode(NULL, (xmlChar*)"dict"));
        BOOK_INFO *binfo;
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), iter, DICT_ALIAS, &title, DICT_BINFO, &binfo, -1);
        xmlNewProp(node, (xmlChar*)"name", (xmlChar*)title);
        xmlNewProp(node, (xmlChar*)"path", (xmlChar*)binfo->path);
        sprintf(buf, "%d", binfo->subbook_no);
        xmlNewProp(node, (xmlChar*)"subbook", (xmlChar*)buf);
        xmlNewProp(node, (xmlChar*)"appendix_path", (xmlChar*)(binfo->appendix_path ? binfo->appendix_path : " "));
        sprintf(buf, "%d", binfo->appendix_subbook_no);
        xmlNewProp(node, (xmlChar*)"appendix_subbook", (xmlChar*)buf);
        sprintf(buf, "%d", binfo->active ? 1 : 0);
        xmlNewProp(node, (xmlChar*)"active", (xmlChar*)buf);
        g_free(title);
    }
    return FALSE;
}

void dicts_save()
{
    gchar filename[PATH_MAX];
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"Dictionaries", NULL);
    xmlNewProp(doc->children, (xmlChar*)"active", (xmlChar*)gtk_combo_box_get_active_text(GTK_COMBO_BOX(dictbar.combo)));
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_DICTIONARIES);

    gtk_tree_model_foreach(GTK_TREE_MODEL(dicts.store), dicts_save_item, doc->children);
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

BOOK_INFO* dicts_load_dict(gchar **data)
{
    gboolean active = FALSE;
    gchar *fs_path, *book_path = NULL, *appendix_path = NULL;
    gint subbook_no = 0, appendix_subbook_no = 0, i;
    BOOK_INFO *binfo;
    for(i = 1; i < g_strv_length(data); i++)
    {
        if(!g_strcmp0(data[i - 1], "active"))
            active = atoi(data[i]);
        else if(!g_strcmp0(data[i - 1], "path") ? strlen(data[i]): FALSE)
            book_path = data[i];
        else if(!g_strcmp0(data[i - 1], "appendix_path") ? strlen(data[i]) : FALSE)
            appendix_path = data[i];
        else if(!g_strcmp0(data[i - 1], "subbook"))
            subbook_no = atoi(data[i]);
        else if(!g_strcmp0(data[i - 1], "appendix_subbook"))
            appendix_subbook_no = atoi(data[i]);
    }
    fs_path = unicode_to_fs(book_path);
    binfo = ebook_load(fs_path, subbook_no);
    g_free(fs_path);
    if(!binfo)
        return NULL;
    binfo->active = active;
    if(appendix_path)
        binfo->appendix_path = g_strdup(appendix_path);
    binfo->appendix_subbook_no = appendix_subbook_no;
    return binfo;
}

void dicts_startElement(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    static GtkTreeIter iter;
    if(!g_strcmp0((gchar*)name, "Dictionaries"))
        dictbar.group = g_strdup((gchar*)atts[1]);
    if(!g_strcmp0((gchar*)name, "group"))
    {
        gtk_tree_store_append(dicts.store, &iter, NULL);
        gtk_tree_store_set(dicts.store, &iter, DICT_ALIAS, g_strdup((gchar*)atts[1]), DICT_EDITABLE, TRUE, -1);
    }
    else if(!g_strcmp0((gchar*)name, "dict"))
    {
        BOOK_INFO *binfo = dicts_load_dict((gchar**)atts);
        if(binfo)
        {
            GtkTreeIter child;
            gtk_tree_store_append(dicts.store, &child, &iter);
            gtk_tree_store_set(dicts.store, &child, DICT_ALIAS, atts[1], DICT_BINFO, binfo, DICT_EDITABLE, TRUE, -1);
        }
    }
}

void dicts_load()
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_DICTIONARIES);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
        return;
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &dicts_startElement;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

