
#include "defs.h"

#include "dictbar.h"
#include "dicts.h"
#include "ebook.h"
#include "headword.h"
#include "hook.h"
#include "jcode.h"
#include "mainwnd.h"
#include "textview.h"

#ifndef MAX_BUFSIZE
#define MAX_BUFSIZE 65535
#endif

struct _search_method methods[] = {
    {SEARCH_METHOD_FORWARD,     SEARCH_FORWARD},
    {SEARCH_METHOD_BACKWARD,    SEARCH_BACKWARD},
    {SEARCH_METHOD_EXACTWORD,   SEARCH_EXACTWORD},
    {SEARCH_METHOD_KEYWORD,     SEARCH_KEYWORD},
    {0, NULL}
};

gint ebook_search_method()
{
    gint i;
    const gchar *method_name = gtk_combo_box_get_active_text(GTK_COMBO_BOX(mainwnd.combo_method));
    for(i = 0; methods[i].name != NULL; i++)
        if(!g_strcmp0(method_name, methods[i].name))
            return methods[i].code;
    return SEARCH_METHOD_UNKNOWN;
}

BOOK_INFO *ebook_load(const gchar *path, int subbook_no)
{
    gint subcount;
    gchar buff[512];
    BOOK_INFO *binfo;
    EB_Error_Code error_code;
    EB_Subbook_Code sublist[EB_MAX_SUBBOOKS];

    if(!path)
        return NULL;

    binfo = (BOOK_INFO*)g_new0(BOOK_INFO, 1);
    if(!binfo)
        return NULL;

    binfo->path = fs_to_unicode((gchar*)path);
    binfo->subbook_no = subbook_no;

    binfo->book = (EB_Book*)g_new0(EB_Book, 1);
    if(!binfo->book)
    {
        ebook_free(binfo);
        return NULL;
    }
    eb_initialize_book(binfo->book);
    error_code = eb_bind(binfo->book, path);
    if(error_code != EB_SUCCESS)
    {
        ebook_free(binfo);
        LOG(LOG_CRITICAL, "Failed to bind the book %s: %s", path, eb_error_message(error_code));
        return NULL;
    }
    error_code = eb_subbook_list(binfo->book, sublist, &subcount);
    if(error_code != EB_SUCCESS)
    {
        ebook_free(binfo);
        LOG(LOG_CRITICAL, "Failed to get a subbook list: %s", eb_error_message(error_code));
        return NULL;
    }
    if(subbook_no >= subcount)
    {
        ebook_free(binfo);
        return NULL;
    }
    error_code = eb_subbook_directory2(binfo->book, sublist[binfo->subbook_no], buff);
    if (error_code != EB_SUCCESS)
    {
        ebook_free(binfo);
        LOG(LOG_CRITICAL, "Failed to get the directory : %s", eb_error_message(error_code));
        return NULL;
    }
    binfo->subbook_dir = g_strdup(buff);
    error_code = eb_subbook_title2(binfo->book, sublist[binfo->subbook_no], buff);
    if(error_code != EB_SUCCESS)
    {
        ebook_free(binfo);
        LOG(LOG_CRITICAL, "Failed to get the title : %s", eb_error_message(error_code));
        return NULL;
    }
    binfo->title = iconv_convert(ENC_EUC_JP, ENC_UTF8, buff);
    if(!binfo->title)
    {
        ebook_free(binfo);
        return NULL;
    }
    error_code = eb_set_subbook(binfo->book, binfo->subbook_no);
    if(error_code != EB_SUCCESS)
    {
        ebook_free(binfo);
        LOG(LOG_CRITICAL, "Failed to get the title : %s", eb_error_message(error_code));
        return NULL;
    }
    binfo->available = TRUE;

    if(eb_have_font(binfo->book, EB_FONT_16))
        eb_set_font(binfo->book, EB_FONT_16);

    return binfo;
}

void ebook_free(BOOK_INFO *binfo)
{
    if(binfo->title)
        g_free(binfo->title);
    if(binfo->path)
        g_free(binfo->path);
    if(binfo->appendix_path)
        g_free(binfo->appendix_path);
    if(binfo->subbook_dir)
        g_free(binfo->subbook_dir);
    if(binfo->appendix)
        eb_finalize_appendix(binfo->appendix);
    if(binfo->book)
    {
        eb_finalize_book(binfo->book);
        g_free(binfo->book);
    }
    g_free(binfo);
}

gint ebook_guess_gaiji_size(EB_Book *book, gint font_h)
{
    gint size = EB_FONT_16;

    if(font_h < 24)
        size = EB_FONT_16;
    else if(font_h < 30)
        size = EB_FONT_24;
    else if(font_h < 48)
        size = EB_FONT_30;
    else
        size = EB_FONT_48;
    return size;
}

void ebook_clear_results(GSequence *results)
{
    GSequenceIter *begin, *end;
    begin = g_sequence_get_begin_iter(results);
    end = g_sequence_get_end_iter(results);
    g_sequence_remove_range(begin, end);
}

gchar *ebook_get_text(EB_Book *book, EB_Position *pos)
{
    gchar *p, text[MAX_BUFSIZE + 1];
    ssize_t len;
    EB_Error_Code error_code;

    error_code = eb_seek_text(book, pos);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to seek text : %s", eb_error_message(error_code));
        return NULL;
    }

    error_code = eb_read_text(book, NULL, &hooksets.text, NULL, MAX_BUFSIZE, text, &len);
    if (error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to read text : %s", eb_error_message(error_code));
        return NULL;
    }
    text[len] = '\0';
    p = malloc(len + 1);
    g_strlcpy(p, text, len + 1);
    return(p);
}

gchar *ebook_get_raw_text(EB_Book *book, gint page, gint offset)
{
    gchar *data;
    ssize_t len;
    EB_Position pos;
    EB_Error_Code error_code;

    data = g_strnfill(EB_SIZE_PAGE, 0);

    pos.page = page;
    pos.offset = offset;
    eb_seek_text(book, &pos);
    error_code = eb_read_rawtext(book, EB_SIZE_PAGE, data, &len);
    if(error_code != EB_SUCCESS || !len)
        return NULL;

    return data;
}

gchar **ebook_bitmap_to_xpm(const gchar *bitmap, gint width, gint height, gchar *color)
{
    gchar **xpm, *xpm_p;
    const unsigned char *bitmap_p = (const unsigned char *)bitmap;
    gint i, j, gaiji = 2;

    xpm = g_new(gchar *, height + 4 + gaiji);
    xpm[0] = g_strdup_printf("%d %d 2 1", width, height + gaiji);
    xpm[1] = g_strdup_printf(" 	c None");
    if(color == NULL)
        xpm[2] = g_strdup_printf(". 	c Black");
    else
        xpm[2] = g_strdup_printf(". 	c %s", color);

    for(i = 0; i < gaiji; i++)
    {
        xpm[i+3] = g_new(gchar, width + 1);
        memset(xpm[i+3], ' ', width);
        xpm[i+3][width] = '\0';
    }

    for(; i < height + 2; i++)
    {
        xpm[i+3] = g_new(gchar, width + 1);
        xpm_p = xpm[i+3];
        for (j = 0; j + 7 < width; j += 8, bitmap_p++)
        {
            *xpm_p++ = (*bitmap_p & 0x80) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x40) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x20) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x10) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x08) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x04) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x02) ? '.' : ' ';
            *xpm_p++ = (*bitmap_p & 0x01) ? '.' : ' ';
        }
        if (j < width)
        {
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x80) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x40) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x20) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x10) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x08) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x04) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x02) ? '.' : ' ';
            if (j++ < width)
                *xpm_p++ = (*bitmap_p & 0x01) ? '.' : ' ';
            bitmap_p++;
        }
        *xpm_p = '\0';
    }
    xpm[i+3] = '\0';

    return xpm;
}

GdkPixbuf* ebook_read_gaiji(EB_Book *book, gchar *code)
{
    gint char_no = strtol(&code[1], NULL, 16), w, h, size;
    gchar bitmap[EB_SIZE_WIDE_FONT_48], **xpm;
    EB_Error_Code ec = EB_SUCCESS;
    GdkPixbuf *pixbuf;

    if((size = ebook_guess_gaiji_size(book, headword.font_h)) < 0)
        return NULL;
    ec = eb_set_font(book, EB_FONT_16);

    //memset(bitmap, 0, EB_SIZE_WIDE_FONT_48*sizeof(gchar));

    ec = eb_font_height(book, &h);
    if(code[0] == 'h')
    {
        ec = eb_narrow_font_width(book, &w);
        ec = eb_narrow_font_character_bitmap(book, char_no, bitmap);
    }
    else
    {
        ec = eb_wide_font_width(book, &w);
        ec = eb_wide_font_character_bitmap(book, char_no, bitmap);
    }
    if(ec != EB_SUCCESS)
        return NULL;

    xpm = ebook_bitmap_to_xpm(bitmap, w, h, NULL);

    pixbuf = gdk_pixbuf_new_from_xpm_data((const char**)xpm);
    return pixbuf;
}

static gint ebook_save_search_results(BOOK_INFO *binfo, const gchar *word, gint maxhits)
{
    gint i, n, hits_count;
    gchar heading[MAX_BUFSIZE];
    ssize_t len;
    EB_Hit *hits = (EB_Hit*)g_new0(EB_Hit, maxhits);
    EB_Error_Code error_code;
    RESULT *rp;

    error_code = eb_hit_list(binfo->book, maxhits, hits, &hits_count);
    if(error_code != EB_SUCCESS)
    {
        g_free(hits);
        return error_code;
    }
    n = g_sequence_get_length(ebook.results);
    for(i = 0; ((i + n) < maxhits) && (i < hits_count); i++)
    {
        rp = (RESULT*)g_new0(RESULT, 1);
        if(!rp)
        {
            LOG(LOG_ERROR, "No memory");
            break;
        }
        error_code = eb_seek_text(binfo->book, &(hits[i].heading));
        if(error_code != EB_SUCCESS)
        {
            g_free(rp);
            continue;
        }
        error_code = eb_read_heading(binfo->book, binfo->appendix, NULL, NULL, MAX_BUFSIZE, heading, &len);
        if(error_code != EB_SUCCESS)
        {
            g_free(rp);
            break;
        }
        heading[len] = '\0';

        rp->heading = iconv_convert(ENC_EUC_JP, ENC_UTF8, heading);

        rp->book = binfo->book;
        rp->pos = hits[i].text;
        g_sequence_append(ebook.results, rp);
    }
    g_free(hits);
    return error_code;
}

void ebook_search_book(const gchar *word, gint method, GSequence *results, gint maxhits, BOOK_INFO *binfo)
{
    gchar **words;
    EB_Error_Code error_code;

    if(results)
    {
        ebook_clear_results(results);
        ebook.results = results;
    }
    switch(method)
    {
    case SEARCH_METHOD_FORWARD:
        if(eb_have_word_search(binfo->book))
            error_code = eb_search_word(binfo->book, word);
        else
            return;
        break;
    case SEARCH_METHOD_BACKWARD:
        if(eb_have_endword_search(binfo->book))
            error_code = eb_search_endword(binfo->book, word);
        else
            return;
        break;
    case SEARCH_METHOD_EXACTWORD:
        if(eb_have_exactword_search(binfo->book))
            error_code = eb_search_exactword(binfo->book, word);
        else
            return;
        break;
    case SEARCH_METHOD_KEYWORD:
        if(eb_have_keyword_search(binfo->book))
        {
            words = g_strsplit_set(word, " \n\r\t", 0);
            error_code = eb_search_keyword(binfo->book, (const gchar* const*)words);
            g_strfreev(words);
        }
        else
            return;
        break;
    default:
        return;
    }
    if(error_code == EB_SUCCESS)
        ebook_save_search_results(binfo, word, maxhits);
}

void ebook_search(const gchar *word, gint method, GSequence *results, gint maxhits)
{
    GtkTreeIter iter;
    BOOK_INFO *binfo;
    if(!dictbar_get_group(&iter))
        return;
    ebook_clear_results(results);
    ebook.results = results;
    do {
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_BINFO, &binfo, -1);
        if(!binfo->active)
            continue;
        ebook_search_book(word, method, NULL, maxhits, binfo);
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(dicts.store), &iter));
    ebook.results = NULL;
}

void ebook_menu_or_copyright(gboolean menu, GSequence *results)
{
    GtkTreeIter   iter;
    EB_Position pos;
    RESULT *rp;
    BOOK_INFO *binfo;
    EB_Error_Code err;
    ebook.results = results;

    textview_clear_textbuf(NULL);
    headword_clear();
    ebook_clear_results(results);

    if(!dictbar_get_group(&iter))
        return;
    
    do {
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_BINFO, &binfo, -1);
        if(!binfo->active)
            continue;
        if(menu ? !eb_have_menu(binfo->book) : !eb_have_copyright(binfo->book))
            continue;
        err = menu ? eb_menu(binfo->book, &pos) : eb_copyright(binfo->book, &pos);
        if(err != EB_SUCCESS)
            continue;
        rp = (RESULT *)calloc(sizeof(RESULT), 1);
        rp->heading = g_strdup_printf("%s : %s", menu ? _("menu") : _("copyright"), binfo->title);
        rp->book = binfo->book;
        rp->pos = pos;
        g_sequence_append(ebook.results, rp);
    } while(gtk_tree_model_iter_next(GTK_TREE_MODEL(dicts.store), &iter));
    ebook.results = NULL;

    headword_show_results();
}

GdkPixbuf *ebook_load_image(EB_Book *book, EB_Position *pos)
{
    gchar *data = NULL, page[EB_SIZE_PAGE]; 
    ssize_t len = 0, len1 = 0;
    GError *err = NULL;
    GdkPixbuf *pixbuf = NULL;
    GInputStream *ism = 0;
    EB_Error_Code error_code = eb_set_binary_color_graphic(book, pos);
    if(error_code != EB_SUCCESS)
    {
        LOG(LOG_CRITICAL, "Failed to set binary color graphic : %s", eb_error_message(error_code));
        return NULL;
    }
    for(;;)
    {
        error_code = eb_read_binary(book, EB_SIZE_PAGE, page, &len1);
        if(error_code != EB_SUCCESS || !len1)
            break;
        data = g_renew(gchar, data, len + len1 + 1);
        g_memmove(&(data[len]), page, len1);
        len += len1;
        data[len] = 0;
    }
    if(!data)
        return NULL;
    ism = g_memory_input_stream_new_from_data(data, len, g_free);
    pixbuf = gdk_pixbuf_new_from_stream(ism, NULL, &err);
    if(g_input_stream_close(ism, NULL, &err))
        g_free(data);
    return pixbuf;
}

