
#include "defs.h"

#include "dicts.h"
#include "ebook.h"
#include "history.h"
#include "jcode.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "render.h"
#include "textview.h"
#include "toolbar.h"

void popupwnd_close(GtkWidget *widget, gpointer data)
{
    if(popupwnd.wnd)
    {
        gtk_widget_destroy(popupwnd.view);
        gtk_widget_destroy(popupwnd.wnd);
        gtk_widget_destroy(popupwnd.menu);
        popupwnd.wnd = NULL;
        popupwnd.menu = NULL;
    }
}

static gboolean popupwnd_close_(gpointer data)
{
    popupwnd_close(NULL, NULL);
    return FALSE;
}

static void popupwnd_start_stop_timer(gboolean start)
{
    static gint timeout_id = 0;
    if(timeout_id != 0)
        g_source_remove(timeout_id);
    timeout_id = 0;
    if(!start || popupwnd.lock)
        return;
    timeout_id = g_timeout_add(popupwnd.timeout, popupwnd_close_, NULL);
}

void popupwnd_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer user_data)
{
    popupwnd_start_stop_timer((event->detail == GDK_NOTIFY_NONLINEAR) || (event->detail == GDK_NOTIFY_NONLINEAR_VIRTUAL));
}

void popupwnd_enter_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer user_data)
{
    popupwnd_start_stop_timer(FALSE);
}

void popupwnd_lock(GtkWidget *widget, gpointer user_data)
{
    popupwnd.lock = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(widget));
}

void popupwnd_query_in_mainwnd(GtkWidget *widget, gpointer data)
{
    const gchar *str = gtk_label_get_text(GTK_LABEL(popupwnd.label));
    mainwnd_search_word(str);
}

void popupwnd_dict_changed(GtkWidget *w, gpointer data)
{
    if(w)
    {
        popupwnd.binfo = (BOOK_INFO*)data;
        popupwnd.lock = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(popupwnd.lockbtn));
    }
    if(!popupwnd.binfo)
    {
        g_free(popupwnd.group);
        popupwnd.group = NULL;
        g_free(popupwnd.dict);
        popupwnd.dict = NULL;
        return;
    }
    GtkTreePath *path = dicts_find_path(popupwnd.binfo->book);
    if(path)
    {
        GtkTreeIter iter, iter1;
        gtk_tree_model_get_iter(GTK_TREE_MODEL(dicts.store), &iter1, path);
        gtk_tree_model_iter_parent(GTK_TREE_MODEL(dicts.store), &iter, &iter1);
        gtk_tree_path_free(path);
        g_free(popupwnd.group);
        g_free(popupwnd.dict);
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter, DICT_ALIAS, &popupwnd.group, -1);
        gtk_tree_model_get(GTK_TREE_MODEL(dicts.store), &iter1, DICT_ALIAS, &popupwnd.dict, -1);
    }
    popupwnd_search(NULL);
}

gboolean popupwnd_dict_menu_append(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer data)
{
    static gboolean sel_group = FALSE;
    static GSList *group = NULL;
    GtkWidget *item;
    if(!model)
    {
        if(popupwnd.menu)
            gtk_widget_destroy(popupwnd.menu);
        group = NULL;
        popupwnd.menu = gtk_menu_new();
        item = gtk_radio_menu_item_new_with_label(group, _("Main window selection"));
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed), NULL);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(popupwnd.menu), item);
        return FALSE;
    }
    if(gtk_tree_path_get_depth(path) == 1)
    {
        gchar *name;
        gtk_menu_shell_append(GTK_MENU_SHELL(popupwnd.menu), gtk_separator_menu_item_new());
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, -1);
        sel_group = !g_strcmp0(popupwnd.group, name);
        g_free(name);
    }
    else
    {
        gchar *name;
        BOOK_INFO *binfo;
        gtk_tree_model_get(model, iter, DICT_ALIAS, &name, DICT_BINFO, &binfo, -1);
        item = gtk_radio_menu_item_new_with_label(group, binfo->title);
        g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(popupwnd_dict_changed), binfo);
        group = gtk_radio_menu_item_get_group(GTK_RADIO_MENU_ITEM(item));
        gtk_menu_shell_append(GTK_MENU_SHELL(popupwnd.menu), item);
        if(sel_group && !g_strcmp0(popupwnd.dict, name))
        {
            popupwnd.binfo = binfo;
            gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(item), TRUE); 
        }
        g_free(name);
        
    }
    return FALSE;
}

void popupwnd_dict_menu_create()
{
    if(!popupwnd.menu)
    {
        popupwnd_dict_menu_append(NULL, NULL, NULL, NULL);
        gtk_tree_model_foreach(GTK_TREE_MODEL(dicts.store), popupwnd_dict_menu_append, NULL);
    }
}

void popupwnd_set_dict(GtkWidget *widget, gpointer data)
{
    popupwnd_dict_menu_create();
    popupwnd.lock = TRUE;
    gtk_widget_show_all(popupwnd.menu);
    gtk_menu_popup(GTK_MENU(popupwnd.menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
}

static gint popupwnd_view_button_press(GtkWidget *widget, GdkEventButton *event)
{
    if((event->type == GDK_BUTTON_PRESS) && (event->button == 1))
    {
        GtkTextTag *tag = textview_get_link_under_cursor(widget, (gint)(event->x), (gint)(event->y));
        if(tag)
        {
            RESULT *res = (RESULT*)g_object_get_data(G_OBJECT(tag), "link");
            if(!res)
                return TRUE;
            textview_clear_textbuf(popupwnd.buffer);
            popupwnd_open(res);
        }
    }
    return FALSE;
}

void popupwnd_underline_link(GtkWidget *widget, GdkEventMotion *event)
{
    static GtkTextTag *tag_lnk = NULL; 
    textview_underline_lnk(widget, event, &tag_lnk);
}

gint popupwnd_view_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
    GdkCursor *cursor = gdk_cursor_new(GDK_LEFT_PTR);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT), cursor);
    gdk_cursor_unref(cursor);
    popupwnd_underline_link(widget, NULL);
    return FALSE;
}

static gint popupwnd_view_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
    popupwnd_underline_link(widget, event);
    return TRUE;
}

static void popupwnd_title_event(gboolean drag)
{
    static gboolean bdrag = FALSE;
    static gint sx, sy;
    gint x, y; 
    GdkModifierType mask;
    GdkWindow *root_win = gdk_window_foreign_new(GDK_ROOT_WINDOW());
    gdk_window_get_pointer(root_win, &x, &y, &mask);
    if(drag && bdrag)
    {
        gint wx, wy;
        gtk_window_get_position(GTK_WINDOW(popupwnd.wnd), &wx, &wy);
        gtk_window_move(GTK_WINDOW(popupwnd.wnd), wx + x - sx, wy + y - sy);
    }
    sx = x; sy = y;
    bdrag = drag;
}

static gint popupwnd_title_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
    popupwnd_title_event(TRUE);
    return TRUE;
}

static gint popupwnd_title_button_press(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    popupwnd_title_event(TRUE);
    return FALSE;
}

static gint popupwnd_title_button_release(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
    popupwnd_title_event(FALSE);
    return FALSE;
}

static void popupwnd_scroll_mark(GtkTextBuffer *buffer, gboolean next)
{
    gint i, x, y, n, len, d;;
    gchar buf[16];
    GtkTextIter iter, iter1;
    GtkTextMark *mark;
    gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(popupwnd.view), GTK_TEXT_WINDOW_TEXT, 1, 1, &x, &y);
    gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(popupwnd.view), &iter, x, y);
    len = g_sequence_get_length(popupwnd.results);
    if(next)
    {
        n = 0; d = 1;
    }
    else
    {
        n = len - 1; d = -1;
    }
    for(i = n;; i += d)
    {
        sprintf(buf, "%d", i);
        mark = gtk_text_buffer_get_mark(buffer, buf);
        if(!mark)
            break;
        gtk_text_buffer_get_iter_at_mark(buffer, &iter1, mark);
        if(gtk_text_iter_compare(&iter, &iter1) == -d)
        {
            gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(popupwnd.view), mark, 0.0, TRUE, 0.0, 0.0);
            break;
        }
    }
}

static void popupwnd_scroll_next(GtkWidget *widget, gpointer data)
{
    popupwnd_scroll_mark(popupwnd.buffer, TRUE);
}

static void popupwnd_scroll_prev(GtkWidget *widget, gpointer data)
{
    popupwnd_scroll_mark(popupwnd.buffer, FALSE);
}

static void popupwnd_window_create()
{
    GdkModifierType mask;
    gint pos_x, pos_y;
    gint pointer_x, pointer_y;
    gint root_x, root_y;

    GtkWidget *vbox, *hbox, *eventbox, *frame, *scroll, *btn, *image;

    GdkWindow *root_win = gdk_window_foreign_new(GDK_ROOT_WINDOW());
    gdk_drawable_get_size(root_win, &root_x, &root_y);

    // If there is no window, determine from the mouse position.
    gdk_window_get_pointer(root_win, &pointer_x, &pointer_y, &mask);
    pos_x = pointer_x + 10;
    pos_y = pointer_y + 10;

    if(pos_x + popupwnd.width > root_x)
        pos_x = root_x - popupwnd.width;
    
    if(pos_y + popupwnd.height > root_y)
        pos_y = root_y - popupwnd.height;

    popupwnd.wnd = gtk_widget_new(GTK_TYPE_WINDOW,
                            "type", GTK_WINDOW_POPUP,
                             "default-width", popupwnd.width,
                             "default-height", popupwnd.height,
                            NULL);
    gtk_window_move(GTK_WINDOW(popupwnd.wnd), pos_x, pos_y);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "delete_event", G_CALLBACK(popupwnd_close), NULL);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "leave_notify_event", G_CALLBACK(popupwnd_leave_notify), (gpointer)NULL);
    g_signal_connect(G_OBJECT(popupwnd.wnd), "enter_notify_event", G_CALLBACK(popupwnd_enter_notify), (gpointer)NULL);

    vbox = gtk_vbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(popupwnd.wnd), vbox);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);

    hbox = gtk_hbox_new(FALSE, 0);
    gtk_container_add(GTK_CONTAINER(frame), hbox);

    eventbox = gtk_event_box_new();
    gtk_box_pack_start(GTK_BOX(hbox), eventbox, TRUE, TRUE, 2);
    g_signal_connect(G_OBJECT(eventbox), "button_press_event", G_CALLBACK(popupwnd_title_button_press), (gpointer)NULL);
    g_signal_connect(G_OBJECT(eventbox), "motion_notify_event", G_CALLBACK(popupwnd_title_motion_notify), (gpointer)NULL);
    g_signal_connect(G_OBJECT(eventbox), "button_release_event", G_CALLBACK(popupwnd_title_button_release), (gpointer)NULL);
    popupwnd.label = gtk_label_new("");
    gtk_container_add(GTK_CONTAINER(eventbox), popupwnd.label);

    btn = toolbar_button(popupwnd_query_in_mainwnd, GTK_STOCK_FIND, _("Query in the main window"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_set_dict, GTK_STOCK_INDEX, _("Select dictionary"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_scroll_prev, GTK_STOCK_GO_BACK, _("Scroll to the previous result"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    btn = toolbar_button(popupwnd_scroll_next, GTK_STOCK_GO_FORWARD, _("Scroll to the next result"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    popupwnd.lockbtn = gtk_toggle_button_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(popupwnd.lockbtn), popupwnd.lock);
    gtk_button_set_relief(GTK_BUTTON(popupwnd.lockbtn), GTK_RELIEF_NONE);
    g_signal_connect(G_OBJECT(popupwnd.lockbtn), "toggled", G_CALLBACK(popupwnd_lock), (gpointer)NULL);
    gtk_box_pack_start(GTK_BOX(hbox), popupwnd.lockbtn, FALSE, FALSE, 0);
    gtk_widget_set_tooltip_text(popupwnd.lockbtn, _("Lock popup window on the screen."));
    image = gtk_image_new_from_stock(GTK_STOCK_REFRESH, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_container_add(GTK_CONTAINER(popupwnd.lockbtn), image);

    btn = toolbar_button(popupwnd_close, GTK_STOCK_CLOSE, _("Close popup window"), TRUE);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 2);

    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
    gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 0);

    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

    gtk_container_add(GTK_CONTAINER(frame), scroll);

    textview_create_textbuf(&popupwnd.buffer);
    popupwnd.view = gtk_text_view_new_with_buffer(popupwnd.buffer);
    gtk_text_view_set_editable(GTK_TEXT_VIEW(popupwnd.view), FALSE);
    gtk_text_view_set_left_margin(GTK_TEXT_VIEW(popupwnd.view), 5);
    gtk_text_view_set_right_margin(GTK_TEXT_VIEW(popupwnd.view), 5);
    gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(popupwnd.view), 3);
    gtk_text_view_set_pixels_inside_wrap(GTK_TEXT_VIEW(popupwnd.view), 3);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(popupwnd.view), FALSE);
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(popupwnd.view), GTK_WRAP_WORD);

    g_signal_connect(G_OBJECT(popupwnd.view), "button_press_event", G_CALLBACK(popupwnd_view_button_press), (gpointer)NULL);
    g_signal_connect(G_OBJECT(popupwnd.view), "leave_notify_event", G_CALLBACK(popupwnd_view_leave_notify), (gpointer)NULL);
    g_signal_connect(G_OBJECT(popupwnd.view), "motion_notify_event", G_CALLBACK(popupwnd_view_motion_notify), (gpointer)NULL);
    gtk_widget_add_events(popupwnd.view, GDK_LEAVE_NOTIFY_MASK);

    gtk_container_add(GTK_CONTAINER(scroll), popupwnd.view);

    gtk_widget_realize(popupwnd.wnd);
    gdk_window_set_decorations(popupwnd.wnd->window, 0);

    gtk_widget_show_all(popupwnd.wnd);
}

static gint popupwnd_scroll_to_top()
{
    GtkTextIter iter;
    GtkTextMark *mark;
    gtk_text_buffer_get_start_iter(popupwnd.buffer, &iter);
    mark =  gtk_text_buffer_create_mark(popupwnd.buffer, "start", &iter, TRUE);
    gtk_text_view_scroll_to_mark(GTK_TEXT_VIEW(popupwnd.view), mark, 0.0, TRUE, 0.0, 0.0);
    gtk_text_buffer_delete_mark(popupwnd.buffer, mark);
    return(0);
}

void popupwnd_open(RESULT *res)
{
    static int counter = 0;
    if(!res)
    {
        counter = 0;
        return;
    }
    gchar *text = ebook_get_text(res->book, &res->pos);
    if(text)
    {
        gchar buf[16];
        GtkTextIter iter;
        gtk_text_buffer_get_end_iter(popupwnd.buffer, &iter);
        if(gtk_text_buffer_get_char_count(popupwnd.buffer))
            gtk_text_buffer_insert(render.buffer, &iter, "\n", -1);
        g_sprintf(buf, "%d", counter++);
        gtk_text_buffer_create_mark(popupwnd.buffer, buf, &iter, TRUE);
        render_content(res->book, popupwnd.buffer, text);
        g_free(text);
        popupwnd_scroll_to_top();
    }
}

void popupwnd_render(gpointer data, gpointer user_data)
{
    RESULT *res = (RESULT*)data;
    if(res)
        popupwnd_open(res);
}

void popupwnd_show(const gchar *word)
{
    textview_clear_textbuf(popupwnd.buffer);
    popupwnd_open(NULL);
    popupwnd_render(NULL, NULL);
    if(g_sequence_get_length(popupwnd.results))
        g_sequence_foreach(popupwnd.results, popupwnd_render, NULL);
    else
        textview_insert_message(popupwnd.buffer, "No hit.");
    gtk_label_set_text(GTK_LABEL(popupwnd.label), word);
    gtk_window_present(GTK_WINDOW(popupwnd.wnd));
}

void popupwnd_search(const gchar *str)
{
    if(!str)
        str = gtk_label_get_text(GTK_LABEL(popupwnd.label));

    gchar *euc_str = iconv_convert(ENC_UTF8, ENC_EUC_JP, str);
    if(!validate_euc(euc_str))
    {
        g_free(euc_str);
        return;
    }
    if(!popupwnd.wnd)
    {
        popupwnd_window_create();
        popupwnd.binfo = NULL;
    }
    popupwnd_dict_menu_create();
    if(popupwnd.binfo)
        ebook_search_book(euc_str, popupwnd.search_method, popupwnd.results, popupwnd.maxhits, popupwnd.binfo);
    else
        ebook_search(euc_str, popupwnd.search_method, popupwnd.results, popupwnd.maxhits);
    g_free(euc_str);
    popupwnd_show(str);
}

