
#include "defs.h"

#include <sys/stat.h>

#include <langinfo.h>

#include <pango/pangox.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>

#include "dictbar.h"
#include "dicts.h"
#include "ebook.h"
#include "headword.h"
#include "history.h"
#include "mainwnd.h"
#include "popupwnd.h"
#include "preferences.h"
#include "render.h"
#include "selection.h"
#include "shortcut.h"
#include "textview.h"

#include "pref_mainwnd.h"
#include "pref_popupwnd.h"
#include "pref_shortcut.h"

struct pref_page
{
    gchar *title;
    gboolean is_child;
    GtkWidget *(* start_func)();
    void (* end_func)(gboolean);
};

struct pref_page prefs[] =
{
    { N_("Main Window"), FALSE, pref_mainwnd_start, pref_mainwnd_end},
    { N_("Font"), TRUE, pref_font_start, pref_font_end},
    { N_("Colors"), TRUE, pref_color_start, pref_color_end},
    { N_("Search"), TRUE, pref_mainwnd_search_start, pref_mainwnd_search_end},
    { N_("Popup Window"), FALSE, pref_popupwnd_start, pref_popupwnd_end},
    { N_("Search"), TRUE, pref_popupwnd_search_start, pref_popupwnd_search_end},
    { N_("Shortcuts"), FALSE, pref_shortcut_start, pref_shortcut_end},
    {NULL, FALSE, NULL, NULL}
};

void preferences_init()
{
    gchar *home_dir;
    GDir *dir;

    shortcuts.ignore_locks = TRUE;
    textview.line_space = 0.3;

    headword.maxhits = 50;
    dictbar.word_hist = 30;
    mainwnd.search = SEARCH_METHOD_FORWARD;
    selection.interval = 1000;
    selection.maxchar = 32;
    popupwnd.lock = FALSE;
    popupwnd.width = 350;
    popupwnd.height = 450;
    popupwnd.timeout = 2000;
    popupwnd.maxhits = 5;
    popupwnd.search_method = SEARCH_METHOD_EXACTWORD;
    popupwnd.group = NULL;
    popupwnd.dict = NULL;
    mainwnd.x = 0;
    mainwnd.y = 0;
    mainwnd.w = 670;
    mainwnd.h = 440;
    mainwnd.font = NULL;
    mainwnd.remember_pos = TRUE;
    paned.treew = 185;
    paned.treeh = 344;
    textview.scroll_smooth = TRUE;
    textview.scroll_step = 10;
    textview.scroll_time = 100000;
    textview.scroll_margin= 30;

    pref.codeset = nl_langinfo(CODESET);
    textview.colors[COLOR_LINK] = g_strdup("#0000c0");
    textview.colors[COLOR_KEYWORD] = g_strdup("#c00000");
    textview.colors[COLOR_AUDIO] = g_strdup("#00c000");
    textview.colors[COLOR_VIDEO] = g_strdup("#00c000");
    textview.colors[COLOR_TITLE] = g_strdup("#306090");

    dicts.store = gtk_tree_store_new(DICT_N, G_TYPE_STRING, G_TYPE_POINTER, G_TYPE_BOOLEAN);

    mainwnd.results = g_sequence_new(result_free);
    popupwnd.results = g_sequence_new(result_free);

    mainwnd.combo_word = gtk_combo_box_entry_new_text();

    shortcuts_datalist_init();

    home_dir = getenv("HOME");
    pref.user = g_strdup_printf("%s%s.%s", home_dir, G_DIR_SEPARATOR_S, PACKAGE);

    if((dir = g_dir_open(pref.user, 0, NULL)) == NULL)
    {
        if(g_mkdir_with_parents(pref.user, 493))
        {
            LOG(LOG_CRITICAL, "Failed to create directory : %s\n", pref.user);
            exit(1);
        }
    }
    else
        g_dir_close(dir);
}

static void preferences_apply(gboolean save)
{
    gint i;
    for(i = 0;; i++)
    {
        if(prefs[i].title == NULL) break;
        if(prefs[i].end_func != NULL)
            prefs[i].end_func(save);
    }
    preferences_save();
    gtk_grab_remove(pref.dlg);
    gtk_widget_destroy(pref.dlg);
    popupwnd_close(NULL, NULL);
    dictbar_update();
}

static void preferences_ok(GtkWidget *widget, gpointer data)
{
    preferences_apply(TRUE);
}

static void preferences_cancel(GtkWidget *widget, gpointer data)
{
    preferences_apply(FALSE);
}

static void delete_event(GtkWidget *widget, GdkEvent  *event, gpointer data)
{
    preferences_apply(FALSE);
}

enum
{
    PREF_TITLE,
    PREF_NUMBER,
    PREF_N
};

static void preflist_selection_changed(GtkTreeSelection *selection, gpointer data)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    gint number;
    gchar *title;

    if(gtk_tree_selection_get_selected(selection, &model, &iter) == FALSE) return;

    gtk_tree_model_get (model, &iter, PREF_TITLE, &title, -1);
    g_free (title);

    gtk_tree_model_get (model, &iter, PREF_NUMBER, &number, -1);
    if(number >= 0)
        gtk_notebook_set_current_page(GTK_NOTEBOOK(pref.note), number);
}

void preferences_show_cb(GtkWidget *w, gpointer data)
{
    preferences_show();
}

void preferences_show()
{
    gint i;
    GtkWidget       *hbox, *vbox, *frame, *button, *label, *widget, *preflist_view;
    GtkTreeIter     parent_iter, child_iter;
    GtkTreeStore    *pref_store;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkTreeSelection *select;

    pref.dlg = gtk_dialog_new();
    gtk_window_set_position(GTK_WINDOW(pref.dlg), GTK_WIN_POS_CENTER);
    g_signal_connect(G_OBJECT(pref.dlg), "delete_event", G_CALLBACK(delete_event), NULL);

    hbox = gtk_hbox_new(FALSE, 10);
    gtk_box_pack_start (GTK_BOX(GTK_DIALOG(pref.dlg)->vbox), hbox, TRUE, TRUE, 0);
    gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
    
    frame = gtk_frame_new(NULL);
    gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
    gtk_box_pack_start(GTK_BOX(hbox), frame, FALSE, FALSE, 0);

    pref_store = gtk_tree_store_new(PREF_N, G_TYPE_STRING, G_TYPE_INT);
    preflist_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(pref_store));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(preflist_view), FALSE);
    gtk_container_add(GTK_CONTAINER(frame), preflist_view);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new_with_attributes(_("Items"), renderer, "text", PREF_TITLE, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(preflist_view), column);

    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (preflist_view));
    gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
    g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK (preflist_selection_changed), NULL);

    vbox = gtk_vbox_new(FALSE, 10);
    gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);

    pref.note = gtk_notebook_new();
    gtk_notebook_set_show_border(GTK_NOTEBOOK(pref.note), FALSE);
    gtk_notebook_set_show_tabs(GTK_NOTEBOOK(pref.note), FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), pref.note, TRUE, TRUE, 0);

    for(i = 0; prefs[i].title != NULL; i++)
    {
        if(prefs[i].is_child == TRUE)
        {
            gtk_tree_store_append(pref_store, &child_iter, &parent_iter);
            gtk_tree_store_set(pref_store, &child_iter, PREF_TITLE, _(prefs[i].title), PREF_NUMBER, i, -1);
        }
        else
        {
            gtk_tree_store_append(pref_store, &parent_iter, NULL);
            gtk_tree_store_set(pref_store, &parent_iter, PREF_TITLE, _(prefs[i].title), PREF_NUMBER, i, -1);
        }
        label = gtk_label_new(prefs[i].title);
        frame = gtk_frame_new(NULL);
        gtk_notebook_append_page(GTK_NOTEBOOK(pref.note), frame, label);

        hbox = gtk_hbox_new(FALSE, 0);
        gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
        gtk_container_add(GTK_CONTAINER(frame), hbox);

        widget = (prefs[i].start_func == NULL) ? gtk_label_new("") : prefs[i].start_func();
        if(widget != NULL)
            gtk_box_pack_start(GTK_BOX(hbox), widget, TRUE, TRUE, 0);
    }

    button = gtk_button_new_from_stock(GTK_STOCK_OK);
    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG (pref.dlg)->action_area), button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(preferences_ok), (gpointer)pref.dlg);

    button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
    GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG (pref.dlg)->action_area), button, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(button), "clicked", G_CALLBACK(preferences_cancel), (gpointer)pref.dlg);

    selection_lookup_stop();
    gtk_tree_view_expand_all(GTK_TREE_VIEW(preflist_view));
    gtk_window_set_modal(GTK_WINDOW(pref.dlg), TRUE);
    gtk_window_set_transient_for(GTK_WINDOW(pref.dlg), GTK_WINDOW(mainwnd.wnd));
    gtk_widget_show_all(pref.dlg);
}

struct _preferences preferences[] = {
    {"headword.maxhits", G_TYPE_INT, &headword.maxhits},
    {"dictbar.word_hist", G_TYPE_INT, &dictbar.word_hist},
    {"selection.interval", G_TYPE_INT, &selection.interval},
    {"selection.maxchar", G_TYPE_INT, &selection.maxchar},
    {"selection.lookup", G_TYPE_BOOLEAN, &selection.lookup},
    {"shortcuts.ignore_locks", G_TYPE_BOOLEAN, &shortcuts.ignore_locks},
    {"popupwnd.width", G_TYPE_INT, &popupwnd.width},
    {"popupwnd.height", G_TYPE_INT, &popupwnd.height},
    {"popupwnd.lock", G_TYPE_BOOLEAN, &popupwnd.lock},
    {"popupwnd.timeout", G_TYPE_INT, &popupwnd.timeout},
    {"popupwnd.maxhits", G_TYPE_INT, &popupwnd.maxhits},
    {"popupwnd.search_method", G_TYPE_INT, &popupwnd.search_method},
    {"popupwnd.group", G_TYPE_STRING, &popupwnd.group},
    {"popupwnd.dict", G_TYPE_STRING, &popupwnd.dict},
    {"mainwnd.x", G_TYPE_INT, &mainwnd.x},
    {"mainwnd.y", G_TYPE_INT, &mainwnd.y},
    {"mainwnd.w", G_TYPE_INT, &mainwnd.w},
    {"mainwnd.h", G_TYPE_INT, &mainwnd.h},
    {"mainwnd.remember_pos", G_TYPE_INT, &mainwnd.remember_pos},
    {"mainwnd.search", G_TYPE_INT, &mainwnd.search},
    {"mainwnd.font", G_TYPE_STRING, &mainwnd.font},
    {"paned.tree_width", G_TYPE_INT, &paned.treew},
    {"paned.tree_height", G_TYPE_INT, &paned.treeh},
    {"color.link", G_TYPE_STRING, &textview.colors[COLOR_LINK]},
    {"color.keyword", G_TYPE_STRING, &textview.colors[COLOR_KEYWORD]},
    {"color.video", G_TYPE_STRING, &textview.colors[COLOR_VIDEO]},
    {"color.audio", G_TYPE_STRING, &textview.colors[COLOR_AUDIO]},
    {"color.title", G_TYPE_STRING, &textview.colors[COLOR_TITLE]},
    {"textview.line_space", G_TYPE_INT, &textview.line_space},
    {"textview.scroll_smooth", G_TYPE_INT, &textview.scroll_smooth},
    {"textview.scroll_step", G_TYPE_INT, &textview.scroll_step},
    {"textview.scroll_time", G_TYPE_INT, &textview.scroll_time},
    {"textview.scroll_margin", G_TYPE_INT, &textview.scroll_margin},
};

void preferences_save()
{
    gint i, n;
    gchar filename[PATH_MAX], buff[128];
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"Preferences", NULL);

    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_PREFERENCES);
    
    n = sizeof (preferences) / sizeof (preferences[0]);
    for(i = 0 ; i < n; i++)
    {
        switch(preferences[i].type)
        {
        case G_TYPE_INT:
        case G_TYPE_BOOLEAN:
            sprintf(buff, "%d", *((int *)preferences[i].addr));
            break;
        case G_TYPE_FLOAT:
            sprintf(buff, "%f", *((float *)preferences[i].addr));
            break;
        case G_TYPE_STRING:
            if(!(*(gchar**)preferences[i].addr))
                continue;
            sprintf(buff, "%s", *((gchar **)preferences[i].addr));
            break;
        }
        xmlNodePtr node = xmlAddChild(doc->children, xmlNewNode(NULL, (xmlChar*)preferences[i].name));
        xmlNewProp(node, (xmlChar*)"value", (xmlChar*)buff);
    }
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void preferences_load_elem(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    gint i, n;

    n = sizeof(preferences) / sizeof(preferences[0]);
    for(i = 0; i < n; i++)
    {
        if(!g_strcmp0((gchar*)name, preferences[i].name))
        {
            switch(preferences[i].type){
            case G_TYPE_INT:
            case G_TYPE_BOOLEAN:
                *(int *)(preferences[i].addr) = atoi((gchar*)atts[1]);
                break;
            case G_TYPE_STRING:
                *(gchar **)(preferences[i].addr) = strdup((gchar*)atts[1]);
                break;
            case G_TYPE_FLOAT:
                *(float *)(preferences[i].addr) = atof((gchar*)atts[1]);
                break;
            }
        }
    }
}

void preferences_load()
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_PREFERENCES);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
        return;
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &preferences_load_elem;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

