
#include "defs.h"

#include "dictbar.h"
#include "eb123.h"
#include "headword.h"
#include "history.h"
#include "mainwnd.h"
#include "preferences.h"
#include "shortcut.h"
#include "textview.h"

#include <libxml/tree.h>
#include <gdk/gdkkeysyms.h>

struct _shortcut_command commands[] = {
    { _("Scroll Mainview Down"),       textview_scroll_down},
    { _("Scroll Mainview Up"),         textview_scroll_up},
    { _("Copy To Clipboard"),          textview_copy_to_clipboard},
    { _("Go Back"),                    history_prev},
    { _("Next Hit"),                   headword_next},
    { _("Previous Hit"),               headword_prev},
    { _("Quit Program"),               app_quit},
    { _("Iconify Window"),             mainwnd_iconify},
    {NULL, NULL}
};

gboolean shortcut_perform_by_event(GdkEventKey *event);
gboolean perform_shortcut(const gchar *name);

GtkAccelGroup *accel_group=NULL;

void shortcuts_datalist_init()
{
    gint i;
    KEY_EVENT *events = g_try_new0(KEY_EVENT, sizeof(commands));
    if(!events) return;

    g_datalist_init(&shortcuts.list);
    for(i = 0; commands[i].name; i++)
        g_datalist_id_set_data_full(&shortcuts.list, g_quark_from_static_string(commands[i].name), &(events[i]), g_free);
}

gboolean shortcut_accel_handler(GtkAccelGroup *accelgroup, GObject *arg1, guint arg2, GdkModifierType arg3, gpointer user_data)
{
    GdkEventKey event;

    event.keyval = (gulong)user_data;
    event.state = arg3;
    return shortcut_perform_by_event(&event);
}

void shortcut_install(GQuark key_id, gpointer data, gpointer user_data)
{
    GClosure *closure;
    KEY_EVENT *evt = (KEY_EVENT*)data;
    gulong code = evt->code;
    if(!evt->enabled) return;
    if((evt->mask == 0) && (evt->code == GDK_Return)) return;
    closure = g_cclosure_new(G_CALLBACK(shortcut_accel_handler), (gpointer)code, NULL);
    gtk_accel_group_connect(accel_group, evt->code, evt->mask, GTK_ACCEL_VISIBLE, closure);
}

void shortcuts_install()
{
    if(accel_group)
    {
        gtk_window_remove_accel_group(GTK_WINDOW(mainwnd.wnd), accel_group);
        g_object_unref(accel_group);
    }
    accel_group = gtk_accel_group_new();
    gtk_window_add_accel_group (GTK_WINDOW (mainwnd.wnd), accel_group);
    g_signal_connect(G_OBJECT(accel_group), "accel-activate", G_CALLBACK(shortcut_accel_handler), (gpointer)NULL);

    g_datalist_foreach(&shortcuts.list, shortcut_install, NULL);
}

void shortcuts_uninstall()
{
#if 0
    GList *item;
    GtkWidget *widget;
        item = g_list_first(accel_item_list);
        while(item){
            widget = (GtkWidget *)(item->data);
            gtk_widget_destroy(widget);
            item = g_list_next(item);
        }
        g_list_free(accel_item_list);
        accel_item_list = NULL;
        g_object_unref(accel_group);
#endif
}

gboolean shortcut_perform(const gchar *name)
{
    gint i;
    for(i = 0;; i++)
    {
        if(!commands[i].name)
            break;

        if(!g_strcmp0(commands[i].name, name))
        {
            commands[i].func();
            return TRUE;
        }
    }
    return FALSE;
}

void shortcut_perform_single_by_event(GQuark key_id, gpointer data, gpointer user_data)
{
    GdkEventKey *event = (GdkEventKey*)user_data;
    KEY_EVENT *evt = (KEY_EVENT*)data;
    if(evt->code == event->keyval)
    {
        if(evt->mask == event->state)
        {
            shortcut_perform(g_quark_to_string(key_id));
            return;
        }
        else if((shortcuts.ignore_locks) && ((event->state | GDK_MOD2_MASK | GDK_LOCK_MASK) == (evt->mask | GDK_MOD2_MASK | GDK_LOCK_MASK)))
        {
            shortcut_perform(g_quark_to_string(key_id));
            return;
        }
    }
}

gboolean shortcut_perform_by_event(GdkEventKey *event)
{
    g_datalist_foreach(&shortcuts.list, shortcut_perform_single_by_event, (gpointer)event);
    return FALSE;
}

void shortcuts_save_item(GQuark key_id, gpointer data, gpointer user_data)
{
    gchar buff[16];
    const gchar *command = g_quark_to_string(key_id);
    KEY_EVENT *evt = (KEY_EVENT*)data;
    if(evt->enabled)
    {
        xmlNodePtr node1 = xmlAddChild((xmlNodePtr)user_data, xmlNewNode(NULL, (xmlChar*)"shortcut"));
        xmlNewProp(node1, (xmlChar*)"command", (xmlChar*)command);
        sprintf(buff, "0x%04x", evt->mask);
        xmlNewProp(node1, (xmlChar*)"mask", (xmlChar*)buff);
        sprintf(buff, "0x%04x", evt->code);
        xmlNewProp(node1, (xmlChar*)"code", (xmlChar*)buff);
    }
}

void shortcuts_save()
{
    gchar filename[PATH_MAX];
    xmlDocPtr doc = xmlNewDoc((xmlChar*)"1.0");
    doc->children = xmlNewDocRawNode(doc, NULL, (xmlChar*)"Shortcuts", NULL);
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_SHORTCUTS);
    g_datalist_foreach(&shortcuts.list, shortcuts_save_item, (gpointer)doc->children);
    xmlSaveFormatFileEnc(filename, doc, "utf8", 0);
    xmlFreeDoc(doc);
}

void shortcuts_load_item(void *ctx, const xmlChar *name, const xmlChar **atts)
{
    if(!atts) return;
    KEY_EVENT *evt = (KEY_EVENT*)g_datalist_id_get_data(&shortcuts.list, g_quark_try_string((gchar*)atts[1]));
    if(evt)
    {
        evt->mask = strtol((gchar*)atts[3], NULL, 16);
        evt->code = strtol((gchar*)atts[5], NULL, 16);
        evt->enabled = TRUE;
    }
}

/* Add "Ctrl + q" shortcut */
void shortcuts_add_defaults()
{
    gint i;
    KEY_EVENT *evt;
    for(i = 0; commands[i].name; i++)
    {
        evt = (KEY_EVENT*)g_datalist_id_get_data(&shortcuts.list, g_quark_try_string(commands[i].name));
        if(commands[i].func == app_quit)
        {
            evt->mask = 0x4;
            evt->code = 0x71;
            evt->enabled = TRUE;
        }
    }
}

void shortcuts_load()
{
    gchar filename[PATH_MAX];
    xmlSAXHandler cb;
    sprintf(filename, "%s%s%s", pref.user, G_DIR_SEPARATOR_S, FILENAME_SHORTCUTS);
    if(!g_file_test(filename, G_FILE_TEST_IS_REGULAR))
    {
        shortcuts_add_defaults();
        return;
    }
    memset(&cb, 0, sizeof(xmlSAXHandler));
    cb.startElement = &shortcuts_load_item;
    xmlDocPtr doc = xmlSAXParseFile(&cb, filename, 0);
    xmlFreeDoc(doc);
}

