
#include "defs.h"

#include "dictbar.h"
#include "dump.h"
#include "headword.h"
#include "history.h"
#include "mainwnd.h"
#include "textview.h"

static void textview_search_selection(GtkWidget *w, gpointer data)
{
    gchar *text;
    GtkTextIter start, end;

    gtk_text_buffer_get_selection_bounds(textview.buffer, &start, &end);
    text = gtk_text_buffer_get_text(textview.buffer, &start, &end, FALSE);
    mainwnd_search_word(text);
}

void textview_copy_to_clipboard(GtkWidget *w, gpointer data)
{
    gtk_text_buffer_copy_clipboard(textview.buffer, gtk_clipboard_get(NULL));
}

void textview_create_textbuf(GtkTextBuffer **buffer)
{
    if(*buffer)
        g_object_unref(G_OBJECT(*buffer));

    GtkTextTagTable *table = gtk_text_tag_table_new();
    *buffer = gtk_text_buffer_new(table);
}

GtkTextTag *textview_get_link_under_cursor(GtkWidget *widget, gint ex, gint ey)
{
    gint        x, y;
    GSList      *tag_list;
    GtkTextIter iter;
    GtkTextTag  *res = NULL;

    gtk_text_view_window_to_buffer_coords(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT, ex, ey, &x, &y);
    gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(widget), &iter, x, y);
    tag_list = gtk_text_iter_get_tags(&iter);
    if(tag_list)
    {
        GSList *tag_l = tag_list;
        while(tag_l)
        {
            GtkTextTag *tag = GTK_TEXT_TAG(tag_l->data);
            gpointer p1 = g_object_get_data(G_OBJECT(tag), "link");
            gpointer p2 = g_object_get_data(G_OBJECT(tag), "audio");
            gpointer p3 = g_object_get_data(G_OBJECT(tag), "video");
            if(p1 || p2 || p3)
            {
                res = tag;
                break;
            }
            tag_l = g_slist_next(tag_l);
        }
        g_slist_free(tag_list);
    }
    return res;
}

void textview_underline_lnk(GtkWidget *widget, GdkEventMotion *event, GtkTextTag **tag_lnk)
{
    GdkCursor   *cursor;
    if(*tag_lnk)
        g_object_set(G_OBJECT(*tag_lnk), "underline", PANGO_UNDERLINE_NONE, NULL);

    if(!event)
        return;

    *tag_lnk = textview_get_link_under_cursor(widget, (gint)(event->x), (gint)(event->y));
    cursor = gdk_cursor_new(*tag_lnk ? GDK_HAND2 : GDK_LEFT_PTR);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT), cursor);
    gdk_cursor_unref(cursor);
    if(*tag_lnk)
        g_object_set(G_OBJECT(*tag_lnk), "underline", PANGO_UNDERLINE_SINGLE, NULL);
}

void textview_underline_link(GtkWidget *widget, GdkEventMotion *event)
{
    static GtkTextTag *tag_lnk = NULL;
    textview_underline_lnk(widget, event, &tag_lnk);
}

gint textview_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
    textview_underline_link(widget, event);
    return !(event->state & GDK_BUTTON1_MASK);
}

gint textview_leave_notify(GtkWidget *widget, GdkEventCrossing *event, gpointer data)
{
    GdkCursor *cursor = gdk_cursor_new(GDK_LEFT_PTR);
    gdk_window_set_cursor(gtk_text_view_get_window(GTK_TEXT_VIEW(widget), GTK_TEXT_WINDOW_TEXT), cursor);
    gdk_cursor_unref(cursor);
    textview_underline_link(widget, NULL);
    return FALSE;
}

gint textview_button_press(GtkWidget *widget, GdkEventButton *event)
{
    if((event->type == GDK_BUTTON_PRESS) && (event->button == 1))
    {
        GtkTextTag *tag = textview_get_link_under_cursor(widget, (gint)(event->x), (gint)(event->y));
        if(tag)
        {
            RESULT *res = (RESULT*)g_object_get_data(G_OBJECT(tag), "link");
            if(!res)
                return TRUE;
            mainwnd_open(res);
        }
    }
    else if((event->type == GDK_BUTTON_PRESS) && ((event->button == 2) || (event->button == 3)))
    {
        gtk_widget_show_all(textview.menu);
        gtk_menu_popup(GTK_MENU(textview.menu), NULL, NULL, NULL, NULL, 0, gtk_get_current_event_time());
        return TRUE;
    }
    return FALSE;
}

GtkWidget *textview_create_menu()
{
    GtkWidget *menu = gtk_menu_new(), *item;
   
    item = gtk_image_menu_item_new_from_stock(GTK_STOCK_FIND, NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(textview_search_selection), NULL);

    item = gtk_image_menu_item_new_from_stock(GTK_STOCK_COPY, NULL);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), item);
    g_signal_connect(G_OBJECT(item), "activate", G_CALLBACK(textview_copy_to_clipboard), NULL);

    gtk_menu_shell_append(GTK_MENU_SHELL(menu), gtk_separator_menu_item_new());

    dump.item[DUMP_HEX] = gtk_check_menu_item_new_with_label(_("Hex dump"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), dump.item[DUMP_HEX]);
    dump.handler[DUMP_HEX] = g_signal_connect(G_OBJECT(dump.item[DUMP_HEX]), "activate", G_CALLBACK(dump_hex), NULL);

    dump.item[DUMP_TEXT] = gtk_check_menu_item_new_with_label(_("Text dump"));
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), dump.item[DUMP_TEXT]);
    dump.handler[DUMP_TEXT] = g_signal_connect(G_OBJECT(dump.item[DUMP_TEXT]), "activate", G_CALLBACK(dump_text), NULL);

    return menu;
}

void textview_set_pixels(gint n)
{
    gtk_text_view_set_pixels_inside_wrap(GTK_TEXT_VIEW(textview.view), n);
    gtk_text_view_set_pixels_above_lines(GTK_TEXT_VIEW(textview.view), n);
    gtk_text_view_set_pixels_inside_wrap(GTK_TEXT_VIEW(textview.view), n);
}

GtkWidget *textview_create()
{
    textview.scroll = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(textview.scroll), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(textview.scroll), GTK_SHADOW_IN);

    textview_create_textbuf(&textview.buffer);

    textview.view = gtk_text_view_new_with_buffer(textview.buffer);

    g_signal_connect(G_OBJECT(textview.view), "motion_notify_event", G_CALLBACK(textview_motion_notify), (gpointer)NULL);
    g_signal_connect(G_OBJECT(textview.view), "button_press_event", G_CALLBACK(textview_button_press), (gpointer)NULL);
    g_signal_connect(G_OBJECT(textview.view), "leave_notify_event", G_CALLBACK(textview_leave_notify), (gpointer)NULL);
    gtk_widget_add_events(textview.view, GDK_LEAVE_NOTIFY_MASK);
    
    gtk_text_view_set_editable(GTK_TEXT_VIEW(textview.view), FALSE);
    gtk_text_view_set_left_margin(GTK_TEXT_VIEW(textview.view), 10);
    gtk_text_view_set_right_margin(GTK_TEXT_VIEW(textview.view), 10);
    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview.view), GTK_WRAP_WORD);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(textview.view), FALSE);

    gtk_container_add(GTK_CONTAINER (textview.scroll), textview.view);

    textview.menu = textview_create_menu();
    return textview.scroll;
}

void textview_scroll(gboolean down)
{
    GtkTextIter iter;
    GdkRectangle rect;
    gint distance;
    gint i, m = down ? +1 : -1;

    gtk_text_view_get_visible_rect(GTK_TEXT_VIEW(textview.view), &rect);

    distance = rect.height - textview.scroll_margin;

    if(textview.scroll_smooth)
    {
        for(i = 0; i < textview.scroll_step; i++)
        {
            gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(textview.view), &iter, rect.x, rect.y + m*(distance / textview.scroll_step)*(i+1));
            gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(textview.view), &iter, 0.0, TRUE, 0.0, 0.0);
            usleep(textview.scroll_time / textview.scroll_step);
        }
    }
    else
    {
        gtk_text_view_get_iter_at_location(GTK_TEXT_VIEW(textview.view), &iter, rect.x, rect.y + m*distance);
        gtk_text_view_scroll_to_iter(GTK_TEXT_VIEW(textview.view), &iter, 0.0, TRUE, 0.0, 0.0);
    }
}

void textview_scroll_down()
{
    textview_scroll(TRUE);
}

void textview_scroll_up()
{
    textview_scroll(FALSE);
}

void textview_clear_textbuf(GtkTextBuffer *buf)
{
    GtkTextIter start, end;
    if(!buf) buf = textview.buffer;
    gtk_text_buffer_get_bounds(buf, &start, &end);
    gtk_text_buffer_delete(buf, &start, &end);
}

void textview_insert_message(GtkTextBuffer *buf, gchar *msg)
{
    GtkTextIter iter;
    if(!buf) buf = textview.buffer;
    textview_clear_textbuf(buf);
    gtk_text_buffer_get_end_iter(buf, &iter);
    gtk_text_buffer_insert(buf, &iter, msg, -1);
}

