package fuku.eb4j.io;

import java.io.*;

import fuku.eb4j.EBException;

/**
 * 書籍用入力ストリームの基底クラス。
 *
 * @author Hisaya FUKUMOTO
 * @version 0.3.5
 */
public abstract class BookInputStream {

    /** ページサイズ */
    public static final int PAGE_SIZE = 2048;

    /** ファイル情報 */
    protected FileInfo _info = null;
    /** 入力ストリーム */
    protected RandomAccessFile _stream = null;
    /** ファイルポインタ位置 */
    protected long _filePos = 0;

    /** キャッシュ */
    protected byte[] _cache = null;
    /** キャッシュデータのファイルポインタ位置 */
    protected long _cachePos = -1;


    /**
     * コンストラクタ。
     *
     * @param info ファイル情報
     */
    BookInputStream(FileInfo info) {
        super();
        _info = info;
    }


    /**
     * このオブジェクトで使用されているシステムリソースを破棄します。
     *
     * @exception Throwable このメソッドで生じた例外
     */
    protected void finalize() throws Throwable {
        super.finalize();
        close();
    }

    /**
     * このファイルのファイルサイズを返します。
     *
     * @return ファイルサイズ
     */
    public final long getFileSize() {
        return _info.fileSize;
    }

    /**
     * このファイルの実ファイルサイズを返します。
     *
     * @return 実ファイルサイズ
     */
    public final long getRealFileSize() {
        return _info.realFileSize;
    }

    /**
     * このファイルのスライスサイズを返します。
     *
     * @return スライスサイズ
     */
    public final int getSliceSize() {
        return _info.sliceSize;
    }

    /**
     * ファイル情報を初期化します。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    protected void initFileInfo() throws EBException {
    }

    /**
     * このファイルを開きます。
     *
     * @exception EBException 入出力エラーが発生した場合
     */
    protected final void open() throws EBException {
        if (_stream != null) {
            close();
        }

        try {
            _stream = new RandomAccessFile(_info.file, "r");
        } catch (FileNotFoundException e) {
            EBException exp =
                new EBException(EBException.FILE_NOT_FOUND, _info.file.getPath());
            exp.setStackTrace(e.getStackTrace());
            throw exp;
        }
        _filePos = 0;
    }

    /**
     * このファイルを閉じます。
     *
     */
    public final void close() {
        if (_stream != null) {
            try {
                _stream.close();
            } catch (IOException e) {
            }
        }
    }

    /**
     * このファイルからb.lengthバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @exception EBException 入出力エラーが発生した場合
     */
    public final void readFully(byte[] b) throws EBException {
        readFully(b, 0, b.length);
    }

    /**
     * このファイルからlenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @exception EBException 入出力エラーが発生した場合
     */
    public final void readFully(byte[] b, int off, int len) throws EBException {
        int rlen = len;
        int offset = off;
        while (rlen > 0) {
            int n = read(b, offset, rlen);
            if (n == -1) {
                throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath());
            }
            rlen -= n;
            offset += n;
        }
    }

    /**
     * このファイルから最大b.lengthバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    public final int read(byte[] b) throws EBException {
        return read(b, 0, b.length);
    }

    /**
     * このファイルから最大lenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    public abstract int read(byte[] b, int off, int len) throws EBException;

    /**
     * 指定位置にファイルポインタを設定します。
     *
     * @param page ページ番号
     * @param offset ページ内オフセット
     */
    public final void seek(long page, int offset) {
        seek(getPosition(page, offset));
    }

    /**
     * 指定位置にファイルポインタを設定します。
     *
     * @param pos データ位置
     */
    public final void seek(long pos) {
        if (pos < 0) {
            _filePos = 0;
        } else if (pos > _info.fileSize) {
            _filePos = _info.fileSize;
        } else {
            _filePos = pos;
        }
    }

    /**
     * このファイルから最大b.lengthバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    protected final int readRaw(byte[] b) throws EBException {
        return readRaw(b, 0, b.length);
    }

    /**
     * このファイルから最大lenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @return バッファに読み込まれたバイトの合計数
     *         (ストリームの終わりに達してデータがない場合は-1)
     * @exception EBException 入出力エラーが発生した場合
     */
    protected final int readRaw(byte[] b, int off, int len) throws EBException {
        int ret = -1;
        try {
            ret = _stream.read(b, off, len);
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        }
        return ret;
    }

    /**
     * このファイルからb.lengthバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @exception EBException 入出力エラーが発生した場合
     */
    protected final void readRawFully(byte[] b) throws EBException {
        readRawFully(b, 0, b.length);
    }

    /**
     * このファイルからlenバイトのデータをバイト配列に読み込みます。
     *
     * @param b データの読み込み先のバッファ
     * @param off データの開始オフセット
     * @param len 読み込まれる最大バイト数
     * @exception EBException 入出力エラーが発生した場合
     */
    protected final void readRawFully(byte[] b, int off, int len) throws EBException {
        try {
            _stream.readFully(b, off, len);
        } catch (EOFException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        } catch (IOException e) {
            throw new EBException(EBException.FAILED_READ_FILE, _info.file.getPath(), e);
        }
    }

    /**
     * ファイルの先頭からの位置を返します。
     *
     * @param page ページ番号
     * @param offset ページ内オフセット
     * @return 先頭からの位置
     */
    public static final long getPosition(long page, int offset) {
        return (page - 1) * PAGE_SIZE + offset;
    }

    /**
     * ページ番号を返します。
     *
     * @param pos 先頭からの位置
     * @return ページ番号
     */
    public static final long getPage(long pos) {
        return pos / PAGE_SIZE + 1;
    }

    /**
     * ページ内オフセットを返します。
     *
     * @param pos 先頭からの位置
     * @return ページ内オフセット
     */
    public static final int getOffset(long pos) {
        return (int)(pos % PAGE_SIZE);
    }
}

// end of BookInputStream.java
